"""MaterialComponent class. Data for material Coverage."""
# 1. Standard python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest

# 4. Local modules
from xms.tuflowfv.components.material_component_display import MaterialComponentDisplay
from xms.tuflowfv.components.tuflowfv_component import change_display_options, TuflowFvComponent
from xms.tuflowfv.data.material_data import MaterialData


__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"


MAT_INITIAL_ATT_ID_FILE = 'initial_mat_polys.ids'
MAT_INITIAL_COMP_ID_FILE = 'initial_mat_comp.ids'


class MaterialComponent(TuflowFvComponent):
    """A hidden Dynamic Model Interface (DMI) component for the TUFLOWFV materials coverage."""

    def __init__(self, main_file):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.class_name = 'MaterialComponent'
        self.module_name = 'xms.tuflowfv.components.material_component'
        self.is_sediment = False
        self.data = MaterialData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.tree_commands = [  # [(menu_text, menu_method)...]
            ('Material List and Properties', 'open_material_properties'),
        ]
        self.polygon_commands = [('Assign Material', 'open_assign_poly_materials')]
        self.disp_opts_files = ['']
        # Copy default display options if needed
        display_helper = MaterialComponentDisplay(self)
        display_helper.ensure_display_options_exist()

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        action = ActionRequest(modality='NO_DIALOG', class_name=self.class_name, module_name=self.module_name,
                               main_file=self.main_file, method_name='get_initial_display_options')
        return [], [action]

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                    start with.
                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                    data must use relative file paths.
                SAVE happens when re-saving this project.
                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.
                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                    the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            tuple(str, list, list):
                - new_main_file (str): Path to the new component's mainfile
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            new_json_file = os.path.join(new_path, os.path.basename(self.disp_opts_files[0]))
            new_comp_uuid = os.path.basename(new_path)
            self.data.info.attrs['display_uuid'] = change_display_options(new_json_file, new_comp_uuid)

        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query, params):
        """Get the coverage UUID from XMS and send back the display options list.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        display_helper = MaterialComponentDisplay(self)
        return display_helper.initialize_display(query)

    def open_assign_poly_materials(self, query, params, win_cont):
        """Opens the Assign Materials dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        display_helper = MaterialComponentDisplay(self)
        return display_helper.open_assign_poly_materials(query, params[0], win_cont)

    def open_material_properties(self, query, params, win_cont):
        """Opens the Material Properties dialog and saves component data state on OK.

        Args:
            query (Query): Object for communicating with XMS
            params (list(dict)): Generic map of parameters. Contains selection map and component id files.
            win_cont (QWidget): The window container.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        display_helper = MaterialComponentDisplay(self)
        return display_helper.open_material_properties(query, win_cont)

    def handle_merge(self, merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            merge_list (list(tuple)): tuple containing:
                - main_file (str): The absolute path to the main file of the old component this
                  component is being merged from.
                - id_files (dict): The dictionary keys are 'POINT', 'ARC', and 'POLYGON'.
                  Each value is a tuple that may have two absolute file paths or none. The first
                  file is for the ids in XMS on the coverage. The second file contains the ids the
                  old component used for those objects. Both id files should be equal in length.
                  This dictionary is only applicable if the component derives from
                  CoverageComponentBase.

        Returns:
            tuple(list, list):
                - messages (list(tuple(str, str))): List of tuples with the first element of the tuple being the message
                    level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.
                - action_requests (list(ActionRequest)): List of actions for XMS to perform.
        """
        main_file_idx = 0
        # att_file = 0
        # comp_file = 1
        # id_file_idx = 1
        # all_att_ids = []
        # new_comp_ids = []
        for m in merge_list:
            mat = MaterialData(m[main_file_idx])
            self.data.add_materials(mat)
            # old_to_new_mat_ids = self.data.add_materials(mat)
            # if len(m[id_file_idx]['POLYGON']) < 1:
            #     continue
            # att_ids = read_display_option_ids(m[id_file_idx]['POLYGON'][att_file])
            # comp_ids = read_display_option_ids(m[id_file_idx]['POLYGON'][comp_file])
            # all_att_ids.extend(att_ids)
            # for _, comp_id in zip(att_ids, comp_ids):
            #     new_comp_ids.append(int(old_to_new_mat_ids[comp_id]))
        # save attids and compids to a temp file so they can be processed later
        # tmpfname = os.path.join(os.path.dirname(self.main_file), 'tmp_att_ids_comp_ids.txt')
        # with open(tmpfname, 'w') as file:
        #     for att_id, comp_id in zip(all_att_ids, new_comp_ids):
        #         file.write(f'{att_id} {comp_id}\n')
        display_helper = MaterialComponentDisplay(self)
        display_helper.update_display_options_file()
        all_mat_ids = list(self.data.materials.to_dataframe()['id'].astype(int))
        display_helper.update_display_id_files([], all_mat_ids)
        self.data.commit()

        action = ActionRequest(modality='NO_DIALOG', class_name=self.class_name, module_name=self.module_name,
                               main_file=self.main_file, comp_uuid=self.uuid,
                               method_name='get_initial_display_options')
        return [], [action]
