"""TUFLOW-FV control file writer."""
# 1. Standard python modules
import datetime
from io import StringIO
import logging
import os
import shutil

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import QDateTime

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs
from xms.data_objects.parameters import FilterLocation
from xms.guipy.time_format import ISO_DATETIME_FORMAT, qdatetime_to_datetime

# 4. Local modules
from xms.tuflowfv.components.output_points_component import OutputPointsComponent
from xms.tuflowfv.components.tuflowfv_component import UNINITIALIZED_COMP_ID
from xms.tuflowfv.data import sim_data as smd
from xms.tuflowfv.data.bc_data import ARC_BC_TYPES, CURTAIN_BC_TYPES, GRIDDED_BC_TYPES, MSLP_BC_TYPES
from xms.tuflowfv.data.coverage_collector import GIS_POINT, GIS_POLYGON
from xms.tuflowfv.file_io import io_util
from xms.tuflowfv.file_io.holland_wind_writer import HollandWindWriter
from xms.tuflowfv.file_io.shapefile_copier import ShapefileCopier
from xms.tuflowfv.file_io.shapefile_writer import ShapefileWriter
from xms.tuflowfv.file_io.structure_writer import StructureWriter
from xms.tuflowfv.gui import assign_bc_consts as bc_const
from xms.tuflowfv.gui import model_control_consts as const
from xms.tuflowfv.gui.gui_util import DEFAULT_VALUE_VARIABLE, get_tuflow_zero_time


class ControlWriter:
    """Writer class for the TUFLOW-FV control file."""
    BUFFER_SIZE = 8 * 1024
    COMMENT_LINE = '!_________________________________________________________________\n'
    GRIDDED_BC_VARIABLES = {
        'MSLP_GRID': (2, 2),  # (num_required, num_total)
        'W10_GRID': (3, 3),
        'WAVE': (4, 9),
    }

    def __init__(self, xms_data, coverages):
        """Constructor.

        Args:
            xms_data (XmsData): Simulation data retrieved from SMS
            coverages (CoverageCollector): The simulation coverage data
        """
        self._filename = ''
        self._ss = StringIO()
        self._logger = logging.getLogger('xms.tuflowfv')
        self._xms_data = xms_data
        self._coverages = coverages
        self._used_cov_names = set()
        self._used_output_cov_names = set()
        self._output_suffixes = {
            'XMDF': [],
            'NetCDF': [],
            'Flux': [],
            'Mass': [],
            'Points': [],
            'Transport': [],
        }
        self._child_sims = []  # [(sim_uuid, sim_name)]

    class SplitStruct:
        """Struct to store parameters for split output blocks."""
        def __init__(self, do_comp, do_cov, use_isodate, output_block):
            """Constructor.

            Args:
                do_comp (Component): The data_objects component for the coverage
                do_cov (Coverage): The data_objects coverage
                use_isodate (bool): True if using ISODATE format
                output_block (Dataset): xarray Dataset of the output block
            """
            self.do_comp = do_comp
            self.do_cov = do_cov
            self.use_isodate = use_isodate
            self.output_block = output_block

    def _format_isodate(self, dt_str):
        """Format the datetime item as an isodate.

        Args:
            dt_str (str): date time as a default QT string

        Returns:
            str: date time formatted as an isodate
        """
        dt = None
        # Try to parse using current locale
        try:
            qreftime = QDateTime.fromString(dt_str)
            dt = qdatetime_to_datetime(qreftime)
        except ValueError:
            pass

        # Try to parse using ISO format
        if dt is None:
            try:
                dt = datetime.datetime.strptime(dt_str, ISO_DATETIME_FORMAT)
            except ValueError:
                pass

        # Try to parse using TUFLOWFV format
        if dt is None:
            try:
                dt = datetime.datetime.strptime(dt_str, '%d/%m/%Y %H:%M:%S')
            except ValueError:
                pass

        if dt is None:
            self._logger.error('Unable to format date/time.')
            return ''
        return dt.strftime('%d/%m/%Y %H:%M:%S')

    def _create_run_folders(self):
        """Create the standard folder structure for a run TUFLOWFV run.

        bc_dbase
           flow
           hycom
           meteorology
           tide
           wave
        check
        model
           csv
           geo
           gis
              empty
        results
        runs
           log
        """
        cwd = os.getcwd()
        os.makedirs(os.path.join(cwd, 'bc_dbase', 'flow'), exist_ok=True)  # 'bc_dbase' root folder and children
        os.makedirs(os.path.join(cwd, 'bc_dbase', 'hycom'), exist_ok=True)
        os.makedirs(os.path.join(cwd, 'bc_dbase', 'meteorology'), exist_ok=True)
        os.makedirs(os.path.join(cwd, 'bc_dbase', 'tide'), exist_ok=True)
        os.makedirs(os.path.join(cwd, 'bc_dbase', 'wave'), exist_ok=True)
        os.makedirs(os.path.join(cwd, 'check'), exist_ok=True)  # 'check' root folder
        os.makedirs(os.path.join(cwd, 'model', 'csv'), exist_ok=True)  # 'model' root folder and children
        os.makedirs(os.path.join(cwd, 'model', 'geo'), exist_ok=True)
        os.makedirs(os.path.join(cwd, 'model', 'gis', 'empty'), exist_ok=True)
        os.makedirs(os.path.join(cwd, 'results'), exist_ok=True)  # 'results' root folder
        os.makedirs(os.path.join(cwd, 'runs', 'log'), exist_ok=True)  # 'runs' root folder and children
        self._filename = os.path.normpath(os.path.join(cwd, 'runs', f'{self._xms_data.sim_name}.fvc'))

    def _get_unique_coverage_name(self, do_cov, used_names, extension):
        """Generate a unique name for a coverage so it's files do not collide.

        Args:
            do_cov (Coverage): The data_objects Coverage to find a unique name for
            used_names (set): The already used coverage names for this coverage type (Output Points, BC shapefiles, etc)
            extension (str): The file extension to use in warning message if collision found

        Returns:
            str: See description
        """
        suffix = 1
        orig_covname = do_cov.name.replace(' ', '_')
        cov_name = orig_covname
        while cov_name in used_names:
            cov_name = f'{orig_covname}_{suffix}'.replace(' ', '_')
            suffix += 1
        used_names.add(cov_name)
        if cov_name != orig_covname:
            self._logger.warning(f'Duplicate coverage names found: {do_cov.name}. Coverage will be '
                                 f'exported to: {cov_name}.{extension}')
        return cov_name

    def _write_projection_file(self):
        """Write the display projection to a .prj file."""
        with open(os.path.join(os.getcwd(), 'model', 'gis', 'projection.prj'), 'w') as f:
            f.write(self._xms_data.query.display_projection.well_known_text)

    def _write_projection_commands(self):
        """Write the projection commands at the beginning of the file."""
        self._write_projection_file()
        self._ss.write('GIS FORMAT == SHP\n')  # For now, not supporting mif/mid files
        self._ss.write('SHP Projection == ../model/gis/projection.prj\n')
        if self._xms_data.sim_data.general.attrs['tutorial'] == 'ON':
            self._ss.write('Tutorial Model == ON\n')
        if self._xms_data.sim_data.general.attrs['projection_warning']:
            self._ss.write('GIS Projection Check == WARNING\n')  # By default, throws error

    def _write_simulation_commands(self):
        """Write the simulation commands card section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! SIMULATION CONFIGURATION\n')
        attrs = self._xms_data.sim_data.general.attrs
        if attrs['define_hardware_solver']:
            self._ss.write(f'Hardware == {attrs["hardware_solver"]}\n')
            if attrs['hardware_solver'] == 'GPU':
                self._ss.write(f'Device ID == {attrs["device_id"]}\n')
        if attrs['define_display_interval']:
            self._ss.write(f'Display dt == {int(attrs["display_interval"])}\n')
        if 'DEGREE' in self._xms_data.query.display_projection.horizontal_units.upper():
            self._ss.write('Spherical == 1\n')  # Defaults to Cartesian (1=degrees)
        if attrs['define_spatial_order']:
            self._ss.write(f'Spatial Order == {attrs["horizontal_order"]}, {attrs["vertical_order"]}\n')
        if 'METER' not in self._xms_data.query.display_projection.vertical_units.upper():
            self._ss.write('Units == Imperial\n')
        # Write the wind model type here, parameters later
        wind_attrs = self._xms_data.sim_data.wind_stress.attrs
        if wind_attrs['define_wind']:
            self._ss.write(f'Wind Stress Model == {wind_attrs["method"]}\n')
        # Write bottom drag model type here, parameters later
        mat_attrs = self._xms_data.sim_data.global_set_mat.attrs
        if mat_attrs['define_global_roughness']:
            self._ss.write(f'Bottom Drag Model == {mat_attrs["global_roughness"]}\n')

    def _write_time_commands(self):
        """Write the time commands card section.

        Returns:
            datetime.datetime: The simulation reference timestamp
        """
        self._ss.write(f'\n{self.COMMENT_LINE}! TIME AND TIMESTEP COMMANDS\n! Time Commands\n')
        attrs = self._xms_data.sim_data.time.attrs
        self._ss.write(f'Time Format == {"ISODATE" if attrs["use_isodate"] else "HOURS"}\n')
        if attrs["use_isodate"]:
            self._ss.write(f'Reference Time == {self._format_isodate(attrs["ref_date"])}\n')
            self._ss.write(f'Start Time == {self._format_isodate(attrs["start_date"])}\n')
            self._ss.write(f'End Time == {self._format_isodate(attrs["end_date"])}\n')
            sim_reftime = datetime.datetime.strptime(attrs["ref_date"], ISO_DATETIME_FORMAT)
        else:
            self._ss.write(f'Reference Time == {attrs["ref_date_hours"]}\n')
            self._ss.write(f'Start Time == {attrs["start_hours"]}\n')
            self._ss.write(f'End Time == {attrs["end_hours"]}\n')
            # Need to compute a datetime timestamp if not using ISODATE time format in case this simulation has linked
            # wind boundary conditions, which are always specified with datetimes in SMS.
            sim_reftime = get_tuflow_zero_time() + datetime.timedelta(hours=attrs["ref_date_hours"])
        self._ss.write('\n! Timestepping Commands\n')
        self._ss.write(f'CFL == {attrs["cfl"]}\n')
        self._ss.write(f'Timestep Limits == {attrs["min_increment"]}, {attrs["max_increment"]}\n')
        return sim_reftime

    def _write_model_parameters(self):
        """Write the model parameters card section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! MODEL PARAMETERS\n! Turbulence\n')
        attrs = self._xms_data.sim_data.globals.attrs
        geom_attrs = self._xms_data.sim_data.geometry.attrs
        self._write_horizontal_viscosity()
        self._write_horizontal_diffusivity()
        if attrs['define_vertical_mixing']:
            self._write_vertical_viscosity()
            if attrs['define_vertical_scalar_diffusivity']:
                self._write_vertical_diffusivity()
        self._ss.write('\n! Cell Wet/Dry Depths and Stability\n')
        if geom_attrs['define_cell_depths']:
            self._ss.write(f'Cell Wet/Dry Depths == {geom_attrs["dry_cell_depth"]}, {geom_attrs["wet_cell_depth"]}\n')
        if attrs['define_stability_limits']:
            self._ss.write(f'Stability Limits == {attrs["stability_wse"]}, {attrs["stability_velocity"]}\n')
        self._write_wind_parameters()

    def _write_horizontal_viscosity(self):
        """Write the horizontal viscosity mixing parameters."""
        attrs = self._xms_data.sim_data.globals.attrs
        if attrs['horizontal_mixing'] != 'None':
            self._ss.write(f'Momentum Mixing Model == {attrs["horizontal_mixing"]}\n')
            self._ss.write(f'Global Horizontal Eddy Viscosity == {attrs["global_horizontal_viscosity"]}\n')
        # Viscosity limits
        if attrs['define_horizontal_viscosity_limits']:
            self._ss.write(
                f'Global Horizontal Eddy Viscosity Limits == {attrs["horizontal_viscosity_min"]}, '
                f'{attrs["horizontal_viscosity_max"]}\n'
            )

    def _write_vertical_viscosity(self):
        """Write the vertical viscosity mixing model parameters."""
        attrs = self._xms_data.sim_data.globals.attrs
        mixing_model = attrs["vertical_mixing"]
        self._ss.write(f'Vertical Mixing Model == {mixing_model}\n')
        # Turbulence dt
        if attrs['define_turbulence_update']:
            self._ss.write(f'Turbulence Update dt == {attrs["turbulence_update"]}\n')
        if mixing_model != 'External':
            if mixing_model == 'Constant':
                value = f'{attrs["global_vertical_viscosity"]}'
            else:
                value = f'{attrs["global_vertical_parametric_coefficients1"]}, ' \
                        f'{attrs["global_vertical_parametric_coefficients2"]}'
            self._ss.write(f'Vertical Mixing Parameters == {value}\n')
        elif attrs['external_vertical_viscosity']:  # External turbulence with directory defined
            # Reconstruct absolute path if relative from the project.
            abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'],
                                                 attrs['external_vertical_viscosity'])
            if os.path.isdir(abs_path):
                # Make path relative to the control file
                rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
                self._ss.write(f'External Turbulence Model Directory == {rel_path}\n')
            else:
                self._logger.error(
                    f'Unable to find external turbulence directory.\nResolved to: {io_util.logging_filename(abs_path)}'
                )
        # Viscosity limits
        if attrs['define_vertical_viscosity_limits']:
            self._ss.write(
                f'Global Vertical Eddy Viscosity Limits == {attrs["vertical_viscosity_min"]}, '
                f'{attrs["vertical_viscosity_max"]}\n'
            )

    def _write_horizontal_diffusivity(self):
        """Write the horizontal diffusivity mixing parameters."""
        attrs = self._xms_data.sim_data.globals.attrs
        horiz_diff_type = attrs['horizontal_scalar_diffusivity_type']
        if horiz_diff_type != 'None':
            self._ss.write(f'Scalar Mixing Model == {horiz_diff_type}\n')
            if horiz_diff_type != 'Elder':  # 2 values for Elder, 1 for Constant and Smagorinsky
                value = f'{attrs["horizontal_scalar_diffusivity"]}'
            else:
                value = f'{attrs["horizontal_scalar_diffusivity_coef1"]}, ' \
                        f'{attrs["horizontal_scalar_diffusivity_coef2"]}'
            self._ss.write(f'Global Horizontal Scalar Diffusivity == {value}\n')
            # Diffusivity limits
            if attrs['define_horizontal_diffusivity_limits'] and horiz_diff_type != 'Constant':
                self._ss.write(f'Global Horizontal Scalar Diffusivity Limits == '
                               f'{attrs["horizontal_scalar_diffusivity_min"]}, '
                               f'{attrs["horizontal_scalar_diffusivity_max"]}\n')

    def _write_vertical_diffusivity(self):
        """Write the vertical diffusivity mixing parameters."""
        attrs = self._xms_data.sim_data.globals.attrs
        vert_mixing_model = attrs['vertical_mixing']
        if vert_mixing_model != 'Parametric':  # 1 value for Constant and External, 2 for Parametric
            value = f'{attrs["vertical_scalar_diffusivity"]}'
        else:
            value = f'{attrs["vertical_scalar_diffusivity_coef1"]}, ' \
                    f'{attrs["vertical_scalar_diffusivity_coef2"]}'
        self._ss.write(f'Global Vertical Scalar Diffusivity == {value}\n')
        # Diffusivity limits
        if attrs['define_vertical_diffusivity_limits'] and vert_mixing_model != 'Constant':
            self._ss.write(f'Global Vertical Scalar Diffusivity Limits == '
                           f'{attrs["vertical_scalar_diffusivity_min"]}, '
                           f'{attrs["vertical_scalar_diffusivity_max"]}\n')

    def _write_wind_parameters(self):
        """Write the global wind parameters."""
        wind_attrs = self._xms_data.sim_data.wind_stress.attrs
        if wind_attrs['define_parameters']:
            self._ss.write('\n! Wind Parameters\n')
            method = wind_attrs['method']
            if method == const.WIND_STRESS_OPT_WU:
                self._ss.write(
                    f'Wind Stress Parameters == {wind_attrs["wa"]}, {wind_attrs["ca"]}, {wind_attrs["wb"]}, '
                    f'{wind_attrs["cb"]}\n'
                )
            elif method == const.WIND_STRESS_OPT_CONSTANT:
                self._ss.write(f'Wind Stress Parameters == {wind_attrs["bulk_coefficient"]}\n')
            elif method == const.WIND_STRESS_OPT_KONDO:
                self._ss.write(f'Wind Stress Parameters == {wind_attrs["scale_factor"]}\n')

    def _write_geometry_parameters(self):
        """Write the geometry parameters card section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! GEOMETRY\n')
        _, rel_path = io_util.build_input_filepaths(self._filename, 'model/geo', f'{self._xms_data.do_ugrid.name}.2dm')
        self._ss.write(f'Geometry 2d == {rel_path}\n')
        self._write_z_modifications()
        self._ss.write('\n! BC NODESTRINGS\n')
        self._write_gis_lines("BC")
        self._ss.write('\n! STRUCT NODESTRINGS\n')
        self._write_structure_lines()

    def _write_z_modifications(self):
        """Write all the Z modification commands in the order they were specified."""
        mod_dset = self._xms_data.sim_data.z_modifications
        mod_types = mod_dset.type.data.tolist()
        set_zpts = mod_dset.set_zpts.data.tolist()
        grid_zpts = mod_dset.grid_zpts.data.tolist()
        csv_files = mod_dset.csv.data.tolist()
        csv_types = mod_dset.csv_type.data.tolist()
        zline_idx = 0
        for idx, mod_type in enumerate(mod_types):
            if mod_type == smd.ELEV_TYPE_SET_ZPTS:
                self._ss.write(f'Set Zpts == {set_zpts[idx]}\n')
            elif mod_type == smd.ELEV_TYPE_GRID_ZPTS:
                self._write_grid_zpts(grid_zpts[idx])
            elif mod_type == smd.ELEV_TYPE_CELL_CSV:
                self._write_cell_elevation_file(csv_files[idx], csv_types[idx])
            else:  # smd.ELEV_TYPE_ZLINE
                self._write_gis_zline(zline_idx)
                zline_idx += 1

    def _write_grid_zpts(self, filename):
        """Write a read GRID Zpts command line.

        Args:
            filename (str): Path to the GRID Zpts file. May be absolute or relative to the SMS project
        """
        # Reconstruct absolute path if relative from the project.
        abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'], filename)
        if not os.path.isfile(abs_path):
            self._logger.error(f'Unable to find GRID Zpts file.\nResolved to: {io_util.logging_filename(abs_path)}')
            return
        # Make path relative to the control file
        rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
        self._ss.write(f'Read GRID Zpts == {rel_path}\n')

    def _write_cell_elevation_file(self, csv_file, csv_type):
        """Write a cell elevation file command line.

        Args:
            csv_file (str): Path to the cell elevation file
            csv_type (str): The CSV file format, one of: 'cell_ID', 'coordinate'
        """
        # Reconstruct absolute path if relative from the project.
        abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'], csv_file)
        if not os.path.isfile(abs_path):
            self._logger.error(
                f'Unable to find cell elevation file.\nResolved to: {io_util.logging_filename(abs_path)}'
            )
            return
        rel_path = os.path.basename(abs_path)
        if os.path.normpath(os.path.dirname(abs_path)) != os.path.normpath(os.path.dirname(self._filename)):
            # Copy the file to our run area if it is not in the same directory as the .fvc
            dest_path, rel_path = io_util.build_input_filepaths(self._filename, 'model/geo', rel_path)
            xfs.copyfile(abs_path, dest_path)
        self._ss.write(f'Cell Elevation File == {rel_path}, {csv_type}\n')

    def _write_gis_zline(self, zline_idx):
        """Write a read GIS Z line command."""
        zline, point_layers = self._xms_data.zlines[zline_idx]
        shp_copier = ShapefileCopier(self._filename)
        writer = ShapefileWriter(self._xms_data, self._coverages)
        rel_paths = []
        if isinstance(zline, str):
            shp_copier.copy(zline, rel_paths)
        else:  # This is a data_objects Coverage, convert it to a shapefile.
            abs_path, rel_path = io_util.build_input_filepaths(fvc_filename=self._filename, sub_dir='model/gis',
                                                               basename=zline.name)
            writer.write_zline_shapes(filename=abs_path, do_cov=zline)
            rel_paths.append(rel_path)

        for point_layer in point_layers:  # Up to 9 point layers per line layer
            if isinstance(point_layer, str):
                shp_copier.copy(point_layer, rel_paths)
            else:  # This is a data_objects Coverage, convert it to a shapefile.
                abs_path, rel_path = io_util.build_input_filepaths(fvc_filename=self._filename, sub_dir='model/gis',
                                                                   basename=point_layer.name)
                writer.write_zpoint_shapes(filename=abs_path, do_cov=point_layer)
                rel_paths.append(rel_path)
        self._ss.write(f'Read GIS Z Line == {" | ".join(rel_paths)}\n')

    def _write_gis_lines(self, command_type):
        """Write the read GIS lines and convert coverages to shapefiles."""

        # Grab the data appropriate for the type of shapes we are writing
        original_command_string = 'Nodestring'
        if command_type == "BC":
            cov_indices = self._coverages.shapefile_bc_coverages()
            covs = self._xms_data.bc_covs
        elif command_type == "MAT":
            cov_indices = self._coverages.shapefile_material_coverages()
            covs = self._xms_data.mat_covs
            original_command_string = 'MAT'
        else:
            raise ValueError(f'Invalid command type: {command_type}')

        # Write the file
        writer = ShapefileWriter(self._xms_data, self._coverages)
        for cov_idx in cov_indices:
            bc_format = self._coverages.bc_types.get(cov_idx)
            # Check if this a new GIS type
            command_string = original_command_string
            if bc_format is not None \
                    and original_command_string != "MAT" \
                    and bc_format in [GIS_POINT, GIS_POLYGON]:
                command_string = 'SA'
            shapefile_cov = covs[cov_idx]
            cov_name = self._get_unique_coverage_name(shapefile_cov, self._used_cov_names, 'shp')
            filename = f'{cov_name}.shp'
            abs_path, rel_path = io_util.build_input_filepaths(self._filename, 'model/gis', filename)
            self._ss.write(f'Read GIS {command_string} == {rel_path}\n')
            if command_type == "BC":
                writer.write_bc_shapefile(abs_path, cov_idx)
            elif command_type == "MAT":
                writer.write_material_shapefile(abs_path, cov_idx)

    def _write_structure_lines(self):
        """Write the read GIS structure lines and convert coverages to shapefiles."""
        writer = ShapefileWriter(self._xms_data, self._coverages)
        structure_cov = self._xms_data.structure_cov
        if structure_cov is None:
            return  # No structure lines to write
        cov_name = self._get_unique_coverage_name(structure_cov, self._used_cov_names, 'shp')
        filename = f'{cov_name}.shp'
        abs_path, rel_path = io_util.build_input_filepaths(self._filename, 'model/gis', filename)
        self._ss.write(f'Read GIS Nodestring == {rel_path}\n')
        writer.write_struct_shapefile(abs_path)

    def _write_material_parameters(self):
        """Write the material parameters card section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! MATERIAL PROPERTIES\n')
        attrs = self._xms_data.sim_data.global_set_mat.attrs
        if self._coverages.default_mat_id > UNINITIALIZED_COMP_ID:
            self._ss.write(f'Set Mat == {self._coverages.default_mat_id}\n')
        if attrs['define_global_roughness']:
            self._ss.write(f'Global Bottom Roughness == {attrs["global_roughness_coefficient"]}\n')
        self._write_gis_lines("MAT")
        self._write_material_blocks()

    def _write_material_blocks(self):
        """Write all the material blocks to the file."""
        for mat_id, mat_atts in self._coverages.material_list.items():
            self._ss.write(f'\nmaterial == {mat_id}\n')
            if mat_atts.inactive.item():
                self._ss.write('  inactive == 1\n')
            if mat_atts.override_bottom_roughness.item():
                self._ss.write(f'  bottom roughness == {mat_atts.bottom_roughness.item()}\n')
            if mat_atts.override_horizontal_eddy_viscosity.item():
                self._ss.write(f'  horizontal eddy viscosity == {mat_atts.horizontal_eddy_viscosity.item()}\n')
            if mat_atts.override_horizontal_eddy_viscosity_limits.item():
                self._ss.write('  horizontal eddy viscosity limits == '
                               f'{mat_atts.horizontal_eddy_viscosity_minimum.item()}, '
                               f'{mat_atts.horizontal_eddy_viscosity_maximum.item()}\n')
            if mat_atts.override_vertical_eddy_viscosity_limits.item():
                self._ss.write('  vertical eddy viscosity limits == '
                               f'{mat_atts.vertical_eddy_viscosity_minimum.item()}, '
                               f'{mat_atts.vertical_eddy_viscosity_maximum.item()}\n')
            if mat_atts.override_horizontal_scalar_diffusivity.item():
                self._ss.write(f'  horizontal scalar diffusivity == {mat_atts.horizontal_scalar_diffusivity.item()}\n')
            if mat_atts.override_horizontal_scalar_diffusivity_limits.item():
                self._ss.write('  horizontal scalar diffusivity limits == '
                               f'{mat_atts.horizontal_scalar_diffusivity_minimum.item()}, '
                               f'{mat_atts.horizontal_scalar_diffusivity_maximum.item()}\n')
            if mat_atts.override_vertical_scalar_diffusivity_limits.item():
                self._ss.write('  vertical scalar diffusivity limits == '
                               f'{mat_atts.vertical_scalar_diffusivity_minimum.item()}, '
                               f'{mat_atts.vertical_scalar_diffusivity_maximum.item()}\n')
            if mat_atts.override_bed_elevation_limits.item():
                self._ss.write(f'  bed elevation limits == {mat_atts.bed_elevation_minimum.item()}, '
                               f'{mat_atts.bed_elevation_maximum.item()}\n')
            if mat_atts.spatial_reconstruction.item():  # Defaults to 0
                self._ss.write('  spatial reconstruction == 1\n')
            self._ss.write('end material\n')

    def _write_initial_conditions(self):
        """Write the initial conditions parameters card section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! INITIAL CONDITIONS\n')
        attrs = self._xms_data.sim_data.initial_conditions.attrs
        if attrs['define_initial_water_level']:
            self._ss.write(f'Initial Water Level == {attrs["initial_water_level"]}\n')
        if attrs['use_restart_file']:
            # Reconstruct absolute path if relative from the project.
            abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'], attrs['restart_file'])
            if not os.path.isfile(abs_path):
                self._logger.error(f'Unable to find restart file.\nResolved to: {io_util.logging_filename(abs_path)}')
                return
            # Make path relative to the control file
            rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
            self._ss.write(f'Restart File == {rel_path}\n')
            if not attrs['use_restart_file_time']:  # Defaults to 1
                self._ss.write('Use Restart File Time == 0\n')

    def _write_boundary_conditions(self, sim_reftime):
        """Write the BC parameters card section.

        Args:
            sim_reftime (datetime.datetime): The simulation reference timestamp
        """
        self._ss.write(f'\n{self.COMMENT_LINE}! BOUNDARY CONDITIONS\n')
        attrs = self._xms_data.sim_data.globals.attrs
        if attrs['define_bc_default_update_dt']:
            self._ss.write(f'BC Default Update dt == {attrs["bc_default_update_dt"]}\n')
        wave_note = '! Global wave command only applicable if simulation contains a wave boundary'
        if not attrs['include_wave_stress']:
            self._ss.write(f'include wave stress == 0  {wave_note}\n')
        if not attrs['include_stokes_drift']:
            self._ss.write(f'include stokes drift == 0  {wave_note}\n')
        self._write_bc_nodestring_blocks()
        self._write_bc_point_blocks()
        self._write_global_bcs()
        self._write_wind_boundaries(sim_reftime)
        if attrs['transport_mode']:
            abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'],
                                                 attrs['transport_file'])
            if not os.path.isfile(abs_path):
                self._logger.error(f'Unable to find transport file.\nResolved to: {io_util.logging_filename(abs_path)}')
                return
            # Make path relative to the control file
            rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
            self._ss.write(f'\nbc == transport, {rel_path}\nend bc\n')

    def _write_bc_nodestring_blocks(self):
        """Write all the BC nodestring blocks to the file."""
        for bc_id, atts in self._coverages.bcs.items():
            if not atts:
                continue  # If a monitor line (no atts), don't write a BC block
            bc_atts, bc_curve = atts
            self._write_bc_block_header(bc_id, bc_atts, bc_curve)
            self._write_bc_atts_to_block(bc_atts)

    def _write_bc_point_blocks(self):
        """Write all the BC point blocks to the file."""
        for bc_point, bc_atts, bc_curve in self._coverages.bc_points:
            self._write_bc_block_header(self._coverages.next_bc_id, bc_atts, bc_curve, bc_point)
            self._coverages.next_bc_id += 1
            self._write_bc_atts_to_block(bc_atts)

    def _write_global_bcs(self):
        """Write the global BCs to the file."""
        bc_data = self._xms_data.sim_data.global_bcs
        comp_ids = bc_data.bcs.comp_id.data.tolist()  # Global BCs stored on the arcs Dataset
        for comp_id in comp_ids:
            dset = bc_data.bcs.sel(comp_id=comp_id)
            bc_type = dset.type.item()
            curve = bc_data.get_bc_curve(comp_id=comp_id, bc_type=bc_type, default=False)
            if curve is None or curve.Time.size == 0:
                self._logger.error(f'No curve defined for global {bc_type} boundary, BC will not be exported.')
                continue
            if bc_type == 'QG':  # TUFLOWFV can't handle the underscores, NetCDF can't handle path separators.
                curve = curve.rename({'Q_over_A': 'QA'})
            self._write_bc_block_header(bc_id=self._coverages.next_bc_id, bc_atts=dset, bc_curve=curve)
            self._coverages.next_bc_id += 1
            self._write_bc_atts_to_block(dset)

    def _write_wind_boundaries(self, sim_reftime):
        """Write the Holland wind boundaries to the control file.

        Args:
            sim_reftime (datetime.datetime): The simulation reference timestamp
        """
        if self._xms_data.wind_covs and self._xms_data.query.display_projection.coordinate_system != 'GEOGRAPHIC':
            self._logger.error('Holland wind boundaries have been specified in the simulation but the current '
                               'display projection does not have a geographic coordinate system. This is not currently '
                               'supported in SMS.')
            return

        csv_writer = HollandWindWriter(sim_reftime, self._xms_data.sim_data.time.attrs["use_isodate"])
        for wind_boundary in self._xms_data.wind_covs:
            csv_basename = f'{self._xms_data.sim_name}_bc_{self._coverages.next_bc_id}.csv'
            self._coverages.next_bc_id += 1
            abs_path, rel_path = io_util.build_input_filepaths(self._filename, 'bc_dbase', csv_basename)
            self._ss.write(f'\nbc == CYC_HOLLAND, {rel_path}\nend bc\n')
            csv_writer.write(abs_path, wind_boundary)

    def _write_variable_n_bc_att(self, bc_atts, att_name, num_values):
        """Write BC attribute with a variable number of values.

        Args:
            bc_atts (xr.Dataset): The BC attributes to write
            att_name (str): The name of the attribute: 'offset', 'scale', or 'default'
            num_values (int): 1-based number of values to writ3e for the attribute
        """
        values = [str(bc_atts[f'{att_name}{i + 1}'].item()) for i in range(num_values)]
        self._ss.write(f'  bc {att_name} == {", ".join(values)}\n')

    def _write_bc_atts_to_block(self, bc_atts, num_specified=None):
        """Write the BC attributes.

        Args:
            bc_atts (xr.Dataset): The BC attributes to write
            num_specified (Optional[int]): The number of variables/columns specified. This is for gridded BCs. There
                are varying numbers of required and optional variables. Need to write out default, offset, and scale
                for each specified variable if those options are enabled.
        """
        bc_type = bc_atts.type.item()
        num_values = len(bc_const.BC_VARIABLE_LABEL_TEXT.get(bc_type, ())) if num_specified is None else num_specified
        # Subtract axis dimensions from number of values per att. The time column plus the chainage column for curtains.
        if bc_type in CURTAIN_BC_TYPES:
            num_values -= 2
        elif bc_type == 'QN':  # Special case for QN, no curve just friction value
            num_values = 1
        else:
            num_values -= 1
        mslp_applicable = bc_type in MSLP_BC_TYPES

        if bc_atts.define_offset.item() and num_values > 0:
            self._write_variable_n_bc_att(bc_atts, 'offset', num_values)
        if bc_atts.define_scale.item() and num_values > 0:
            self._write_variable_n_bc_att(bc_atts, 'scale', num_values)
        if bc_atts.define_default.item() and num_values > 0:
            self._write_variable_n_bc_att(bc_atts, 'default', num_values)
        if bc_atts.define_update_dt.item() and num_values > 0:
            self._ss.write(f'  bc update dt == {bc_atts.update_dt.item()}\n')
        self._ss.write(f'  bc time units == {bc_atts.time_units.item()}\n')
        if bc_atts.define_reference_time.item():
            bc_time_format = bc_atts.use_isodate.item()
            # This requirement doesn't seem to apply to gridded BC types.
            if bc_time_format != self._xms_data.sim_data.time.attrs['use_isodate'] and bc_type not in GRIDDED_BC_TYPES:
                self._logger.error('Inconsistent time formats found between the simulation and a boundary condition. '
                                   'Ensure time formats are consistent for all simulation inputs.')
            if bc_time_format:
                self._ss.write(f'  bc reference time == {self._format_isodate(bc_atts.reference_time_date.item())}\n')
            else:
                self._ss.write(f'  bc reference time == {bc_atts.reference_time_hours.item()}\n')
        if mslp_applicable and not bc_atts.include_mslp.item():  # Only write if disabled, on by default if applicable
            self._ss.write('  includes MSLP == 0\n')
        if bc_atts.define_vertical_distribution.item():
            self._ss.write(f'  vertical coordinate type == {bc_atts.vertical_distribution_type.item()}\n')
            # Reconstruct absolute path if relative from the project.
            abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'],
                                                 bc_atts.vertical_distribution_file.item())
            # Make path relative to the control file
            rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
            self._ss.write(f'  vertical distribution file == {rel_path}\n')
        self._ss.write('end bc\n')

    def _write_bc_block_header(self, bc_id, bc_atts, bc_curve, bc_point=None):
        """Write the BC block header for an arc.

        Args:
            bc_id (int): Id of the BC (in the .fvc file, not feature id)
            bc_atts (xr.Dataset): The BC attributes to write
            bc_curve (xr.Dataset): The BC curve
            bc_point (Point): The data_object point
        """
        bc_type = bc_atts.type.item()
        bc_name = bc_atts.name.item()
        if not bc_name:
            bc_name = bc_id
        if bc_type not in ['QN', 'ZG']:
            self._check_bc_time_format(bc_id, bc_atts)
            if bc_type in CURTAIN_BC_TYPES:
                rel_path, bc_header = self._get_curtain_netcdf_info(bc_id, bc_atts)
            else:
                rel_path, bc_header = self._write_bc_csv(bc_id, bc_curve)

            if bc_point:  # Point BC - x,y coordinates
                self._ss.write(f'\nbc == {bc_type}, {bc_point.x}, {bc_point.y}, {rel_path}\n')
            elif bc_type == 'QG':  # Global BC - no ID
                self._ss.write(f'\nbc == {bc_type}, {rel_path}\n')
            else:  # Arc BCs with curves
                self._ss.write(f'\nbc == {bc_type}, {bc_name}, {rel_path}\n')
            self._ss.write(f'  bc header == {bc_header}\n')
        elif bc_type == 'ZG':  # No BC curve for zero gradient
            self._ss.write(f'\nbc == ZG, {bc_name}\n')
        else:
            self._ss.write(f'\nbc == QN, {bc_name}, {bc_atts.friction_slope.item()}\n')
        if bc_type != 'ZG':
            self._ss.write(f'  sub-type == {int(bc_atts.subtype.item())}\n')

    def _check_bc_time_format(self, bc_id, bc_atts):
        """Write the BC curve CSV for an arc or point.

        Args:
            bc_id (int): Id of the BC (in the .fvc file, not feature id)
            bc_atts (xr.Dataset): The BC attributes to write
        """
        sim_isodate = self._xms_data.sim_data.time.attrs['use_isodate']
        formats_match = True
        if bc_atts.define_reference_time.item():  # Check for incompatible time formats
            if sim_isodate != bc_atts.use_isodate.item():
                formats_match = False  # Time formats for the reference date do not match.
        if formats_match:
            if bc_atts.time_units.item() == 'Isotime' and not sim_isodate:
                formats_match = False
        if not formats_match:
            self._logger.error(f'Inconsistent time formats found between the simulation and BC {bc_id} ensure that '
                               f'all time formats for input BCs match the simulation time format.')

    def _write_bc_csv(self, bc_id, bc_curve):
        """Write the BC curve CSV for an arc or point.

        Args:
            bc_id (int): Id of the BC (in the .fvc file, not feature id)
            bc_curve (xr.Dataset): The BC attributes curve

        Returns:
            tuple(str, str): The path to the CSV file, the BC column header
        """
        if bc_curve is None:
            self._logger.error(f'No curve defined for BC {bc_id}. Ensure boundary condition specification is complete.')
            return '', ''
        self._logger.info(f'Writing curve for BC {bc_id}...')
        csv_basename = f'{self._xms_data.sim_name}_bc_{bc_id}.csv'
        abs_path, rel_path = io_util.build_input_filepaths(self._filename, 'bc_dbase', csv_basename)
        df = bc_curve.to_dataframe()
        # Some of the column headers have '/' characters
        df.columns = [column.replace('/', '') for column in df.columns]
        df.to_csv(abs_path, index=False, date_format='%d/%m/%Y %H:%M:%S')  # TUFLOW ISODATE is not true ISODATE
        return rel_path, ', '.join(df.columns)

    def _get_curtain_netcdf_info(self, bc_id, bc_atts):
        """Get a reference to a curtain BCs NetCDF file and the NetCDF variables BC header.

        Args:
            bc_id (int): Id of the BC (in the .fvc file, not feature id)
            bc_atts (xr.Dataset): The BC attributes to write

        Returns:
            tuple(str, str): The path to the CSV file, the BC column header
        """
        # Reconstruct absolute path if relative from the project and make it relative to the .fvc, if possible.
        abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'],
                                             bc_atts.grid_dataset_file.item())
        if not os.path.isfile(abs_path):
            self._logger.error(f'Unable to find referenced NetCDF file for curtain boundary with ID {bc_id}. Resolved '
                               f'to {io_util.logging_filename(abs_path)}')
            return '', ''
        rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
        variables = [bc_atts[f'variable{i + 1}'].item() for i in range(len(ARC_BC_TYPES.get(bc_atts.type.item(), {})))]
        return rel_path, ', '.join(variables)

    def _write_hydraulic_structures(self):
        """Write the hydraulic structures parameters card section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! HYDRAULIC STRUCTURES\n')
        py_comp = self._xms_data.structure_comp
        cov = self._xms_data.structure_cov
        if py_comp is None or cov is None:
            return  # No structures
        # Write the structure blocks to the control file.
        writer = StructureWriter(py_comp, cov, self._filename, self._ss)
        writer.write()

        # TODO: Need to write one or two shapefiles (lines and/or polygons), and reference them in the control file.
        # Write the structure geometry to shapefiles.
        # writer = ShapefileWriter(self._xms_data.structure_cov)
        # writer.write_structure_shapefile()

    def _write_linked_simulations(self):
        """Write the linked child simulations section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! EXTERNAL FILES\n')
        child_sim_uuids = self._xms_data.sim_data.linked_simulations.uuid.data.tolist()
        for child_sim_uuid in child_sim_uuids:
            child_sim_item = self._xms_data.get_tree_item(child_sim_uuid)
            if child_sim_item:
                self._child_sims.append((child_sim_uuid, child_sim_item.name))
                self._ss.write(f'read file == ./{child_sim_item.name}.fvc\n')

    def _write_grid_definitions(self):
        """Write the grid definitions in the simulation."""
        self._ss.write(f'\n{self.COMMENT_LINE}! GRID DEFINITIONS\n')
        grid_defs = self._xms_data.sim_data.grid_definitions
        for i in range(len(grid_defs.grid_id.data)):
            grid_id = grid_defs.grid_id.data[i]
            grid_def = grid_defs.sel(dict(grid_id=grid_id))
            # Reconstruct absolute path to grid file if relative from the project.
            grid_file = grid_def.file.item()
            abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'], grid_file)
            if not os.path.isfile(abs_path):
                self._logger.error(
                    f'Unable to find referenced grid definition file: {io_util.logging_filename(abs_path)}'
                )
                continue
            # Make path relative to the control file, if possible (might be on different drive)
            rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
            self._ss.write(f'grid definition file == {rel_path}\n')
            self._write_grid_definition_block(grid_def)
            self._ss.write('end grid\n')

    def _write_grid_definition_block(self, grid_def):
        """Write a grid definition block (the innards).

        Args:
            grid_def (xr.Dataset): The Dataset for the grid definition
        """
        grid_name = grid_def.name.item()
        if grid_name:
            self._ss.write(f'  grid definition label == {grid_name}\n')
        x_var = grid_def.x_variable.item()
        y_var = grid_def.y_variable.item()
        variables = [x_var, y_var]
        z_var = grid_def.z_variable.item()  # We don't have an example of one that needs this yet, but it's there.
        if z_var and z_var != DEFAULT_VALUE_VARIABLE:
            variables.append(z_var)
        self._ss.write(f'  grid definition variables == {", ".join(variables)}\n')
        if grid_def.define_vert_coord_type.item():
            self._ss.write(f'  vertical coordinate type == {grid_def.vert_coord_type.item()}\n')
        if not grid_def.cell_gridmap.item():  # On by default, only write if explicitly disabled
            self._ss.write('  cell gridmap == 0\n')
        if grid_def.boundary_gridmap.item():  # Off by default, only write if explicitly enabled
            self._ss.write('  boundary gridmap == 1\n')
        if grid_def.suppress_coverage_warnings.item():  # Off by default, only write if explicitly enabled
            self._ss.write('  suppress coverage warnings == 1\n')

    def _write_gridded_bcs(self):
        """Write the gridded BCs to the file."""
        gridded_bcs = self._xms_data.sim_data.gridded_bcs
        num_bcs = len(gridded_bcs.bcs.comp_id)
        if num_bcs == 0:
            return  # No gridded BCs defined in the simulation
        self._ss.write(f'\n{self.COMMENT_LINE}! GRID BOUNDARY CONDITIONS\n')
        grid_defs = self._xms_data.sim_data.grid_definitions
        for i in range(num_bcs):
            bc_atts = gridded_bcs.bcs.sel(dict(comp_id=gridded_bcs.bcs.comp_id.data[i]))
            bc_type = bc_atts.type.item()
            grid_name = grid_defs.sel(dict(grid_id=bc_atts.grid_id.item())).name.item()
            # Reconstruct absolute path to grid file if relative from the project.
            grid_file = bc_atts.grid_dataset_file.item()
            abs_path = xfs.resolve_relative_path(self._xms_data.sim_data.info.attrs['proj_dir'], grid_file)
            if not os.path.isfile(abs_path):
                self._logger.error(
                    f'Unable to find referenced gridded boundary condition: {io_util.logging_filename(abs_path)}'
                )
                continue
            # Make path relative to the control file, if possible (might be on different drive)
            rel_path = io_util.compute_relative_path_safe(os.path.dirname(self._filename), abs_path)
            self._ss.write(f'bc == {bc_type}, {grid_name}, {rel_path}\n')
            num_specified = self._write_gridded_bc_variables(bc_atts)
            self._write_bc_atts_to_block(bc_atts, num_specified)
            self._ss.write('\n')

    def _write_gridded_bc_variables(self, bc_atts):
        """Write the NetCDF variables for a gridded BC.

        Args:
            bc_atts (xr.Dataset): The gridded BCs attributes

        Returns:
            int: The number of specified variables
        """
        bc_type = bc_atts.type.item()
        num_required, num_total = self.GRIDDED_BC_VARIABLES[bc_type]
        variables = []
        for i in range(num_total):
            variable_name = bc_atts[f'variable{i + 1}'].item()
            if i >= num_required and (not variable_name or variable_name == DEFAULT_VALUE_VARIABLE):
                # If we have specified the required amount of variables and we reach a non-specified variable name, we
                # are done.
                break
            variables.append(variable_name)
        self._ss.write(f'  bc header == {", ".join(variables)}\n')
        return len(variables)

    def _write_output_commands(self):
        """Write the output commands card section."""
        self._ss.write(f'\n{self.COMMENT_LINE}! OUTPUT COMMANDS\n')
        attrs = self._xms_data.sim_data.output.attrs
        if attrs['log_dir']:  # Only write this card if the user specified something
            self._ss.write(f'log dir == {attrs["log_dir"]}\n')
        if attrs['output_dir']:
            self._ss.write(f'output dir == {attrs["output_dir"]}\n')
        if attrs['write_check_files']:
            self._ss.write(f'write check files == {attrs["check_files_dir"]}\n')
        if attrs['write_empty_gis_files']:
            self._ss.write(f'write empty gis files == {attrs["empty_gis_files_dir"]}\n')
        if attrs['write_restart_file']:
            self._ss.write(f'write restart dt == {attrs["restart_file_interval"]}\n')
            # Default is to overwrite the restart file, so only write this card if user doesn't want to
            if not attrs['overwrite_restart_file']:
                self._ss.write('Restart overwrite == 0\n')
        self._write_output_blocks()
        self._warn_if_colliding_suffixes()

    def _write_output_blocks(self):
        """Write all the simulation output blocks to the file."""
        output_data = self._xms_data.sim_data.output
        index_name = next(iter(output_data.sizes.keys()))
        indices = output_data[index_name].data.tolist()
        use_isodate = self._xms_data.sim_data.time.attrs['use_isodate']
        ds = self._xms_data.sim_data.output_points_coverages
        split_blocks = []
        for index in indices:
            # Cannot have a mix of CSV and Shapefiles in the same output block
            have_gis = False
            have_csv = False
            output_block = output_data.where(output_data[index_name] == index, drop=True)
            file_format = output_block.format.item().lower()
            self._ss.write(f'\noutput == {file_format}\n')
            if file_format == 'points':  # Write the x,y locations to a CSV file if points block.
                row = ds.where(ds.row_index == output_block.row_index)
                cov_uuids = row['uuid'].data.tolist()
                for cov_uuid in cov_uuids:
                    do_cov = self._xms_data.get_output_points_cov(cov_uuid)
                    do_comp = self._xms_data.get_output_points_component(cov_uuid)
                    have_gis, have_csv, split = self._write_output_points_file(do_comp, do_cov, have_gis, have_csv,
                                                                               use_isodate, output_block)
                    if split:  # Save these guys for later
                        split_struct = self.SplitStruct(do_comp=do_comp, do_cov=do_cov, use_isodate=use_isodate,
                                                        output_block=output_block)
                        split_blocks.append(split_struct)
                    else:
                        self._write_output_block(output_block, use_isodate)
                        self._ss.write('end output\n')
            else:  # Not a point output. Just write the block commands
                self._write_output_block(output_block, use_isodate)
                self._ss.write('end output\n')
        self._write_split_output_blocks(split_blocks)

    def _write_split_output_blocks(self, split_blocks):
        """Write the split output blocks after all the others.

        Args:
            split_blocks (list[SplitBlock]): The split output blocks
        """
        for split_block in split_blocks:
            self._ss.write('\noutput == points\n')
            self._write_output_points_file(
                do_comp=split_block.do_comp,
                do_cov=split_block.do_cov,
                have_gis=False,
                have_csv=False,
                use_isodate=split_block.use_isodate,
                output_block=split_block.output_block
            )
            self._write_output_block(split_block.output_block, split_block.use_isodate)
            self._ss.write('end output\n')

    def _write_output_points_file(self, do_comp, do_cov, have_gis, have_csv, use_isodate, output_block):
        """Write a output points to a CSV or GIS shapefile.

        Args:
            do_comp (Component): The data_objects component
            do_cov (Coverage): The data_objects coverage
            have_gis (bool): True if we already have a GIS file in this output block
            have_csv (bool): True if we already have a CSV file in this output block
            use_isodate (bool): True if using ISODATE format
            output_block (Dataset): xarray Dataset for this output block

        Returns:
            tuple(bool, bool, bool): True if we have written a CSV file, True if we have written a GIS shapefile, True
                if we had to split this output block
        """
        write_csv = True
        split = False
        if do_comp:  # This is a TUFLOWFV Output Points coverage, check the export format.
            unique_name, model_name = do_comp.get_unique_name_and_model_name()
            if unique_name == 'OutputPointsComponent' and model_name == 'TUFLOWFV':
                py_comp = OutputPointsComponent(do_comp.main_file)
                if py_comp.data.info.attrs['export_format'] == 'Shapefile':  # Export to new shapefile format
                    write_csv = False
                    have_gis = True
                    rel_path = self._write_output_points_gis(do_cov, py_comp)
                    self._ss.write(f'  Read GIS PO == {rel_path}\n')
        if write_csv:  # Write output points to the old CSV format
            # Check for incompatible output formats
            if have_gis or have_csv:
                split = True
                self._logger.warning(
                    'Incompatible output points formats found in the same output block. Splitting '
                    'into multiple output blocks.'
                )
            else:
                have_csv = True
                rel_path = self._write_output_points_csv(do_cov)
                self._ss.write(f'  output points file == {rel_path}\n')
        return have_gis, have_csv, split

    def _write_output_block(self, output_block, use_isodate):
        """Write a single output block to the file.

        Args:
            output_block (xr.Dataset): The output variables Dataset
            use_isodate (bool): True if the simulation is using ISODATE time format
        """
        output_format = output_block.format.item()
        suffix = output_block.suffix.item()  # Keep track of these so we can warn if multiple will collide
        if output_format not in ['Flux', 'Mass', 'Transport']:
            self._ss.write(f'  output parameters == {self._get_output_data_letters(output_block)}\n')
        if output_block.define_start.item():
            start_time = output_block.output_start_date.item() if use_isodate else output_block.output_start.item()
            self._ss.write(f'  start output == {self._format_isodate(start_time) if use_isodate else start_time}\n')
        if output_block.define_final.item():
            final_time = output_block.output_final_date.item() if use_isodate else output_block.output_final.item()
            self._ss.write(f'  final output == {self._format_isodate(final_time) if use_isodate else final_time}\n')
        if output_block.define_interval.item():
            self._ss.write(f'  output interval == {output_block.output_interval.item()}\n')
        if suffix:
            self._ss.write(f'  suffix == {suffix}\n')
        if output_block.define_compression.item():
            self._ss.write(f'  output compression == {int(output_block.compression.item())}\n')
        if output_block.define_statistics.item():
            stat_type = output_block.statistics_type.item()
            if stat_type == 'Both':  # 'Both' is not a valid card value, just how we represent it in the GUI.
                stat_type = 'min, max'
            self._ss.write(f'  output statistics == {stat_type}\n')
            if output_block.define_statistics_dt.item():
                self._ss.write(f'  output statistics dt == {output_block.statistics_dt.item()}\n')
        if output_format != 'DATV':  # Can have multiple DATV blocks with no suffix but not other types.
            self._output_suffixes[output_format].append(suffix)

    def _write_output_points_csv(self, do_cov):
        """Write an Output Points CSV file.

        Args:
            do_cov (Coverage): The data_objects Coverage geometry

        Returns:
            str: Path to the csv file relative from the .fvc
        """
        points = do_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        if not points:
            self._logger.error(f'No feature nodes, vertices, or points found in coverage, "{do_cov.name}", which was '
                               'specified as the locations for an output block.')
            return ''
        x_coords = [point.x for point in points]
        y_coords = [point.y for point in points]
        ids = [point.id for point in points]
        df = pd.DataFrame({'x': x_coords, 'y': y_coords, 'ID': ids})
        # Come up with a unique name for the file.
        cov_name = self._get_unique_coverage_name(do_cov, self._used_output_cov_names, 'csv')
        abs_path, rel_path = io_util.build_input_filepaths(self._filename, 'model/csv', f'{cov_name}.csv')
        df.to_csv(abs_path, index=False)
        return rel_path

    def _write_output_points_gis(self, do_cov, py_comp):
        """Write an Output Points CSV file.

        Args:
            do_cov (Coverage): The data_objects Coverage geometry
            py_comp (OutputPointsComponent): The Python component of the TUFLOWFV Output Points coverage.

        Returns:
            str: Path to the csv file relative from the .fvc
        """
        # Come up with a unique name for the file.
        cov_name = self._get_unique_coverage_name(do_cov, self._used_output_cov_names, 'shp')
        abs_path, rel_path = io_util.build_input_filepaths(self._filename, 'model/gis', f'{cov_name}.shp')
        writer = ShapefileWriter(xms_data=self._xms_data, coverages=self._coverages)
        if writer.write_output_points_shapefile(filename=abs_path, do_cov=do_cov, py_comp=py_comp):
            return rel_path
        return ''  # No points in the coverage?

    def _get_output_data_letters(self, output_block):
        """Write all the output blocks to the file."""
        letter_list = []
        if output_block.depth_output.item():
            letter_list.append('D')
        if output_block.wse_output.item():
            letter_list.append('H')
        if output_block.bed_shear_stress_output.item():
            letter_list.append('Taub')
        if output_block.surface_shear_stress_output.item():
            letter_list.append('Taus')
        if output_block.velocity_output.item():
            letter_list.append('V')
        if output_block.velocity_mag_output.item():
            letter_list.append('Vmag')
        if output_block.vertical_velocity_output.item():
            letter_list.append('W')
        if output_block.bed_elevation_output.item():
            letter_list.append('ZB')
        if output_block.turb_visc_output.item():
            letter_list.append('turb_visc')
        if output_block.turb_diff_output.item():
            letter_list.append('turb_diff')
        if output_block.air_temp_output.item():
            letter_list.append('Air_temp')
        if output_block.evap_output.item():
            letter_list.append('Evap')
        if output_block.dzb_output.item():
            letter_list.append('DZB')
        if output_block.hazard_z1_output.item():
            letter_list.append('Hazard_z1')
        if output_block.hazard_zaem1_output.item():
            letter_list.append('hazard_zaem1')
        if output_block.hazard_zqra_output.item():
            letter_list.append('hazard_zqra')
        if output_block.lw_rad_output.item():
            letter_list.append('LW_rad')
        if output_block.mslp_output.item():
            letter_list.append('MSLP')
        if output_block.precip_output.item():
            letter_list.append('PRECIP')
        if output_block.rel_hum_output.item():
            letter_list.append('Rel_hum')
        if output_block.rhow_output.item():
            letter_list.append('Rhow')
        if output_block.sal_output.item():
            letter_list.append('Sal')
        if output_block.sw_rad_output.item():
            letter_list.append('SW_rad')
        if output_block.temp_output.item():
            letter_list.append('Temp')
        if output_block.turbz_output.item():
            letter_list.append('TURBZ')
        if output_block.w10_output.item():
            letter_list.append('W10')
        if output_block.wq_all_output.item():
            letter_list.append('WQ_ALL')
        if output_block.wq_diag_all_output.item():
            letter_list.append('WQ_DIAG_ALL')
        if output_block.wvht_output.item():
            letter_list.append('Wvht')
        if output_block.wvper_output.item():
            letter_list.append('Wvper')
        if output_block.wvdir_output.item():
            letter_list.append('Wvdir')
        if output_block.wvstr_output.item():
            letter_list.append('Wvstr')
        return ', '.join(letter_list)

    def _write_advanced_cards(self):
        """Write advanced cards (not directly supported by xmstuflowfv) to the bottom of the file."""
        advanced_cards = self._xms_data.sim_data.general.attrs['advanced_cards']
        if not advanced_cards:
            return
        self._ss.write(f'\n{self.COMMENT_LINE}! ADVANCED CARDS (not supported by SMS interface)\n')
        self._ss.write(advanced_cards)

    def _warn_if_colliding_suffixes(self):
        """Warn if any of the output blocks of the same type have colliding suffixes."""
        for output_type, suffixes in self._output_suffixes.items():
            unique_names = set(suffixes)
            if len(unique_names) != len(suffixes):
                self._logger.warning(
                    f'Multiple output blocks of type "{output_type}" found with conflicting suffixes: {suffixes}'
                )

    def _flush(self):
        """Flush in-memory stream to disk."""
        self._logger.info('Flushing in-memory stream to disk.')
        with open(self._filename, 'w') as f:
            self._ss.seek(0)
            shutil.copyfileobj(self._ss, f, self.BUFFER_SIZE)

    def write(self):
        """Write the TUFLOW-FV control file."""
        self._create_run_folders()
        self._write_projection_commands()
        if self._xms_data.do_ugrid:  # If no geometry, only write output options. Might be a "create GIS empties" run.
            self._write_simulation_commands()
            sim_reftime = self._write_time_commands()
            self._write_model_parameters()
            self._write_geometry_parameters()
            self._write_material_parameters()
            self._write_initial_conditions()
            self._write_boundary_conditions(sim_reftime)
            self._write_hydraulic_structures()
        else:
            self._logger.warning('No geometry found in the simulation. Only writing output options.')
        self._write_linked_simulations()
        # We write the gridded BCs here because they do not require a mesh. Often they are in an external 'read file'
        # simulation.
        self._write_grid_definitions()
        self._write_gridded_bcs()
        self._write_output_commands()
        self._write_advanced_cards()
        self._flush()
        return self._child_sims
