"""Class to copy TUFLOWFV shapefiles."""
# 1. Standard python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as xfs

# 4. Local modules
from xms.tuflowfv.file_io import io_util


class ShapefileCopier:
    """Class to copy TUFLOWFV shapefiles and build relative filepath references to a .fvc control file."""

    def __init__(self, fvc_filename):
        """Constructor.

        Args:
            fvc_filename (str): Absolute path to the TUFLOWFV control file that references the shapefile
        """
        self._fvc_filename = fvc_filename

    def copy(self, shp_filename, rel_paths):
        """Copy a shapefile and its associated files to the model/GIS folder.

        Args:
            shp_filename (str): Path to the .shp file
            rel_paths (list): Out variable that we will append the relative path from the .fvc file to the copied
                shapefile.
        """
        basename = os.path.basename(shp_filename)
        abs_path, rel_path = io_util.build_input_filepaths(self._fvc_filename, 'model/gis', basename)
        xfs.copyfile(shp_filename, abs_path)
        rel_paths.append(rel_path)

        basename = os.path.splitext(basename)[0]
        source_dir = os.path.dirname(shp_filename)
        dest_dir = os.path.dirname(abs_path)
        # Look for associated .dbf file - This is probably an error if it doesn't exist.
        source_dbf = os.path.join(source_dir, f'{basename}.dbf')
        if os.path.isfile(source_dbf):
            xfs.copyfile(source_dbf, os.path.join(dest_dir, os.path.basename(source_dbf)))
        # Look for associated .shx file
        source_shx = os.path.join(source_dir, f'{basename}.shx')
        if os.path.isfile(source_shx):
            xfs.copyfile(source_shx, os.path.join(dest_dir, os.path.basename(source_shx)))
        # Look for associated .prj file
        source_prj = os.path.join(source_dir, f'{basename}.prj')
        if os.path.isfile(source_prj):
            xfs.copyfile(source_prj, os.path.join(dest_dir, os.path.basename(source_prj)))
