"""Class to convert data_objects Coverages to TUFLOWFV shapefiles."""
# 1. Standard python modules
import logging
import os

# 2. Third party modules
import pandas as pd
import shapefile

# 3. Aquaveo modules
from xms.coverage.polygons.polygon_orienteer import get_polygon_point_lists
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.tuflowfv.data.material_data import MaterialData


class ShapefileWriter:
    """Class to convert data_objects Coverages to TUFLOWFV shapefiles."""
    # BC shapefile field names
    BC_FIELD_NAME = 'ID'
    BC_POLY_FIELD_NAME = 'Name'
    BC_FIELD_FLAGS = 'Flags'  # Not supposed to be written for internal boundaries
    # Material shapefile field names
    MATERIAL_FIELD_NAME = 'MATERIAL'
    # Z-line/point shapefile field names
    Z_FIELD_NAME = 'Elevation'
    # Output point shapefile field names
    OP_FIELD_TYPE = 'Type'
    OP_FIELD_LABEL = 'Label'
    OP_FIELD_COMMENT = 'Comment'
    OP_FIELD_VERT_MIN = 'Vert_min'
    OP_FIELD_VERT_MAX = 'Vert_max'

    def __init__(self, xms_data, coverages):
        """Constructor.

        Args:
            xms_data (XmsData): Simulation data retrieved from SMS
            coverages (CoverageCollector): The simulation coverage data
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._xms_data = xms_data
        self._coverages = coverages

    def _write_projection_file(self, filename):
        """Write the display projection to a .prj file.

        Args:
            filename (str): Absolute path to the exported shapefile
        """
        with open(f'{os.path.splitext(filename)[0]}.prj', 'w') as f:
            f.write(self._xms_data.query.display_projection.well_known_text)

    def _build_gis_output_dataframe(self, do_points, py_comp):
        """Build a DataFrame of an output points coverage.

        Args:
            do_points (list[Point]): The data_objects points in the output points coverage
            py_comp (OutputPointsComponent): The Python component of the output points coverage

        Returns:
            pd.DataFrame: DataFrame containing the location and attributes of the output points
        """
        # Build the dataframe
        self._xms_data.query.load_component_ids(py_comp, points=True)
        data = {'x': [], 'y': [], 'type': [], 'label': [], 'comment': [], 'vert_min': [], 'vert_max': []}
        xr_points = py_comp.data.points
        for i, point in enumerate(do_points):
            comp_id = py_comp.get_comp_id(TargetType.point, point.id)
            data['x'].append(point.x)
            data['y'].append(point.y)
            data['type'].append(None)  # Reserved for future use
            label = xr_points['label'][comp_id].item()
            if not label:  # Stringify the ID if no label set.
                label = str(i + 1)
            data['label'].append(label)
            comment = xr_points['label'][comp_id].item()
            if not comment:  # Comment is optional user input
                comment = None
            data['comment'].append(comment)
            # vert_min and vert_max are resesrved for when we support 3-D
            data['vert_min'].append(None)
            data['vert_max'].append(None)
        return pd.DataFrame(data)

    def _write_bc_point_shapefile(self, filename, cov_index, points):
        """Write a BC coverage to a shapefile.

        Args:
            filename (str): Path to the shapefile to write
            cov_index (int): Index of the BC coverage in the XmsData bc_covs list
            points (list[Point]): Points of the BC coverage
        """
        lookup = self._coverages.bc_lookup[cov_index][TargetType.point]
        # TUFLOW doesn't support pointz, so we will just drop the z value and write a point shapefile.
        with shapefile.Writer(filename, shapeType=shapefile.POINT) as f:
            f.field(self.BC_POLY_FIELD_NAME, 'C', size=100)
            for do_point in points:
                bc_id = lookup.get(do_point.id)
                if bc_id is None:
                    do_cov = self._xms_data.bc_covs[cov_index]
                    self._logger.error(
                        f'Could not find attributes for BC point {do_point.id} in coverage {do_cov.name}'
                    )
                    continue
                f.point(x=do_point.x, y=do_point.y)  # Creates the shape
                atts = self._coverages.bcs.get(bc_id)
                name = atts[0].name.item() if atts is not None else f'{bc_id}'
                f.record(name)  # Creates the record

    def _write_bc_arc_shapefile(self, filename, cov_index, arcs):
        """Write a BC coverage to a shapefile.

        Args:
            filename (str): Path to the shapefile to write
            cov_index (int): Index of the material coverage in the XmsData mat_covs list
            arcs (list[Arc]): Arcs of the BC coverage
        """
        lookup = self._coverages.bc_lookup[cov_index][TargetType.arc]
        with shapefile.Writer(filename, shapeType=shapefile.POLYLINE) as f:
            f.field(self.BC_FIELD_NAME, 'C', size=100)
            f.field(self.BC_FIELD_FLAGS, 'C')
            for do_arc in arcs:
                arc_points = do_arc.get_points(FilterLocation.PT_LOC_ALL)
                line = [(arc_point.x, arc_point.y) for arc_point in arc_points]
                f.line([line])  # Creates the shape
                bc_id = lookup.get(do_arc.id, MaterialData.UNASSIGNED_MAT)
                # Only write the `BD` flag if this is not a monitor line (has attributes)
                atts = self._coverages.bcs.get(bc_id)
                name = ''
                if atts is not None:
                    flag = 'BD'
                    name = atts[0].name.item()
                else:
                    flag = ''
                if not name:
                    name = f'{bc_id}'  # Use stringified feature ID if no atts (unassigned).
                f.record(name, flag)  # Creates the record

    def _write_bc_poly_shapefile(self, filename, cov_index, polys):
        """Write a BC coverage to a shapefile.

        Args:
            filename (str): Path to the shapefile to write
            cov_index (int): Index of the material coverage in the XmsData mat_covs list
            polys (list[Polygon]): Polygons of the BC coverage
        """
        lookup = self._coverages.bc_lookup[cov_index][TargetType.polygon]
        with shapefile.Writer(filename, shapeType=shapefile.POLYGON) as f:
            f.field(self.BC_FIELD_NAME, 'C', size=100)
            for do_poly in polys:
                polygon = get_polygon_point_lists(do_poly)
                f.poly(polygon)  # Creates the shape
                bc_id = lookup.get(do_poly.id)
                if bc_id is None:
                    do_cov = self._xms_data.bc_covs[cov_index]
                    self._logger.error(
                        f'Could not find attributes for BC polygon {do_poly.id} in coverage {do_cov.name}'
                    )
                    continue
                atts = self._coverages.bcs.get(bc_id)
                name = atts[0].name.item() if atts is not None else f'{bc_id}'
                f.record(name)  # Creates the record

    def write_bc_shapefile(self, filename, cov_idx):
        """Write a BC coverage to a shapefile.

        Args:
            filename (str): Path to the shapefile to write
            cov_idx (int): Index of the material coverage in the XmsData mat_covs list
        """
        do_cov = self._xms_data.bc_covs[cov_idx]
        do_points = do_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        do_arcs = do_cov.arcs
        do_polys = do_cov.polygons
        if not do_arcs and not do_polys and not do_points:
            self._logger.error(f'The coverage, "{do_cov.name}" contains no feature objects to export')
            return

        if do_points:
            self._write_bc_point_shapefile(filename=filename, cov_index=cov_idx, points=do_points)
        if do_arcs:
            self._write_bc_arc_shapefile(filename=filename, cov_index=cov_idx, arcs=do_arcs)
        if do_polys:
            self._write_bc_poly_shapefile(filename=filename, cov_index=cov_idx, polys=do_polys)
        self._write_projection_file(filename)

    def write_struct_shapefile(self, filename):
        """Write a Structure coverage to a shapefile.

        Args:
            filename (str): Path to the shapefile to write
        """
        do_comp = self._xms_data.structure_comp
        do_cov = self._xms_data.structure_cov
        do_arcs = do_cov.arcs
        if not do_arcs:
            self._logger.error(f'The coverage, "{do_cov.name}", was specified as a structure input but contains no '
                               'feature arcs.')
            return

        # Create a quick lookup table for the structure arcs
        xms_to_comp = comp_to_xms = {}
        # This abomination of a one-liner gets the actual dict we want from the overly nested dict of mappings
        if len(do_comp.comp_to_xms) > 0:
            comp_to_xms = next(iter(next(iter(do_comp.comp_to_xms.values())).values()))
        for comp_id, xms_id in comp_to_xms.items():
            xms_to_comp[xms_id[0]] = comp_id

        with shapefile.Writer(filename, shapeType=shapefile.POLYLINE) as f:
            f.field(self.BC_FIELD_NAME, 'C', size=100)
            f.field(self.BC_FIELD_FLAGS, 'C')
            for do_arc in do_arcs:
                if do_arc.id in xms_to_comp.keys():
                    arc_atts = do_comp.data.arcs.where(do_comp.data.arcs.comp_id == xms_to_comp[do_arc.id], drop=True)
                    name = arc_atts.name.data[0]
                    if not name:
                        name = f'{arc_atts.comp_id.data[0]}'  # Use stringified comp ID if no name is set.
                else:
                    self._logger.info(f"Arc {do_arc.id} was not associated with a component. "
                                      f"Using 'arc_{do_arc.id}' as the name.")
                    name = f"arc_{do_arc.id}"
                arc_points = do_arc.get_points(FilterLocation.PT_LOC_ALL)
                line = [(arc_point.x, arc_point.y) for arc_point in arc_points]
                self._logger.debug(f'Writing structure arc {name} with {len(line)} points')
                f.record(name, "")
                f.line([line])

    def write_material_shapefile(self, filename, cov_idx):
        """Write a Materials coverage to a shapefile.

        Args:
            filename (str): Path to the shapefile to write
            cov_idx (int): Index of the material coverage in the XmsData mat_covs list
        """
        do_cov = self._xms_data.mat_covs[cov_idx]
        do_polygons = do_cov.polygons
        if not do_polygons:
            self._logger.error(f'The coverage, "{do_cov.name}", was specified as a material input but contains no '
                               'feature polygons.')
            return

        mat_comp = self._xms_data.mat_comps[cov_idx]
        lookup = self._coverages.material_lookup[cov_idx]
        with shapefile.Writer(filename, shapeType=shapefile.POLYGON) as f:
            f.field(self.MATERIAL_FIELD_NAME, 'I', decimal=0)
            for do_polygon in do_polygons:
                polygon = get_polygon_point_lists(do_polygon)
                f.poly(polygon)  # Creates the shape
                comp_id = mat_comp.get_comp_id(TargetType.polygon, do_polygon.id)
                mat_id = lookup.get(comp_id, MaterialData.UNASSIGNED_MAT)
                f.record(int(mat_id))  # Creates the record
        self._write_projection_file(filename)

    def write_zline_shapes(self, filename, do_cov):
        """Convert a map module feature coverage to a shapefile.

        Args:
            filename (str): Absolute path to the shapefile to write
            do_cov (xms.data_objects.parameters.Coverage): The coverage to write
        """
        do_arcs = do_cov.arcs
        if not do_arcs:
            self._logger.error(f'The coverage, "{do_cov.name}", was specified as a Z-line input but contains no '
                               'feature arcs.')
            return

        with shapefile.Writer(filename, shapeType=shapefile.POLYLINE) as f:
            f.field(self.Z_FIELD_NAME, 'N', decimal=5)
            for do_arc in do_arcs:
                arc_points = do_arc.get_points(FilterLocation.PT_LOC_ALL)
                line = []
                arc_z = []
                for arc_point in arc_points:
                    line.append((arc_point.x, arc_point.y))
                    arc_z.append(arc_point.z)
                f.line([line])  # Creates the shape
                f.record(sum(arc_z) / len(arc_z))  # Creates the record
        self._write_projection_file(filename)

    def write_zpoint_shapes(self, filename, do_cov):
        """Convert a map module feature coverage to a shapefile.

        Args:
            filename (str): Absolute path to the shapefile to write
            do_cov (xms.data_objects.parameters.Coverage): The coverage to write
        """
        do_points = do_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        if not do_points:
            self._logger.error(f'The coverage, "{do_cov.name}", was specified as a Z-point input but contains no '
                               'feature points, nodes, or vertices.')
            return

        with shapefile.Writer(filename, shapeType=shapefile.POINT) as f:
            f.field(self.Z_FIELD_NAME, 'N', decimal=5)
            for do_point in do_points:
                f.point(do_point.x, do_point.y)  # Creates the shape
                f.record(do_point.z)  # Creates the record
        self._write_projection_file(filename)

    def write_output_points_shapefile(self, filename, do_cov, py_comp):
        """Convert a map module feature coverage to a shapefile.

        Args:
            filename (str): Absolute path to the shapefile to write
            do_cov (xms.data_objects.parameters.Coverage): The coverage to write
            py_comp (OutputPointsComponent): The Python component of the TUFLOWFV Output Points coverage.
        """
        do_points = do_cov.get_points(FilterLocation.PT_LOC_DISJOINT)
        if not do_points:
            self._logger.error(f'No feature points found in coverage, "{do_cov.name}", which was specified as the '
                               'locations for an output block.')
            return False

        df = self._build_gis_output_dataframe(do_points, py_comp)
        with shapefile.Writer(filename, shapeType=shapefile.POINT) as f:
            f.field(self.OP_FIELD_TYPE, 'C', size=100)
            f.field(self.OP_FIELD_LABEL, 'C', size=100)
            f.field(self.OP_FIELD_COMMENT, 'C', size=100)
            f.field(self.OP_FIELD_VERT_MIN, 'N', decimal=5)
            f.field(self.OP_FIELD_VERT_MAX, 'N', decimal=5)
            for row in df.itertuples(index=False):
                f.point(row.x, row.y)  # Creates the shape
                f.record(row.type, row.label, row.comment, row.vert_min, row.vert_max)  # Creates the record
        self._write_projection_file(filename)
        return True
