"""TUFLOWFV solution file importer."""
# 1. Standard python modules
import logging
import os
import shlex

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem

# 4. Local modules
from xms.tuflowfv.file_io.datv_solution_reader import DatvSolutionReader
from xms.tuflowfv.file_io.netcdf_solution_reader import NetcdfSolutionReader
from xms.tuflowfv.file_io.xmdf_solution_reader import XmdfSolutionReader


class TuflowfvSolutionReader:
    """Class for reading TUFLOWFV solutions."""

    def __init__(self, sim_data, fvc_filename, xms_data):
        """Constructor.

        Args:
            sim_data (SimData): The simulation's component data
            fvc_filename (str): Path to the simulation's .fvc file
            xms_data (XmsData): The XmsData used to retrieve the Mesh2D/UGrid geometry if needed. If testing and need
                to avoid Query calls, mock the XmsData.ugrid_item and XmsData.xmugrid properties. Usually like to do
                all XMS queries before worker code, but the UGrid geometry is only required if there are DATV output
                blocks.
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._sim_data = sim_data
        self._fvc_filename = fvc_filename
        self._xms_data = xms_data
        self._sim_name = os.path.splitext(os.path.basename(self._fvc_filename))[0]
        self._netcdf_files = []
        self._datv_files = []
        self._xmdf_files = []
        self._xmdf_sim_groups = []  # Group paths in XMDF file to simulation groups. Parallel with self._xmdf_files.
        self._flux_files = []
        self._mass_files = []
        self._point_files = []
        self._builders = []  # [DatasetWriter] - the imported solution datasets

    def _add_datv_files(self, results_folder):
        """Gather all enabled DATV solution filenames.

        Args:
            results_folder (str): Path to the folder containing the results.
        """
        if self._datv_files:
            # All DATV files listed in the super file, even if they are in separate output blocks. Only read the super
            # file once or we will get duplicate datasets.
            return

        super_file = f'{self._sim_name}.ALL.sup'
        super_file = os.path.join(results_folder, super_file)
        if os.path.isfile(super_file):
            with open(super_file, 'r') as f:
                lines = f.readlines()
            for line in lines:
                if line.startswith('DATA '):
                    line = line.strip()
                    lexed_line = shlex.split(line, posix=False)
                    lexed_line = lexed_line[1].strip('"')
                    datv_filename = filesystem.resolve_relative_path(results_folder, lexed_line)
                    self._datv_files.append(datv_filename)
        else:
            self._logger.error(f'Unable to find super file for DATV output: {super_file}')

    def _find_filenames(self):
        """Find the filenames of the simulation's solution files."""
        results_folder = os.path.dirname(self._fvc_filename)
        output_dir = self._sim_data.output.attrs['output_dir']
        if output_dir:
            results_folder = filesystem.resolve_relative_path(results_folder, output_dir)

        output_data = self._sim_data.output
        index_name = next(iter(output_data.sizes.keys()))
        indices = output_data[index_name].data.tolist()
        for index in indices:
            output_block = output_data.where(output_data[index_name] == index, drop=True)
            format_type = output_block.format.item().lower()
            suffix = output_block.suffix.item()
            suffix = f'_{suffix}' if suffix else ''
            if format_type == 'datv':
                self._add_datv_files(results_folder)
            elif format_type == 'netcdf':
                self._netcdf_files.append(os.path.join(results_folder, f'{self._sim_name}{suffix}.nc'))
            elif format_type == 'xmdf':
                sim_name_with_suffix = f'{self._sim_name}{suffix}'
                self._xmdf_files.append(os.path.join(results_folder, f'{sim_name_with_suffix}.xmdf'))
                self._xmdf_sim_groups.append(sim_name_with_suffix)
            elif format_type == 'flux':
                self._flux_files.append(os.path.join(results_folder, f'{self._sim_name}_FLUX{suffix}.csv'))
            elif format_type == 'mass':
                self._mass_files.append(os.path.join(results_folder, f'{self._sim_name}_MASS{suffix}.csv'))
            elif format_type == 'points':
                self._point_files.append(os.path.join(results_folder, f'{self._sim_name}_POINTS{suffix}.csv'))

    def read(self):
        """Import the TUFLOWFV solution.

        Returns:
            list[DatasetWriter]: The imported datasets
        """
        self._find_filenames()
        # Read the NetCDF solution files.
        if self._netcdf_files:  # Only Query for project tree if we have something to read.
            netcdf_reader = NetcdfSolutionReader(self._netcdf_files, self._xms_data.ugrid_item)
            self._builders.extend(netcdf_reader.read())
        # Read the DATV solution files.
        if self._datv_files:  # Only Query for UGrid geometry if we have something to read.
            datv_reader = DatvSolutionReader(self._datv_files, self._xms_data.ugrid_item, self._xms_data.xmugrid)
            self._builders.extend(datv_reader.read())
        # Read the XMDF solution files.
        xmdf_reader = XmdfSolutionReader(self._xmdf_files, self._xmdf_sim_groups, self._xms_data.ugrid_item.uuid)
        self._builders.extend(xmdf_reader.read())
        return self._builders
