"""Creates a Structure coverage hidden component and Datasets."""
# 1. Standard python modules
import logging
import os
import uuid

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.components.display.display_options_io import write_display_option_ids
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.tuflowfv.components.structure_component import StructureComponent
from xms.tuflowfv.components.structure_component_display import StructureComponentDisplay
from xms.tuflowfv.components.tuflowfv_component import get_component_data_object, UNINITIALIZED_COMP_ID
from xms.tuflowfv.data import structure_data as structd
from xms.tuflowfv.data import structure_parameters as const
from xms.tuflowfv.file_io.io_util import create_component_folder


__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"


def build_structure_dataset(structure_dict, comp_id=UNINITIALIZED_COMP_ID):
    """Build an xarray Dataset for a single Structure arc/polygon and fill with values read from the .fvc file.

    Args:
        structure_dict (dict): The variables for the Structure arc/polygon  read from the .fvc file. Keys should match
            data variables in the StructureData 'arcs' and 'polygons' Datasets.
        comp_id (int): Comp id to assign to the new dataset. Leave default if appending the structure to the
            StructureData as that will increment the component id to the next available one.

    Returns:
        xr.Dataset: See description
    """
    default_data = structd.get_default_structure_data(fill=True)
    coords = {'comp_id': [comp_id]}  # This will get reset when we append to the entire Dataset
    dset = xr.Dataset(data_vars=default_data, coords=coords)
    for variable, value in structure_dict.items():
        dset[variable].loc[dict(comp_id=[comp_id])] = value
    return dset


class StructureComponentBuilder:
    """Class for building Structure components."""

    def __init__(self, cov_uuid, manager):
        """Constructor.

        Args:
            cov_uuid (str): UUID of the component's coverage geometry
            manager (StructureManager): The manager for the structure I/O
        """
        self._logger = logging.getLogger('xms.tuflowfv')
        self._cov_uuid = cov_uuid
        self._main_file = ''
        self._arc_atts = manager.get_all_arc_atts()
        self._polygon_atts = manager.get_all_poly_atts()
        self._arc_id_to_feature_id = manager.arc_struct_id_to_feature_id
        self._arc_id_to_feature_name = manager.arc_struct_id_to_feature_name
        self._poly_id_to_feature_id = manager.poly_struct_id_to_feature_id
        self._poly_id_to_feature_name = manager.poly_struct_id_to_feature_name
        # If we have a set, we need to use the string name from the file to find its partner.
        self._processed_connections = {}  # struct_id: comp_id
        self._set_mapping = {}  # set_id: (comp1_id, comp2_id)

    def _initialize_component(self):
        """Create the component's folder and data_object.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new Structure component
        """
        # Create a new UUID and folder for the component data
        comp_uuid = str(uuid.uuid4())
        struct_comp_dir = create_component_folder(comp_uuid)
        self._main_file = os.path.join(struct_comp_dir, structd.STRUCTURE_MAIN_FILE)
        # Create the data_object Component to send back to SMS
        return get_component_data_object(main_file=self._main_file, comp_uuid=comp_uuid,
                                         unique_name='StructureComponent')

    def _initialize_display(self, struct_comp, arc_att_ids, arc_comp_ids, poly_att_ids, poly_comp_ids):
        """Initialize the component display.

        Args:
            struct_comp (BcComponent): The Python Structure component object
            arc_att_ids (list[int]): The arc feature ids
            arc_comp_ids (list[int]): The arc component ids
            poly_att_ids (list[int]): The polygon feature ids
            poly_comp_ids (list[int]): The polygon component ids
        """
        # Update all the structure id files
        struct_comp_dir = os.path.dirname(self._main_file)
        display_helper = StructureComponentDisplay(struct_comp=struct_comp)
        display_helper.update_all_id_files(comp_path=struct_comp_dir)

        # Write component id and struct arc att ids to a file so we can initialize them in get_initial_display_options
        arc_att_id_file = os.path.join(struct_comp_dir, const.STRUCT_INITIAL_ARC_ATT_ID_FILE)
        write_display_option_ids(arc_att_id_file, arc_att_ids)
        arc_comp_id_file = os.path.join(struct_comp_dir, const.STRUCT_INITIAL_ARC_COMP_ID_FILE)
        write_display_option_ids(arc_comp_id_file, arc_comp_ids)
        poly_att_id_file = os.path.join(struct_comp_dir, const.STRUCT_INITIAL_POLY_ATT_ID_FILE)
        write_display_option_ids(poly_att_id_file, poly_att_ids)
        poly_comp_id_file = os.path.join(struct_comp_dir, const.STRUCT_INITIAL_POLY_COMP_ID_FILE)
        write_display_option_ids(poly_comp_id_file, poly_comp_ids)

    def _add_arc_atts(self, comp_data):
        """Add attribute data for arcs.

        Args:
            comp_data (StructureData): The structure component's data

        Returns:
            tuple: attribute ids, component ids
        """
        return self._add_atts(comp_data, self._arc_atts, TargetType.arc)

    def _add_poly_atts(self, comp_data):
        """Add attribute data for polygons.

        Args:
            comp_data (StructureData): The structure component's data

        Returns:
            tuple: attribute ids, component ids
        """
        return self._add_atts(comp_data, self._polygon_atts, TargetType.polygon)

    def _add_atts(self, comp_data, atts, target_type):
        """Add attribute data for the features.

        Args:
            comp_data (StructureData): The structure component's data
            target_type (TargetType): Feature type of attributes


        Returns:
            tuple: attribute ids, component ids
        """
        att_ids = []
        comp_ids = []
        if target_type == TargetType.arc:
            id_to_feature_id = self._arc_id_to_feature_id
            id_to_feature_name = self._arc_id_to_feature_name
        else:  # TargetType.polygon
            id_to_feature_id = self._poly_id_to_feature_id
            id_to_feature_name = self._poly_id_to_feature_name
        for struct_id, feature_id in id_to_feature_id.items():
            # Find the attributes for this structure.
            struct_dict = atts.get(struct_id)
            feature_name = struct_id = id_to_feature_name.get(struct_id)
            if not struct_dict and id_to_feature_name:  # Check if it had a string name in the file
                struct_id = feature_name
                struct_dict = atts.get(struct_id)
            if not struct_dict:
                continue

            # Use either the feature name or its stringified ID for the structure name.
            if feature_name:
                struct_dict['name'] = str(feature_name)
            else:
                struct_dict['name'] = str(struct_id)

            # Check if this is a linked structure. If so, create a relationship in the sets table.
            partner = struct_dict.get('connection', None)
            partner_comp_id = self._processed_connections.get(partner, None)
            comp_id = comp_data.add_structure_atts(target_type=target_type, dset=build_structure_dataset(struct_dict))
            if partner_comp_id is not None:
                self._set_mapping[len(self._set_mapping)] = partner_comp_id, comp_id

            # Keep track of the component id for this feature in case it is a linked structure.
            self._processed_connections[feature_name] = comp_id
            # Keep track of the component and feature ids for initializing display later.
            comp_ids.append(comp_id)
            att_ids.append(feature_id)
        return att_ids, comp_ids

    def _add_mapping_table(self, comp_data):
        """Add set relationships for the linked features.

        Args:
            comp_data (StructureData): The structure component's data
        """
        coord_vals = list(self._set_mapping.keys())
        if coord_vals:
            comp_data.info.attrs['next_set_id'] = max(coord_vals) + 1
        coords = {'set_id': np.array(list(self._set_mapping.keys()), np.int32)}
        comp1_id = np.array([comp_id[0] for comp_id in self._set_mapping.values()], dtype=np.int32)
        comp2_id = np.array([comp_id[1] for comp_id in self._set_mapping.values()], dtype=np.int32)
        data_vars = {
            'comp1_id': ('set_id', comp1_id),
            'comp2_id': ('set_id', comp2_id),
        }
        comp_data.sets = xr.Dataset(data_vars=data_vars, coords=coords)

    def build_structure_component(self):
        """Creates the structure coverage's hidden component.

        This method is for building a structure coverage from scratch.

        Returns:
            xms.data_objects.parameters.Component: data_object for the new BC component
        """
        # Set things up
        do_comp = self._initialize_component()
        py_comp = StructureComponent(self._main_file)
        comp_data = py_comp.data
        comp_data.info.attrs['cov_uuid'] = self._cov_uuid
        comp_data.info.attrs['export_format'] = 'Shapefile'  # Default to GIS though some nodestrings may be in the .2dm
        py_comp.cov_uuid = comp_data.info.attrs['cov_uuid']

        # Add the structure atts for arcs and polygons
        arc_att_ids, arc_comp_ids = self._add_arc_atts(comp_data=comp_data)
        poly_att_ids, poly_comp_ids = self._add_poly_atts(comp_data=comp_data)
        self._add_mapping_table(comp_data=comp_data)
        # Write data to disk
        comp_data.commit()
        self._initialize_display(py_comp, arc_att_ids, arc_comp_ids, poly_att_ids, poly_comp_ids)
        return do_comp
