"""The TUFLOWFV advanced materials dialog."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtGui import QDoubleValidator
from PySide2.QtWidgets import QLabel

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.validators.number_corrector import NumberCorrector

# 4. Local modules
from xms.tuflowfv.gui.advanced_materials_dialog_ui import Ui_AdvancedMaterialsDialog
from xms.tuflowfv.gui.unit_label_switcher import UnitsLabelSwitcher

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class AdvancedMaterialsDialog(XmsDlg):
    """The TUFLOWFV advanced materials dialog."""

    def __init__(self, parent, row_idx, df, display_projection):
        """Initializes the dialog.

        Args:
            parent (QWidget): Parent dialog
            row_idx (int): Index of the material row being edited
            df (pd.DataFrame): Materials DataFrame
            display_projection (Projection): The current display projection. Used for switching unit labels between SI
                and Imperial.
        """
        super().__init__(parent, 'xms.tuflowfv.gui.advanced_materials_dialog')
        self.help_url = 'https://www.xmswiki.com/wiki/SMS:TUFLOW_FV'
        self.row_idx = row_idx
        self.df = df
        self.ui = Ui_AdvancedMaterialsDialog()
        self.ui.setupUi(self)

        self.number_corrector = NumberCorrector(self)
        self.dbl_validator = QDoubleValidator(self)
        self.dbl_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)

        # Setup the dialog
        self._setup_ui()

        if 'METER' not in display_projection.vertical_units.upper():
            label_switcher = UnitsLabelSwitcher(self)
            label_switcher.switch_to_imperial([QLabel])

    def _setup_ui(self):
        """Setup dialog widgets."""
        if type(self.parent()).__name__ == 'ModelControlDialog':
            # If coming from model control, this is the unnamed global default material
            self.setWindowTitle('Materials (advanced) - Default material (set mat)')
        else:  # Add the material name to the window title
            self.setWindowTitle(f'Materials (advanced) - {self.df["name"].iloc[self.row_idx]}')
        self._add_validators()
        self._load_data()

        # Disable 3D options for now
        self.ui.grp_vert_viscosity_limits.setVisible(False)
        self.ui.grp_vert_diffusivity_limits.setVisible(False)
        # Disable AD (diffusivity) options for now
        self.ui.grp_horiz_diffusivity.setVisible(False)
        self.ui.grp_horiz_diffusivity_limits.setVisible(False)

    def _add_validators(self):
        """Add validators on all edit fields."""
        self.ui.edt_horiz_viscosity.setValidator(self.dbl_validator)
        self.ui.edt_horiz_viscosity.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_diffusivity.setValidator(self.dbl_validator)
        self.ui.edt_horiz_diffusivity.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_viscosity_min.setValidator(self.dbl_validator)
        self.ui.edt_horiz_viscosity_min.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_viscosity_max.setValidator(self.dbl_validator)
        self.ui.edt_horiz_viscosity_max.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_diffusivity_min.setValidator(self.dbl_validator)
        self.ui.edt_horiz_diffusivity_min.installEventFilter(self.number_corrector)
        self.ui.edt_horiz_diffusivity_max.setValidator(self.dbl_validator)
        self.ui.edt_horiz_diffusivity_max.installEventFilter(self.number_corrector)
        # 3D options
        self.ui.edt_vert_viscosity_min.setValidator(self.dbl_validator)
        self.ui.edt_vert_viscosity_min.installEventFilter(self.number_corrector)
        self.ui.edt_vert_viscosity_max.setValidator(self.dbl_validator)
        self.ui.edt_vert_viscosity_max.installEventFilter(self.number_corrector)
        self.ui.edt_vert_diffusivity_min.setValidator(self.dbl_validator)
        self.ui.edt_vert_diffusivity_min.installEventFilter(self.number_corrector)
        self.ui.edt_vert_diffusivity_max.setValidator(self.dbl_validator)
        self.ui.edt_vert_diffusivity_max.installEventFilter(self.number_corrector)
        # Bed elevation limits
        self.ui.edt_bed_elevation_min.setValidator(self.dbl_validator)
        self.ui.edt_bed_elevation_min.installEventFilter(self.number_corrector)
        self.ui.edt_bed_elevation_max.setValidator(self.dbl_validator)
        self.ui.edt_bed_elevation_max.installEventFilter(self.number_corrector)

    """
    File I/O
    """
    def _load_data(self):
        """Populate widget values from persistent data."""
        # Edit fields
        self.ui.edt_horiz_viscosity.setText(str(self.df['horizontal_eddy_viscosity'].iloc[self.row_idx]))
        self.ui.edt_horiz_viscosity_min.setText(str(self.df['horizontal_eddy_viscosity_minimum'].iloc[self.row_idx]))
        self.ui.edt_horiz_viscosity_max.setText(str(self.df['horizontal_eddy_viscosity_maximum'].iloc[self.row_idx]))
        self.ui.edt_horiz_diffusivity.setText(str(self.df['horizontal_scalar_diffusivity'].iloc[self.row_idx]))
        self.ui.edt_horiz_diffusivity_min.setText(
            str(self.df['horizontal_scalar_diffusivity_minimum'].iloc[self.row_idx])
        )
        self.ui.edt_horiz_diffusivity_max.setText(
            str(self.df['horizontal_scalar_diffusivity_maximum'].iloc[self.row_idx])
        )
        self.ui.edt_vert_viscosity_min.setText(str(self.df['vertical_eddy_viscosity_minimum'].iloc[self.row_idx]))
        self.ui.edt_vert_viscosity_max.setText(str(self.df['vertical_eddy_viscosity_maximum'].iloc[self.row_idx]))
        self.ui.edt_vert_diffusivity_min.setText(str(self.df['vertical_scalar_diffusivity_minimum'].iloc[self.row_idx]))
        self.ui.edt_vert_diffusivity_max.setText(str(self.df['vertical_scalar_diffusivity_maximum'].iloc[self.row_idx]))
        self.ui.edt_bed_elevation_min.setText(str(self.df['bed_elevation_minimum'].iloc[self.row_idx]))
        self.ui.edt_bed_elevation_max.setText(str(self.df['bed_elevation_maximum'].iloc[self.row_idx]))
        # Toggles
        self.ui.grp_horiz_viscosity.setChecked(
            bool(int(self.df['override_horizontal_eddy_viscosity'].iloc[self.row_idx]))
        )
        self.ui.grp_horiz_viscosity_limits.setChecked(
            bool(int(self.df['override_horizontal_eddy_viscosity_limits'].iloc[self.row_idx]))
        )
        self.ui.grp_horiz_diffusivity.setChecked(
            bool(int(self.df['override_horizontal_scalar_diffusivity'].iloc[self.row_idx]))
        )
        self.ui.grp_horiz_diffusivity_limits.setChecked(
            bool(int(self.df['override_horizontal_scalar_diffusivity_limits'].iloc[self.row_idx]))
        )
        self.ui.grp_vert_viscosity_limits.setChecked(
            bool(int(self.df['override_vertical_eddy_viscosity_limits'].iloc[self.row_idx]))
        )
        self.ui.grp_vert_diffusivity_limits.setChecked(
            bool(int(self.df['override_vertical_scalar_diffusivity_limits'].iloc[self.row_idx]))
        )
        self.ui.grp_bed_elevation_limits.setChecked(
            bool(int(self.df['override_bed_elevation_limits'].iloc[self.row_idx]))
        )
        self.ui.tog_spatial_reconstruction.setChecked(bool(int(self.df['spatial_reconstruction'].iloc[self.row_idx])))

    def _save_data(self):
        """Saves the dialog data to the SimData.

        Does not flush data to disk, just updates in memory xarray.Datasets. Call flush() on the
        SimData to write to disk.
        """
        # Edit fields
        self.df['horizontal_eddy_viscosity'].values[self.row_idx] = float(self.ui.edt_horiz_viscosity.text())
        self.df['horizontal_eddy_viscosity_minimum'].values[self.row_idx] = \
            float(self.ui.edt_horiz_viscosity_min.text())
        self.df['horizontal_eddy_viscosity_maximum'].values[self.row_idx] = \
            float(self.ui.edt_horiz_viscosity_max.text())
        self.df['horizontal_scalar_diffusivity'].values[self.row_idx] = float(self.ui.edt_horiz_diffusivity.text())
        self.df['horizontal_scalar_diffusivity_minimum'].values[self.row_idx] = \
            float(self.ui.edt_horiz_diffusivity_min.text())
        self.df['horizontal_scalar_diffusivity_maximum'].values[self.row_idx] = \
            float(self.ui.edt_horiz_diffusivity_max.text())
        self.df['vertical_eddy_viscosity_minimum'].values[self.row_idx] = float(self.ui.edt_vert_viscosity_min.text())
        self.df['vertical_eddy_viscosity_maximum'].values[self.row_idx] = float(self.ui.edt_vert_viscosity_max.text())
        self.df['vertical_scalar_diffusivity_minimum'].values[self.row_idx] = \
            float(self.ui.edt_vert_diffusivity_min.text())
        self.df['vertical_scalar_diffusivity_maximum'].values[self.row_idx] = \
            float(self.ui.edt_vert_diffusivity_max.text())
        self.df['bed_elevation_minimum'].values[self.row_idx] = float(self.ui.edt_bed_elevation_min.text())
        self.df['bed_elevation_maximum'].values[self.row_idx] = float(self.ui.edt_bed_elevation_max.text())
        # Toggles
        self.df['override_horizontal_eddy_viscosity'].values[self.row_idx] = \
            1 if self.ui.grp_horiz_viscosity.isChecked() else 0
        self.df['override_horizontal_eddy_viscosity_limits'].values[self.row_idx] = \
            1 if self.ui.grp_horiz_viscosity_limits.isChecked() else 0
        self.df['override_horizontal_scalar_diffusivity'].values[self.row_idx] = \
            1 if self.ui.grp_horiz_diffusivity.isChecked() else 0
        self.df['override_horizontal_scalar_diffusivity_limits'].values[self.row_idx] = \
            1 if self.ui.grp_horiz_diffusivity_limits.isChecked() else 0
        self.df['override_vertical_eddy_viscosity_limits'].values[self.row_idx] = \
            1 if self.ui.grp_vert_viscosity_limits.isChecked() else 0
        self.df['override_vertical_scalar_diffusivity_limits'].values[self.row_idx] = \
            1 if self.ui.grp_vert_diffusivity_limits.isChecked() else 0
        self.df['override_bed_elevation_limits'].values[self.row_idx] = \
            1 if self.ui.grp_bed_elevation_limits.isChecked() else 0
        self.df['spatial_reconstruction'].values[self.row_idx] = \
            1 if self.ui.tog_spatial_reconstruction.isChecked() else 0

    def accept(self):
        """Save dialog data on accepted."""
        self._save_data()
        super().accept()
