"""A Qt XY Series dialog."""
# 1. Standard python modules
import datetime
import os

# 2. Third party modules
from matplotlib.backends.backend_qt5agg import FigureCanvas, NavigationToolbar2QT as NavigationToolbar
import matplotlib.dates as mdates
from matplotlib.figure import Figure
from matplotlib.ticker import AutoLocator
import numpy as np
import pandas as pd
from pandas.plotting import register_matplotlib_converters
from PySide2.QtCore import QItemSelectionModel, QSize, Qt
from PySide2.QtGui import QDoubleValidator
from PySide2.QtWidgets import QApplication, QDialogButtonBox, QHeaderView, QSplitter

# 3. Aquaveo modules
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.models.qx_pandas_table_model import QxPandasTableModel
from xms.guipy.settings import SettingsManager
from xms.guipy.time_format import ISO_DATETIME_FORMAT
from xms.guipy.validators.number_corrector import NumberCorrector
from xms.guipy.widgets import widget_builder

# 4. Local modules
from xms.tuflowfv.data import bc_data as bcd
from xms.tuflowfv.file_io.bc_csv_reader import BcCsvReader
from xms.tuflowfv.gui import assign_bc_consts as const, gui_util
from xms.tuflowfv.gui.bc_series_editor_ui import Ui_BcSeriesEditor
from xms.tuflowfv.gui.bc_series_pandas_model import BcSeriesPandasModel
from xms.tuflowfv.gui.bc_series_table_view import is_float
from xms.tuflowfv.gui.import_csv_dialog import ImportCsvDialog
from xms.tuflowfv.gui.netcdf_dump_viewer_dialog import NetCDFDumpViewerDialog


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


# Default maximum for QSpinBox is 100. I don't want to limit the number of rows. I also hope the curve doesn't have
# 2 billion rows, but I am betting this is the true maximum for the C++ Qt wrappings.
MAX_C_INT = 2147483647


def populate_time_widgets(dlg, bc_data, time_format):
    """Populate time widgets from persistent data.

    Notes:
        Used by the BcSeriesEditor and the AssignBcDialog. Only works because they have consistently named widgets.

    Args:
        dlg (QDialog): The dialog containing the widgets to populate
        bc_data (xr.Dataset): BC attributes for the BC being edited
        time_format (str): The current SMS user preference for formatting absolute datetimes (Qt format specifiers)
    """
    # Load data into widgets
    dlg.ui.grp_reference_time.setChecked(bool(int(bc_data['define_reference_time'][0].item())))
    dlg.ui.tog_use_isodate.setChecked(bool(int(bc_data['use_isodate'][0].item())))
    dlg.ui.edt_reference_time.setText(str(bc_data['reference_time_hours'][0].item()))
    # Convert Qt datetime strings to Python datetime object
    gui_util.intialize_datetime_widget(bc_data['reference_time_date'][0].item(), dlg.ui.date_reference_time,
                                       time_format)
    gui_util.add_combobox_options(const.BC_CBX_OPTS, dlg.ui.cbx_time_units)
    gui_util.set_combobox_from_data(dlg.ui.cbx_time_units, str(bc_data['time_units'][0].item()))


def extract_time_widget_data(dlg, bc_data):
    """Update persistent data from current widget values.

    Notes:
        Used by the BcSeriesEditor and the AssignBcDialog. Only works because they have consistently named widgets.

    Args:
        dlg (QDialog): The dialog containing the widgets to extract
        bc_data (xr.Dataset): BC attributes for the BC being edited
    """
    bc_data['define_reference_time'][0] = 1 if dlg.ui.grp_reference_time.isChecked() else 0
    bc_data['use_isodate'][0] = 1 if dlg.ui.tog_use_isodate.isChecked() else 0
    bc_data['reference_time_hours'][0] = float(dlg.ui.edt_reference_time.text())
    bc_data['reference_time_date'][0] = dlg.ui.date_reference_time.dateTime().toString(ISO_DATETIME_FORMAT)
    bc_data['time_units'][0] = dlg.ui.cbx_time_units.itemData(dlg.ui.cbx_time_units.currentIndex())


class BcSeriesEditor(XmsDlg):
    """Dialog to display and edit a BC curve Series.

    From https://doc-snapshots.qt.io/qtforpython/tutorials/datavisualize/plot_datapoints.html
    """
    def __init__(self, bc_data, bc_curve, series_name, x_is_time, defaults, time_formats, dist_df, variable_dset,
                 netcdf_filename, parent=None):
        """Constructor that takes an in-memory DataFrame.

        Args:
            bc_data (xarray.Dataset): Dataset for the BC feature being assigned
            bc_curve (DataFrame): The DataFrame.
            series_name (str): The name of the series.
            x_is_time (bool): True if the X column is time. If True dtype may switch between datetime64 and float.
            defaults (Union[None, list[float]]): Default values for missing data, if defined. Should be length of the
                number of y-columns.
            time_formats (tuple(str)): The current SMS user preference for formatting absolute datetimes. Should have
                specifiers for (strftime, qt)
            dist_df (Optional[np.array]): Only for curtain boundaries. DataFrame with single column that contains the
                distance along the arc where data values are located (X-axis).
            variable_dset (Optional[xr.Dataset]): Only for curtain boundaries. Lazily opened handle to the NetCDF data.
                Will read timesteps from file as user changes time rows.
            netcdf_filename (Optional[str]): Only for curtain boundaries. Absolute path to the curtain NetCDF file. For
                viewing NetCDF dump info.
            parent (QWidget): The dialog parent
        """
        super().__init__(parent, 'xms.tuflowfv.gui.bc_series_editor')
        self.series_name = series_name  # == bc_type (str card)
        self.x_is_time = x_is_time
        self.bc_data = bc_data
        self.defaults = defaults
        self.time_format_std = time_formats[0]
        self.time_format_qt = time_formats[1]
        self.curtain_dist = dist_df  # Only for curtain boundaries
        self.curtain_dset = variable_dset  # Only for curtain boundaries
        self.raw_curtain_times_df = None  # The raw NetCDF time array. Backed up/restored when switching ISODATE view
        self.netcdf_filename = netcdf_filename
        self.netcdf_dump_dlg = None  # Modeless dialog for displaying curtain NetCDF info dump
        self.warned_about_curtain_editing = False  # Only for curtain boundaries
        self.model = None
        self.filter_model = None
        self.plot_model = None
        self.splitter = None
        self.figure = None  # matplotlib.figure Figure
        self.canvas = None  # matplotlib.backends.backend_qt5agg FigureCanvas
        self.ax = None      # matplotlib Axes
        self.in_on_spn_rows = False
        self.last_x = None
        self.in_setup = True
        self.can_add_rows = True
        self.previous_unit_index = -1

        register_matplotlib_converters()  # I think this is required by mdates.date2num call below

        self.ui = Ui_BcSeriesEditor()
        self._setup_ui(bc_curve)
        self.ui.tbl_xy_data.x_is_time_column = x_is_time

    """
    UI setup
    """
    def _setup_ui(self, df):
        """Sets up the UI.

        Args:
            df (DataFrame): Pandas DataFrame
        """
        self.ui.setupUi(self)
        self._setup_model(df)
        self._setup_table()
        self._setup_plot()
        self._setup_rows_spin_control()
        self._setup_context_menus()
        self._setup_ui_for_bc_type()
        self._add_splitter()
        self._connect_signals()
        self._hidden_button_hack()
        self.in_setup = False

    """
    Data model/view setup
    """
    def _setup_model(self, df):
        """Sets up the model."""
        self.model = BcSeriesPandasModel(df, self.time_format_std, self)
        _, defaults = self._column_info()
        self.model.set_default_values(defaults)
        if self.curtain_dist is not None:  # If a curtain BC, only the time column in the table and it is read-only
            self.model.set_read_only_columns({0})
        self.model.dataChanged.connect(self.on_data_changed)
        self.model.set_show_nan_as_blank(True)
        self._setup_plot_model()

    def _setup_plot_model(self):
        """Creates a QxPandasTableModel for the plot.

        Alters the data if necessary, e.g. stair-step.
        """
        if self.curtain_dist is not None:
            # If curtain BC, x-column is distance along nodestring, y columns are added later based on the selected row
            # in the read-only XY series editor (current timestep).
            self.plot_model = QxPandasTableModel(self.curtain_dist)
        else:  # Create a copy of the model for the plot if plotting curve in XY series editor (non-curtain BC)
            self.plot_model = QxPandasTableModel(self.model.data_frame.copy())
        # Stop if the model is empty
        if self.plot_model.rowCount() == 0:
            return

        # Convert all non-numeric entries in the Y column to math.nan (mathplotlib requires this)
        if self.curtain_dist is None:  # No y-column in initial curtain BC distance DataFrame
            for i in range(1, len(self.plot_model.data_frame.columns)):
                if pd.api.types.is_object_dtype(self.plot_model.data_frame.iloc[:, i]):
                    self.plot_model.data_frame.iloc[:, i] = pd.to_numeric(self.plot_model.data_frame.iloc[:, i],
                                                                          errors='coerce')

    def _setup_table(self):
        """Sets up the table."""
        self.ui.tbl_xy_data.size_to_contents = True
        widget_builder.style_table_view(self.ui.tbl_xy_data)
        self.ui.tbl_xy_data.setModel(self.model)

        # QTableView Headers
        resize_mode = QHeaderView.Interactive
        horizontal_header = self.ui.tbl_xy_data.horizontalHeader()
        vertical_header = self.ui.tbl_xy_data.verticalHeader()
        self.ui.tbl_xy_data.resizeColumnsToContents()
        horizontal_header.setSectionResizeMode(0, QHeaderView.Stretch)
        horizontal_header.setStretchLastSection(True)
        horizontal_header.setSectionResizeMode(resize_mode)
        vertical_header.setSectionResizeMode(resize_mode)

    """
    Plot setup
    """
    def _setup_plot(self):
        """Sets up the plot."""
        self.figure = Figure()
        self.figure.set_tight_layout(True)  # Frames the plots
        self.canvas = FigureCanvas(self.figure)
        self.canvas.setMinimumWidth(100)  # So user can't resize it to nothing
        self.ui.vlay_grp_box.addWidget(self.canvas)
        self.ui.vlay_grp_box.addWidget(NavigationToolbar(self.canvas, self))
        self._add_series()

    def _add_series(self):
        """Adds the XY line series to the plot."""
        # Add a series for each column in the dataframe
        num_cols = len(self.model.data_frame.columns) if self.curtain_dist is None else \
            len(bcd.ARC_BC_TYPES[self.series_name]) - 1  # We do not plot time for curtain BCs
        if not self.ax:
            self.ax = self.figure.add_subplot(111)
        self.ax.clear()
        self.ax.set_title(self.series_name)
        self.ax.grid(True)
        for i in range(1, num_cols):
            # Get the default value for the y-column, if there is one
            default = self.defaults[i - 1] if self.defaults else None
            self._add_line_series(i, default)
        self.ax.legend(loc='best')
        self.canvas.draw()

    def _add_line_series(self, data_column, default_value):
        """Adds an XY line series to the plot.

        Args:
            data_column (int): Index of the column to plot against the X column
            default_value (Union[None, float]): The default value for the Y column if defined else None.
        """
        if self.plot_model.rowCount() == 0:
            return  # No data to plot or a variable we are currently hiding
        elif self.curtain_dist is not None and const.BC_VARIABLE_HIDE_IDX.get(self.series_name) - 1 == data_column:
            # This is a curtain boundary variable we are not ready to support yet. Need to subtract one from the column
            # index because we exclude time from the plot DataFrame for curtain boundaries.
            return

        # Extract data for the series
        y_column, series_label = self._extract_y_column_for_series(data_column)
        x_column = self.plot_model.data_frame.iloc[:, 0]
        x_is_date_times = (
            pd.core.dtypes.common.is_timedelta64_dtype(x_column)) \
            or pd.core.dtypes.common.is_datetime64_any_dtype(x_column)
        x_column = mdates.date2num(x_column) if x_is_date_times else x_column
        # Plot the data
        self._plot_series_with_default(default_value, series_label, x_column, y_column)
        self._setup_x_tick_formatting(x_is_date_times)
        # Set axis titles
        x_label = self.get_time_units_label() if self.x_is_time else self.plot_model.data_frame.columns[0]
        self.ax.set_xlabel(x_label)
        self.ax.set_ylabel(series_label)

    def _extract_y_column_for_series(self, data_column):
        """Get the y-column data for a series.

        Args:
            data_column (int): Index of the column to plot against the X column

        Returns:
            tuple(str, np.array): y-column data, the series label
        """
        if self.curtain_dist is None:  # If a non-curtain BC get Y column from curve DataFrame
            y_column = self.plot_model.data_frame.iloc[:, data_column]
            series_label = self.plot_model.data_frame.columns[data_column]
        else:  # If a curtain BC, read current time step (selected XY series table row) from NetCDF file.
            y_column, series_label = self._read_netcdf_for_selected_row(data_column)
        return y_column, series_label

    def _plot_series_with_default(self, default_value, series_label, x_column, y_column):
        """Plot a series and a curve for its default values, if applicable.

        Args:
            default_value (float): Default value to be used where data is missing
            series_label (str): Name to give the series
            x_column (np.array): The x-column values
            y_column (np.array): The y-column values
        """
        y_has_data = True
        if default_value is not None:  # Plot NaN values if default provided
            nan_mask = np.isnan(y_column)
            if nan_mask.any():  # Check if there is missing data in the curve
                y_has_data = False
                if not nan_mask.all():  # If there is any data in the column, plot it with gaps first.
                    self.ax.plot(x_column, y_column, label=series_label)
                # Now plot the values where the default value will be applied.
                y_column[nan_mask] = default_value
                y_column[~nan_mask] = np.nan
                self.ax.plot(x_column, y_column, label=f'{series_label} - default')
        if y_has_data:  # No missing data, just plot data as is
            self.ax.plot(x_column, y_column, label=series_label)

    def _setup_x_tick_formatting(self, x_is_date_times):
        """Setup formatting for tick labels on the plot x-axis.

        Args:
            x_is_date_times (bool): True if x-column is absolute datetimes
        """
        if x_is_date_times:  # Format time column if absolute datetimes
            formatter = mdates.DateFormatter(self.time_format_std)
            self.ax.xaxis.set_major_formatter(formatter)
            self.ax.xaxis.set_major_locator(mdates.AutoDateLocator())
            self.figure.autofmt_xdate()
        else:
            self.ax.xaxis.set_major_locator(AutoLocator())

    """
    Widget setup
    """
    def _setup_ui_for_bc_type(self):
        """Setup widgets widgets based on the BC type whose curve is being edited/viewed."""
        is_curtain = self.curtain_dist is not None
        if self.x_is_time or is_curtain:  # We don't plot time for curtain BCs but still want widgets enabled
            self.ui.tog_use_isodate.stateChanged.connect(self.on_tog_use_isodate)
            self.ui.cbx_time_units.currentIndexChanged.connect(self.on_time_units_changed)
            self._load_time_widgets()
            if is_curtain:
                self._setup_ui_for_curtain()
        else:  # Not a curve through time, hide time units and reference date widgets.
            self.ui.cbx_time_units.setVisible(False)
            self.ui.lbl_time_units.setVisible(False)
            self.ui.grp_reference_time.setVisible(False)
        if not is_curtain:  # Hide view as ISODATE command if not a curtain
            self.ui.tog_view_isodate.setVisible(False)

    def _setup_ui_for_curtain(self):
        """Setup UI for curtain-specific stuff."""
        # Only want this slot connected if a curtain BC
        self.ui.tbl_xy_data.selectionModel().selectionChanged.connect(self.on_timestep_changed)
        # Spinbox changes selected timestep (row) if curtain BC
        self.ui.txt_row_count.setText('Current time step:')
        # File button brings up ncdump info dialog
        self.ui.btn_file.setText('NetCDF Info...')
        # Connect slots to update the ISODATE view if needed
        self.ui.tog_view_isodate.stateChanged.connect(self.on_tog_view_isodate)
        self.ui.grp_reference_time.toggled.connect(self.update_isodate_view)
        self.ui.tog_use_isodate.stateChanged.connect(self.update_isodate_view)
        self.ui.edt_reference_time.editingFinished.connect(self.update_isodate_view)
        self.ui.date_reference_time.editingFinished.connect(self.update_isodate_view)

    def _add_splitter(self):
        """Adds a QSplitter between the tables so the sizes can be adjusted."""
        # The only way this seems to work right is to parent it to
        # self and then insert it into the layout.
        self.splitter = QSplitter(self)
        self.splitter.setOrientation(Qt.Horizontal)
        self.splitter.addWidget(self.ui.wid_left)
        self.splitter.addWidget(self.ui.grp_plot)
        # Just use a fixed starting width of 300 for the table for now
        self.splitter.setSizes([300, 600])
        self.splitter.setChildrenCollapsible(False)
        self.splitter.setStyleSheet('QSplitter::handle:horizontal { background-color: lightgrey; }'
                                    'QSplitter::handle:vertical { background-color: lightgrey; }')
        pos = self.ui.hlay_main.indexOf(self.ui.grp_plot)
        self.ui.hlay_main.insertWidget(pos, self.splitter)

    def _connect_signals(self):
        """Connect Qt widget signal/slots."""
        self.ui.spn_row_count.valueChanged.connect(self.on_spn_rows)
        if self.curtain_dist is not None:  # For curtain boundaries, button brings up NetCDF info dialog
            self.ui.btn_file.clicked.connect(self.on_btn_netcdf_info)
        else:  # For curve BCs, button imports from a CSV
            self.ui.btn_file.clicked.connect(self.on_btn_import)

    def _hidden_button_hack(self):
        """Adds a hidden button with the default focus to the standard QDialogButtonBox.

        Notes:
            This is a workaround because Qt gives focus to the 'Ok' button by default, so pressing enter in the
            spreadsheet often leads to inadvertently accepting changes in the dialog.
        """
        button = self.ui.buttonBox.addButton('', QDialogButtonBox.ActionRole)
        button.setDefault(True)
        button.hide()

    def _setup_rows_spin_control(self):
        """Sets up the 'Number of rows' spin control."""
        if self.curtain_dist is not None:  # For curtain boundaries, spinbox controls current timestep
            self.ui.spn_row_count.setMinimum(1)
            self.ui.spn_row_count.setMaximum(len(self.model.data_frame.index))
            self.ui.spn_row_count.setValue(1)
        else:  # Otherwise it inserts/removes rows
            self.ui.spn_row_count.setMinimum(0)
            self.ui.spn_row_count.setMaximum(MAX_C_INT)
            self.ui.spn_row_count.setValue(self.model.rowCount())
        self.ui.spn_row_count.setKeyboardTracking(False)
        self.ui.spn_row_count.setEnabled(self.can_add_rows)

    def _setup_context_menus(self):
        """Sets up the context menus."""
        if self.curtain_dist is not None:
            return  # Don't want copy/paste for curtain BCs
        self.ui.tbl_xy_data.verticalHeader().setContextMenuPolicy(Qt.CustomContextMenu)
        self.ui.tbl_xy_data.verticalHeader().customContextMenuRequested.connect(self.on_index_column_click)
        self.ui.tbl_xy_data.setContextMenuPolicy(Qt.CustomContextMenu)
        self.ui.tbl_xy_data.customContextMenuRequested.connect(self.on_right_click)

    def _column_info(self):
        """Returns the column names and default values.

        Returns:
            (:obj:`tuple`): tuple containing:
                - columns (list of str): List of column names.
                - defaults (dict of str and values): Dictionary of column default values.

        """
        columns = list(self.model.data_frame.columns)
        defaults = {}
        for column in columns:
            if pd.core.dtypes.common.is_timedelta64_dtype(self.model.data_frame[column]) \
                    or pd.core.dtypes.common.is_datetime64_any_dtype(self.model.data_frame[column]):
                defaults[column] = self._get_bc_reference_datetime()
            else:
                defaults[column] = 0.0
        return columns, defaults

    def _update_table_view(self):
        """Update the table view when the data has changed (significantly, not user edits)."""
        # Make sure the header changes to reflect the new units, even if we didn't do a conversion.
        self.model.headerDataChanged.emit(Qt.Horizontal, 0, 0)
        self.ui.tbl_xy_data.viewport().update()
        self.ui.tbl_xy_data.resizeColumnsToContents()
        self.ui.tbl_xy_data.horizontalHeader().setStretchLastSection(True)

    """
    Time conversions
    """
    def _get_bc_reference_datetime(self):
        """Get the reference time for the BC curve.

        If not defined, returns the current TUFLOWFV zero time. If defined as an ISODate, returns that. If defined as
        an offset relative to the TUFLOWFV in hours, return the current TUFLOWFV zero time plus the hour offset.

        Returns:
            datetime.datetime: See description
        """
        reftime = gui_util.get_tuflow_zero_time()
        if self.ui.grp_reference_time.isChecked():
            if self.ui.tog_use_isodate.isChecked():
                reftime = self.ui.date_reference_time.dateTime().toPython()
            else:  # Offset in hours from simulation start, use SMS zero time since we don't know
                reftime = reftime + datetime.timedelta(hours=float(self.ui.edt_reference_time.text()))
        return reftime

    def _convert_time_offset_units(self, from_units_idx, to_units_idx):
        """Convert from one time offset units to another.

        Args:
            from_units_idx (int): Combobox option index of the current units
            to_units_idx (int): Combobox option index of the new units
        """
        conversion_value = const.UNIT_CONVERSIONS[from_units_idx][to_units_idx]
        self.model.data_frame.loc[:, 'Time'] *= conversion_value

    def _convert_offset_to_isodate(self, from_units_idx, time_column):
        """Convert from a time offset to a ISODate.

        Args:
            from_units_idx (int): Combobox option index of the new units
            time_column (str): Name of the time column in the DataFrame
        """
        # If a reference time has been defined for the BC use that, else the SMS zero time
        times = self.model.data_frame.loc[:, time_column].values.tolist()
        time_units = const.BC_CBX_OPTS['cbx_time_units'][0][from_units_idx].lower()
        timedeltas = [datetime.timedelta(**{time_units: time_value}) for time_value in times]
        reftime = self._get_bc_reference_datetime()
        abs_times = [reftime + timedelta for timedelta in timedeltas]
        self.model.data_frame[time_column] = pd.to_datetime(abs_times)

    def _convert_isodate_to_offset(self, to_units_idx):
        """Convert from an ISODate to a time offset.

        Args:
            to_units_idx (int): Combobox option index of the new units
        """
        reftime = self._get_bc_reference_datetime()
        times = self.model.data_frame.loc[:, 'Time'].to_list()
        time_units = const.BC_CBX_OPTS['cbx_time_units'][0][to_units_idx][0]
        if time_units != 'D':  # Day kwarg is capital, others are lower case
            time_units = time_units.lower()
        one_unit = np.timedelta64(1, time_units)
        time_offsets = [(time - reftime) / one_unit for time in times]
        self.model.data_frame['Time'] = time_offsets

    def _convert_time_units(self, index, old_index):
        """Convert the units of data in the time column when user changes the time units combobox.

        Args:
            index (int): Combobox option index of the new units
            old_index (int): Combobox option index of the old units
        """
        self.ui.tbl_xy_data.pasting = True
        QApplication.setOverrideCursor(Qt.WaitCursor)  # This can take a second with a big curve
        # If switching from Isodate to an offset subtract consecutive rows.
        if old_index == const.ISODATE_UNITS_IDX:
            self._convert_isodate_to_offset(index)
            self.model.defaults['Time'] = 0.0  # Set float dtype for new row default
        # If switching units from an offset to Isodate, construct datetimes relative to the BC reference time if
        # defined as an Isodate else use the SMS zero time.
        elif index == const.ISODATE_UNITS_IDX:
            time_var = self.model.data_frame.columns[0] if self.curtain_dist is not None else 'Time'
            self.model.defaults[time_var] = self._get_bc_reference_datetime()  # Set datetime dtype as new row default
            self._convert_offset_to_isodate(old_index, time_var)
        elif old_index > -1:  # Convert between different offset units
            self._convert_time_offset_units(old_index, index)
        self.ui.tbl_xy_data.pasting = False
        self.on_data_changed(None, None)

    """
    Data I/O
    """
    def _restore_splitter_geometry(self):
        """Restore the position of the splitter."""
        settings = SettingsManager()
        splitter = settings.get_setting('xmstuflowfv', f'{self._dlg_name}.splitter')
        if not splitter:
            return
        splitter_sizes = [int(size) for size in splitter]
        self.splitter.setSizes(splitter_sizes)

    def _save_splitter_geometry(self):
        """Save the current position of the splitter."""
        settings = SettingsManager()
        settings.save_setting('xmstuflowfv', f'{self._dlg_name}.splitter', self.splitter.sizes())

    def _load_time_widgets(self):
        """Load the time widgets from the BC dataset."""
        # Install a validator on the edit field
        number_corrector = NumberCorrector(self)
        dbl_validator = QDoubleValidator(self)
        dbl_validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
        self.ui.edt_reference_time.setValidator(dbl_validator)
        self.ui.edt_reference_time.installEventFilter(number_corrector)
        # Load data into widgets
        populate_time_widgets(self, self.bc_data, self.time_format_qt)
        if self.curtain_dist is not None:  # If a curtain boundary, remove the ISOTIME option
            self.ui.cbx_time_units.removeItem(const.ISODATE_UNITS_IDX)
        # Initialize state
        self.on_tog_use_isodate(self.ui.tog_use_isodate.checkState())

    def _save_time_widgets(self):
        """Save the time widgets to the BC dataset."""
        if not self.x_is_time and self.curtain_dist is None:  # We don't plot time for curtains but still have widgets
            return
        extract_time_widget_data(self, self.bc_data)

    def _read_netcdf_for_selected_row(self, column_idx):
        """Read a y-column for a curtain boundary at a given timestep.

        Args:
            column_idx (int): 0-base index of the plot column. The minimum should be 1 (0 is x-column). Time is actually
                the first NetCDF variable for curtain BCs so it is not in the plot. So, should be the third specified
                NetCDF variable for the BC at a minimum.

        Returns:
            tuple(np.array, str): The y-column data, y-column series name
        """
        # Get the default NetCDF variable name for this y-column. We don't plot 'TIME' variable, so add 1 to skip it.
        series_label = bcd.ARC_BC_TYPES[self.series_name][column_idx + 1]
        # Get the specified NetCDF variable name. BcData stores the curtain NetCDF variable in DataArrays named
        # "variable1", ..., "variableN", so add 2 to skip 1st column ('TIME') and switch to 1-based.
        netcdf_variable_name = self.bc_data[f'variable{column_idx + 2}'][0].item()
        if netcdf_variable_name in self.curtain_dset:  # Variable may not be in file if user is defaulting the column
            timestep_idx = self.ui.spn_row_count.value() - 1  # Slice a row from the time dimension
            series_label += f' ({netcdf_variable_name})'
            return self.curtain_dset[netcdf_variable_name][timestep_idx].data, series_label
        return np.full(len(self.curtain_dist.index), np.nan), series_label  # Defaulting the entire column

    """
    Slots
    """
    def on_right_click(self, point):
        """Slot called when user right-clicks in the table.

        Args:
            point(QPoint): The point clicked.
        """
        # row = self.ui.table_view.logicalIndexAt(point)
        menu_list = [['', 'Copy', self.ui.tbl_xy_data.on_copy],
                     ['', 'Paste', self.ui.tbl_xy_data.on_paste]]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_xy_data.viewport().mapToGlobal(point))

    def on_index_column_click(self, point):
        """Called on right-click on the index column (vertical header).

        Args:
            point (QPoint): The point clicked
        """
        row = self.ui.tbl_xy_data.verticalHeader().logicalIndexAt(point)
        self.ui.tbl_xy_data.selectRow(row)
        menu_list = [['icons/row-insert.svg', 'Insert', self.on_insert_rows],
                     ['icons/row-delete.svg', 'Delete', self.on_delete_rows],
                     ['', 'Copy', self.ui.tbl_xy_data.on_copy],
                     ['', 'Paste', self.ui.tbl_xy_data.on_paste]]
        menu = widget_builder.setup_context_menu(self, menu_list)
        menu.popup(self.ui.tbl_xy_data.viewport().mapToGlobal(point))

    def on_btn_import(self):
        """Import a .xys file."""
        # Check if this is an arc BC type
        column_names = bcd.ARC_BC_TYPES.get(self.series_name)
        if not column_names:  # Check if this is a point BC type
            column_names = bcd.POINT_BC_TYPES.get(self.series_name)
        if not column_names:  # Check if this is a global BC type
            column_names = bcd.GLOBAL_BC_TYPES.get(self.series_name, ('', ''))
        default_columns = {
            column_name: label_text for column_name, label_text in
            zip(column_names, const.BC_VARIABLE_LABEL_TEXT.get(self.series_name, ('', '')))
        }

        # Show Open File dialog to user so they can pick a file
        dlg = ImportCsvDialog(columns=default_columns, parent=self)
        if dlg.exec():
            filename = dlg.selected_file()
            if not filename or not os.path.isfile(filename):
                return
            user_columns = dlg.column_names()
            try:
                self.import_file(filename, list(default_columns.keys()), user_columns)
            except Exception as r:
                message_with_ok(parent=self, message=str(r), app_name='SMS', icon='Critical',
                                win_icon=self.windowIcon())

    def on_btn_netcdf_info(self):
        """Display a non-modal dialog displaying the NetCDF dump info (curtains only)."""
        if self.netcdf_dump_dlg is not None:
            self.netcdf_dump_dlg.show()  # Bring existing modeless dialog to front
        else:
            self.netcdf_dump_dlg = NetCDFDumpViewerDialog(filename=self.netcdf_filename, parent=self)
            self.netcdf_dump_dlg.finished.connect(self.on_netcdf_dump_dlg_closed)
            self.netcdf_dump_dlg.show()

    def on_netcdf_dump_dlg_closed(self):
        """Clear the modeless dialog reference when it is closed."""
        self.netcdf_dump_dlg = None

    def on_tog_use_isodate(self, state):
        """Called when use ISODATE toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        use_isodate = state == Qt.Checked
        self.ui.edt_reference_time.setVisible(not use_isodate)
        self.ui.date_reference_time.setVisible(use_isodate)
        label_text = 'Reference date' if use_isodate else 'Reference time (h)'
        self.ui.lbl_reference_time.setText(label_text)

    def on_tog_view_isodate(self, state):
        """Called when view ISODATE toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        if state == Qt.Checked:
            # If enabling, copy DataFrame, convert to ISODATE using current units, and set as DataFrame of model after
            # storing the original DataFrame.
            self.raw_curtain_times_df = self.model.data_frame
            self.model.data_frame = self.raw_curtain_times_df.copy()
            self._convert_time_units(const.ISODATE_UNITS_IDX, self.previous_unit_index)
        else:
            # If disabling, restore original DataFrame
            self.ui.tbl_xy_data.pasting = True
            self.model.data_frame = self.raw_curtain_times_df
            self.raw_curtain_times_df = None
            self.ui.tbl_xy_data.pasting = False
            self.on_data_changed(None, None)
        self._update_table_view()
        QApplication.restoreOverrideCursor()

    def update_isodate_view(self, _=None):
        """Update the ISODATE view if needed when reference time changes (curtains only)."""
        if self.raw_curtain_times_df is None:
            return  # Not viewing as ISODATE, nothing to do
        self.model.data_frame = self.raw_curtain_times_df
        self.raw_curtain_times_df = None
        self.on_tog_view_isodate(Qt.Checked)

    def on_time_units_changed(self, index):
        """Called when the time units combobox is changed.

        Args:
            index (int): Combobox option index of the new BC type
        """
        if index < 0:
            return
        old_index = self.previous_unit_index
        self.previous_unit_index = index
        if old_index == index or old_index < 0 or self.ui.tbl_xy_data.pasting or self.in_setup:
            return  # Avoid infinite/excessive recursion
        elif self.curtain_dist is not None:  # If a curtain BC let user know we won't convert NetCDF data.
            if not self.warned_about_curtain_editing:  # But only bug them once about it
                msg = 'Warning: Editing curtain NetCDF files is not currently supported in SMS. Ensure selected time ' \
                      'units are consistent with data as no conversions will be made of the data in the NetCDF file.'
                message_with_ok(parent=self, message=msg, app_name='SMS', icon='Warning', win_icon=self.windowIcon())
                self.warned_about_curtain_editing = True
            self.update_isodate_view()
        else:  # We are good to go
            self._convert_time_units(index, old_index)
        self._update_table_view()
        QApplication.restoreOverrideCursor()

    def on_timestep_changed(self, selected, deselected):
        """Change the current timestep when the user changes the selected row.

        Args:
            selected (list): The newly selected index. Should only ever be size 0 or 1
            deselected (list): The previously selected index. Should only ever be size 0 or 1
        """
        if self.in_on_spn_rows or self.ui.tbl_xy_data.pasting or self.in_setup:
            return  # Avoid recursion or excessive, unnecessary calls
        if not selected or not selected[0].isValid():
            return  # Don't update plot if selection cleared
        selected_timestep_idx = selected[0].indexes()[0].row()
        previous_timestep_idx = -1
        if deselected:
            previous_timestep_idx = deselected[0].indexes()[0].row()
        if selected_timestep_idx != previous_timestep_idx:
            self.ui.spn_row_count.setValue(selected_timestep_idx + 1)

    def on_spn_rows(self):
        """Called when rows spin control is changed."""
        if self.in_on_spn_rows:  # Don't come in here recursively
            return
        self.in_on_spn_rows = True
        rows = self.ui.spn_row_count.value()
        if self.curtain_dist is not None:  # If a curtain boundary, spinbox changes timestep
            self._add_series()  # Update the timestep being plotted
            selection = self.ui.tbl_xy_data.selectionModel().selectedIndexes()
            if not selection or selection[0].row() != rows - 1:
                new_index = self.model.index(rows - 1, 0)  # Change the selected row in the XY series table
                self.ui.tbl_xy_data.selectionModel().setCurrentIndex(
                    new_index, QItemSelectionModel.SelectCurrent | QItemSelectionModel.Clear | QItemSelectionModel.Rows
                )
        else:  # Otherwise changing the spinbox value inserts/removes a row
            rows_old = self.model.rowCount()
            diff = abs(rows - rows_old)
            if rows > rows_old:
                self.model.insertRows(rows_old, diff)
            elif rows_old > rows:
                self.model.removeRows(rows_old - diff, diff)
        self.in_on_spn_rows = False

    def on_insert_rows(self):
        """Inserts rows into the table."""
        selected_list = self.ui.tbl_xy_data.selectedIndexes()
        if not selected_list:
            return
        minrow = selected_list[0].row()
        self.model.insertRows(minrow, 1)
        self.ui.spn_row_count.setValue(self.model.rowCount())

    def on_delete_rows(self):
        """Deletes rows from the table."""
        selected_list = self.ui.tbl_xy_data.selectedIndexes()
        if not selected_list:
            return
        minrow = selected_list[0].row()
        self.model.removeRows(minrow, 1)
        self.ui.spn_row_count.setValue(self.model.rowCount())

    def on_data_changed(self, top_left_index, bottom_right_index):
        """Called when the data in the table view has changed.

        Args:
            top_left_index (QModelIndex): Top left index.
            bottom_right_index (QModelIndex): Bottom right index.
        """
        if self.in_setup or self.ui.tbl_xy_data.pasting:
            return  # Avoid recursion
        if self.curtain_dist is not None:
            return  # Curve is read-only if curtain BC
        self._setup_plot_model()
        self._add_series()
        row_count = self.model.rowCount()
        self.ui.spn_row_count.setValue(row_count)

    """
    Qt overloads
    """
    def sizeHint(self):  # noqa: N802
        """Overridden method to help the dialog have a good minimum size.

        Returns:
            (QSize): Size to use for the initial dialog size.
        """
        return QSize(800, 500)

    def showEvent(self, event):  # noqa: N802
        """Restore last position and geometry when showing dialog."""
        super().showEvent(event)
        self._restore_splitter_geometry()

    def accept(self):
        """Override default accept slot to update persistent dataset."""
        self._save_time_widgets()
        self._save_splitter_geometry()
        super().accept()

    def reject(self):
        """Save position and geometry when closing dialog."""
        self._save_splitter_geometry()
        super().reject()

    """
    Public methods
    """
    def import_file(self, filename, default_columns, user_columns):
        """Populate the series editor and plot from a CSV file.

        Args:
            filename (str): CSV filename
            default_columns (list[str]): Default column names in order as defined by TUFLOWFV
            user_columns (list[str]): User defined column names in same order as `default_columns`
        """
        QApplication.setOverrideCursor(Qt.WaitCursor)  # This can take a second with a big curve
        csv_reader = BcCsvReader(filename=filename, default_columns=default_columns, user_columns=user_columns,
                                 x_is_time=self.x_is_time)
        df = csv_reader.read()
        # If X column is not a float, assume it's a date/time. Disable conversion of units by saying that we
        # are in paste mode.
        self.ui.tbl_xy_data.pasting = True
        if not is_float(df[df.columns[0]][1]):
            if not self.x_is_time:  # If X-column is not defined as time, raise an error
                raise RuntimeError('Unexpected datetime field found in X column.')
            self.model.defaults['Time'] = self._get_bc_reference_datetime()
            self.ui.cbx_time_units.setCurrentIndex(const.ISODATE_UNITS_IDX)
        elif self.x_is_time:  # Float offset as time
            self.model.defaults['Time'] = 0.0
            # Change time units to default hours if set to isodate
            if self.ui.cbx_time_units.currentIndex() == const.ISODATE_UNITS_IDX:
                self.ui.cbx_time_units.setCurrentIndex(const.HOUR_UNITS_IDX)
        self.ui.tbl_xy_data.pasting = False
        # Setup the table and the plot
        self._setup_model(df)
        self._setup_table()
        self._setup_rows_spin_control()
        self._add_series()
        row_count = self.model.rowCount()
        self.ui.spn_row_count.setValue(row_count)
        QApplication.restoreOverrideCursor()

    def get_time_units_label(self):
        """Get a label for the time column/axis that is appropriate for the current time units.

        Returns:
            str: The time label
        """
        label = 'Time'
        units_index = self.ui.cbx_time_units.currentIndex()
        if units_index == const.DAY_UNITS_IDX:
            label = f'{label} (d)'
        elif units_index == const.HOUR_UNITS_IDX:
            label = f'{label} (h)'
        elif units_index == const.MIN_UNITS_IDX:
            label = f'{label} (m)'
        elif units_index == const.SEC_UNITS_IDX:
            label = f'{label} (s)'
        elif units_index == const.ISODATE_UNITS_IDX:
            label = 'Date'
        return label
