"""Dialog for defining gridded BC NetCDF variable names."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtWidgets import QComboBox, QGridLayout, QGroupBox, QLabel, QSizePolicy, QSpacerItem, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.message_box import message_with_ok
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.tuflowfv.gui import assign_bc_consts as const
from xms.tuflowfv.gui import gui_util

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class GridVariablesDialog(XmsDlg):
    """Dialog for defining gridded BC NetCDF variable names."""

    def __init__(self, bc_data, bc_type, grid_bc_file, parent):
        """Constructor.

        Args:
            bc_data (xarray.Dataset): Dataset containing the boundary conditions parameters
            bc_type (str): The gridded BC type. Should match one of the const.BC_VARIABLE_LABEL_TEXT keys. Need to pass
                this in because the bc_data won't be updated if the user has changed the BC type combobox option since
                opening the parent dialog.
            grid_bc_file (str): Absolute path to the gridded BC dataset file
            parent (QWidget): Parent dialog
        """
        super().__init__(parent, 'xms.tuflowfv.gui.grid_variables_dialog')
        self._data = bc_data
        self._bc_type = bc_type
        self._comboboxes = []
        self._setup_ui(grid_bc_file)

    def _setup_ui(self, grid_bc_file):
        """Intialize the dialog widgets and values.

        Args:
            grid_bc_file (str): Absolute path to the gridded BC dataset file
        """
        self._add_widgets(grid_bc_file)
        self._load_data()
        self.setWindowTitle('Select NetCDF Gridded BC Variable Names')

    def _add_widgets(self, grid_bc_file):
        """Add the widgets to the layout.

        Args:
            grid_bc_file (str): Absolute path to the gridded BC dataset file
        """
        # Create labels and comboboxes for each potential NetCDF variable
        label_texts = const.BC_VARIABLE_LABEL_TEXT[self._bc_type]
        hide_idx = const.BC_VARIABLE_HIDE_IDX.get(self._bc_type, -1)  # Temporary kludge to hide elevation for WL_CURT
        grid_layout = QGridLayout()
        for i, label_text in enumerate(label_texts):
            label = QLabel(label_text)
            grid_layout.addWidget(label, i, 0)
            self._comboboxes.append(QComboBox())
            grid_layout.addWidget(self._comboboxes[-1], i, 1)
            if i == hide_idx:
                self._comboboxes[-1].hide()
                label.hide()
        # For wave BCs, everything after the 4th variable is optional.
        gui_util.populate_gridded_bc_netcdf_variables(grid_bc_file, self._comboboxes)
        grp_variables = QGroupBox('Variable names')
        grp_variables.setLayout(grid_layout)

        # Create the button box
        button_box = gui_util.build_ok_cancel_buttons(self)

        # Add the widgets to the dialog
        main_layout = QVBoxLayout(self)
        main_layout.addWidget(grp_variables)
        main_layout.addSpacerItem(QSpacerItem(20, 40, QSizePolicy.Minimum, QSizePolicy.Expanding))
        main_layout.addWidget(button_box)
        self.setLayout(main_layout)

    """
    File I/O
    """
    def _load_data(self):
        """Populate previously selected variable names."""
        for i, combobox in enumerate(self._comboboxes):
            var_name = f'variable{i + 1}'  # xarray Variables named 'variable1',...,'variableN'
            gui_util.set_combobox_from_data(combobox, self._data[var_name][0].item())

    def _save_data(self):
        """Save data from the 'Output' tab to in-memory Datasets."""
        for i, combobox in enumerate(self._comboboxes):
            var_name = f'variable{i + 1}'  # xarray Variables named 'variable1',...,'variableN'
            self._data[var_name][0] = combobox.itemData(combobox.currentIndex())

    def exec(self):
        """Ensure we were able to load NetCDF variables into comboboxes before showing dialog.

        Returns:
            bool: True if dialog was shown and accepted, False if error reading NetCDF variable names or
        """
        if not self._comboboxes or self._comboboxes[0].count() == 0:
            msg = 'Unable to read NetCDF variables from gridded dataset file. Ensure selected file is a valid ' \
                  'NetCDF file'
            message_with_ok(parent=self.parent(), message=msg, app_name='SMS', win_icon=self.parent().windowIcon())
            self.reject()
            return False
        return super().exec_()

    def accept(self):
        """Save dialog data on accepted."""
        self._save_data()
        super().accept()
