"""Dialog for viewing NetCDF dump info."""
# 1. Standard python modules
from io import StringIO

# 2. Third party modules
from PySide2.QtGui import QTextCursor, QTextOption
from PySide2.QtWidgets import QGroupBox, QPlainTextEdit, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.tuflowfv.file_io.ncdump import ncdump
from xms.tuflowfv.gui import gui_util

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class NetCDFDumpViewerDialog(XmsDlg):
    """Dialog for viewing NetCDF dump info."""

    def __init__(self, filename, parent):
        """Constructor.

        Args:
            filename (str): Absolute path to the NetCDF file
            parent (QWidget): Parent dialog
        """
        super().__init__(parent, 'xms.tuflowfv.gui.netcdf_dump_viewer_dialog')
        self._text_viewer = None
        self._setup_ui(filename)

    def _setup_ui(self, filename):
        """Intialize the dialog widgets and values.

        Args:
            filename (str): Absolute path to the NetCDF file
        """
        self._add_widgets()
        self._load_data(filename)
        self.setWindowTitle('NetCDF Info')

    def _add_widgets(self):
        """Add the widgets to the layout."""
        # Create labels and comboboxes for each potential NetCDF variable
        self._text_viewer = QPlainTextEdit()
        self._text_viewer.setWordWrapMode(QTextOption.NoWrap)
        self._text_viewer.setReadOnly(True)
        text_viewer_layout = QVBoxLayout()
        text_viewer_layout.addWidget(self._text_viewer)

        grp_text_viewer = QGroupBox('NetCDF dump output')
        grp_text_viewer.setLayout(text_viewer_layout)

        # Create the button box
        button_box = gui_util.build_ok_cancel_buttons(self, cancel=False)

        # Add the widgets to the dialog
        main_layout = QVBoxLayout(self)
        main_layout.addWidget(grp_text_viewer)
        main_layout.addWidget(button_box)
        self.setLayout(main_layout)

    def _load_data(self, filename):
        """Populate previously selected variable names.

        Args:
            filename (str): Absolute path to the NetCDF file
        """
        ss = StringIO()
        ncdump(filename, ss)
        ss.seek(0)
        self._text_viewer.document().setPlainText(ss.read())
        # Move to the beginning of the text
        cursor = self._text_viewer.textCursor()
        cursor.movePosition(QTextCursor.Start, QTextCursor.MoveAnchor, 1)
        self._text_viewer.setTextCursor(cursor)
