"""Dialog to select a table of tree items."""
# 1. Standard python modules

# 2. Third party modules
from PySide2.QtWidgets import QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.tuflowfv.gui import gui_util
from xms.tuflowfv.gui.tree_item_selector_table import TreeItemSelectorTableWidget


__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"


class TreeItemSelectorTableDialog(XmsDlg):
    """Dialog to select a table of tree items."""

    def __init__(self, title, tree_node, tree_type, data_frame, max_rows, parent, sim_uuid=None, layer_type=''):
        """Initializes the class, sets up the ui.

        Args:
            title (str): Window title of the dialog
            tree_node (TreeNode): The root project explorer tree node
            tree_type (str): The target tree item type's XMS enum string, e.g. TI_GENERIC_POINT
            data_frame (pd.DataFrame): The model data
            max_rows (Union[int, None]): The maximum number of selected tree items
            parent (QDialog): The parent dialog
            sim_uuid (Optional[str]): UUID of the simulation selecting data. Only need to provide if
                tree_type==TI_DYN_SIM (selecting a linked child simulation).
            layer_type (Optional[str]): The type of the layer to select, e.g. 'Z', 'Output'
        """
        super().__init__(parent, 'xms.tuflowfv.gui.tree_item_selector_table_dialog')
        self.selector_table = None
        self._layer_type = layer_type
        self._setup_ui(title=title, tree_node=tree_node, tree_type=tree_type, data_frame=data_frame, max_rows=max_rows,
                       sim_uuid=sim_uuid)

    def _setup_ui(self, title, tree_node, tree_type, data_frame, max_rows, sim_uuid):
        """Add the table to the dialog.

        Args:
            title (str): Window title of the dialog
            tree_node (TreeNode): The root project explorer tree node
            tree_type (str): The target tree item type's XMS enum string, e.g. TI_GENERIC_POINT
            data_frame (pd.DataFrame): The model data
            max_rows (Union[int, None]): The maximum number of selected tree items
            sim_uuid (Union[str, None]): UUID of the simulation selecting data. Only need to provide if
                tree_type==TI_DYN_SIM (selecting a linked child simulation).
        """
        self.setWindowTitle(title)
        # Create the table
        # def __init__(self, tree_node, tree_type, data_frame, max_rows, parent, sim_uuid=None):
        self.selector_table = TreeItemSelectorTableWidget(tree_node=tree_node, tree_type=tree_type,
                                                          data_frame=data_frame, max_rows=max_rows, parent=self,
                                                          sim_uuid=sim_uuid, layer_type=self._layer_type)
        # Create the button box
        button_box = gui_util.build_ok_cancel_buttons(self)
        # Add the widgets to the dialog.
        layout = QVBoxLayout(self)
        layout.addWidget(self.selector_table)
        layout.addWidget(button_box)
        self.setLayout(layout)
