"""Defines the boundary conditions."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Curve, GenericModel
from xms.tool_core.table_definition import FloatColumnType, TableDefinition

# 4. Local modules


def create() -> GenericModel:
    """Creates and returns the generic model.

    Returns:
        (GenericModel): The generic model.
    """
    builder = BcGenericModelBuilder()
    return builder.build()


class BcGenericModelBuilder:
    """Creates the BC generic model."""
    def __init__(self):
        """Initializes the class."""
        self._desc = {}  # descriptions
        self._gm = None  # The generic model
        self._section = None  # The current section
        self._section_name = ''  # The current section name
        self._gp = None  # The current group
        self._group_name = ''  # The current group name

        self._init_descriptions()

    def _init_descriptions(self):
        """Initializes the descriptions."""
        self._desc = {
            'name': 'Descriptive name for the well.',
            'well_screen_opts': 'Choose to use one well screen, or multiple.',
            'screen_top': 'Elevation of the top of the screened interval [L].',
            'screen_bottom': 'Elevation of the bottom of the screened interval [L].',
            'well_screens': 'Screened intervals',
            'head': 'Head (IRWTYPF3) [L].',
            'profile_type': 'Profile type (IDBH3).',
            'ref_elev_option': 'Reference elevation will be the top of the grid or a specified value.',
            'ref_elev': 'Reference elevation used to indicate whether overflow occurs (Z_RWF3). [L]',
            'sat_k': 'Saturated hydraulic conductivity (AK_RWF3). [L/T]',
            'well_diameter': 'Well diameter (D_RWF3). [L]',
            'well_screen_resistance_coeff': 'Well screen resistance coefficient (R_RWF3).',
        }

    def build(self) -> GenericModel:
        """Creates and returns the GenericModel."""
        self._gm = GenericModel(
            exclusive_point_conditions=True, exclusive_arc_conditions=True, exclusive_polygon_conditions=True
        )
        self._create_point_parameters()
        return self._gm

    def _create_point_parameters(self):
        """Creates the point parameters."""
        self._section = self._gm.point_parameters
        self._section_name = 'point_parameters'
        self._create_relief_wells()

    def _create_relief_wells(self):
        """Creates the relief wells."""
        self._group_name = 'Relief well'
        self._gp = self._section.add_group(group_name=self._group_name, label='Relief well')
        self._add_name()
        self._add_well_screens()
        self._add_head()
        self._add_profile_type()
        self._add_reference_elevation()
        self._add_saturated_k()
        self._add_well_diameter()
        self._add_resistance_coefficient()

    def _add_name(self):
        """Adds a name."""
        self._gp.add_text(name='name', label='Name', default='Well', description=self._desc['name'], required=False)

    def _add_well_screens(self):
        """Adds the well screens."""
        well_screen_opts = ['One well screen', 'Multiple well screens']
        self._gp.add_option(
            name='well_screen_opts',
            label='Well screen options',
            default=well_screen_opts[0],
            options=well_screen_opts,
            description=self._desc['well_screen_opts']
        )
        parent = [self._section_name, self._group_name, 'well_screen_opts']
        p = self._gp.add_float(name='screen_top', label='Screen top', default=0.0, description=self._desc['screen_top'])
        p.add_dependency(parent=parent, flags={well_screen_opts[0]: True, well_screen_opts[1]: False})
        p = self._gp.add_float(
            name='screen_bottom', label='Screen bottom', default=0.0, description=self._desc['screen_bottom']
        )
        p.add_dependency(parent=parent, flags={well_screen_opts[0]: True, well_screen_opts[1]: False})
        table_def = TableDefinition(
            [
                FloatColumnType(
                    header='Screen top', tool_tip='Top of the screened interval [L]', default=0.0, low=-1e20
                ),
                FloatColumnType(
                    header='Screen bottom', tool_tip='Bottom of the screened interval [L]', default=0.0, low=-1e20
                ),
            ]
        )
        p = self._gp.add_table(
            name='well_screens',
            label='Well screens',
            default=[[0.0, 0.0]],
            table_definition=table_def,
            description=self._desc['well_screens']
        )
        p.add_dependency(parent=parent, flags={well_screen_opts[0]: False, well_screen_opts[1]: True})

    def _add_head(self):
        """Adds the head."""
        self._gp.add_checkbox_curve(
            name='head',
            label='Head',
            checkbox_label='Transient',
            axis_titles=['Time', 'Head'],
            modes=(Curve.CONSTANT, Curve.CURVE),
            default_check_state=True,
            default_constant=0.0,
            description=self._desc['head'],
            checkbox_description='Check to define time varying head.'
        )

    def _add_profile_type(self):
        """Adds the profile type."""
        profile_types = ['Pressure head (0)', 'Total head (1)']
        self._gp.add_option(
            name='profile_type',
            label='Profile type',
            default=profile_types[1],
            options=profile_types,
            description=self._desc['profile_type']
        )

    def _add_reference_elevation(self):
        """Adds the reference elevation."""
        ref_elev_opts = ['Specify value', 'Use grid top']
        self._gp.add_option(
            name='ref_elev_option',
            label='Reference elevation option',
            default=ref_elev_opts[0],
            options=ref_elev_opts,
            description=self._desc['ref_elev_option']
        )
        parent = [self._section_name, self._group_name, 'ref_elev_option']
        p = self._gp.add_float(
            name='ref_elev', label='Reference elevation', default=0.0, description=self._desc['ref_elev']
        )
        p.add_dependency(parent=parent, flags={ref_elev_opts[0]: True, ref_elev_opts[1]: False})

    def _add_saturated_k(self):
        """Adds saturated k."""
        self._gp.add_float(
            name='sat_k', label='Saturated hydraulic conductivity', default=0.0, description=self._desc['sat_k']
        )

    def _add_well_diameter(self):
        """Adds well diameter."""
        self._gp.add_float(
            name='well_diameter', label='Well diameter', default=0.0, description=self._desc['well_diameter']
        )

    def _add_resistance_coefficient(self):
        """Adds the resistance coefficient."""
        self._gp.add_float(
            name='well_screen_resistance_coeff',
            label='Well screen resistance coefficient',
            default=0.0,
            description=self._desc['well_screen_resistance_coeff']
        )
