"""Class to read a WaveWatch3 namelists namelist file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
import shlex
import uuid

# 2. Third party modules
import numpy as np
import xarray as xr

# 3. Aquaveo modules
from xms.api.dmi import Query

# 4. Local modules
from xms.wavewatch3.data.model import get_model
from xms.wavewatch3.file_io.io_util import READ_BUFFER_SIZE


def _get_true_false_as_int(value):
    """Gets the various formats of true/false in as an integer.

    value (:obj:`str`):  The string from the file.

    Returns:
        (:obj:`int`):  1 if some form of true, 0 if false
    """
    if value.upper() == '.FALSE.' or value.upper() == 'F' or value.upper() == "'F'" or value == '0':
        return 0
    elif value.upper() == '.TRUE.' or value.upper() == 'T' or value.upper() == "'T'" or value == '1':
        return 1
    else:
        raise ValueError(f"Unable to determine value of {value}.")


class NamelistsNmlReader:
    """Class to read a WaveWatch3 namelists nml file."""
    def __init__(self, filename, sim_data):
        """Constructor.

        Args:
            filename (:obj:`str`): Path to the nml file. If not provided (not testing or control file read),
                will retrieve from Query.
            sim_data(:obj:`xms.wavewatch3.data.SimData`):  The simulation data to edit.
        """
        self._filename = filename
        self._query = None
        self._sim_data = sim_data
        self._sim_comp_uuid = str(uuid.uuid4())
        self._setup_query()
        self._lines = []
        self._current_line = 0
        self._logger = logging.getLogger('xms.wavewatch3')
        self._global_values = get_model().global_parameters
        self._global_values.restore_values(self._sim_data.global_values)

    def _setup_query(self):
        """Setup the xmsapi Query for sending data to SMS and get the import filename."""
        if not self._filename:  # pragma: no cover - slow to setup Query for the filename
            self._query = Query()
            self._filename = self._query.read_file

    def _parse_next_line(self, shell=False):
        """Parse the next line of text from the file.

        Skips empty and comment lines.

        Args:
            shell (:obj:`bool`): If True will parse line using shlex. Slower but convenient for quoted tokens.

        Returns:
            (:obj:`list[str]`: The next line of text, split on whitespace
        """
        line = None
        comment_prefixes = ('!', '$')
        while not line or line.startswith(comment_prefixes):  # blank lines and control file identifier
            if self._current_line >= len(self._lines):
                # raise RuntimeError('Unexpected end of file.')
                return None
            line = self._lines[self._current_line].strip()
            self._current_line += 1
        if shell:
            return shlex.split(line, posix=False)
        return line.split()

    def _get_namelist_cards_and_values(self, firstline_data):
        """Reads the entire namelist until the closing / is found.  Stores cards and values.

        Handles cases where the data is on multiple lines, or on a single line.

        Args:
            firstline_data (:obj:`list[str]`):  List of data parsed on the opening line.

        Returns:
            (:obj:`dict`):  Dictionary with the keys being the cards read, and values of corresponding data.
        """
        namelist_list = firstline_data
        if '/' not in namelist_list:
            # We haven't found the end of the namelist yet.  Read more lines as necessary.
            end_not_found = True
            while end_not_found:
                # Grab the next line of data
                data = self._parse_next_line()
                # Extend the list of all data by the current line read
                namelist_list.extend(data)
                # Check if we've got the end of the namelist yet
                if '/' in data:
                    end_not_found = False

        # Now, we have the entire namelist as a list of values.  Parse into cards/commands and values
        cards = []
        values = []
        for i in range(len(namelist_list)):
            # Get the list elements on either side of each =
            if namelist_list[i] == '=':
                if 0 < i < len(namelist_list):
                    cards.append(namelist_list[i - 1].rstrip(','))
                    values.append(namelist_list[i + 1].rstrip(','))

        # Return the cards and values found throughout the namelist read
        return cards, values

    def _get_namelist_table_values(self, firstline_data):
        """Reads the entire namelist until the closing / is found.  Stores cards and values, for a table namelist.

        Handles cases where the data is on multiple lines, or on a single line.
        (Use only for ANL2 and ANL3 namelists)

        Args:
            firstline_data (:obj:`list[str]`):  List of data parsed on the opening line.

        Returns:
            (:obj:`list`):  List of values read.
        """
        namelist_list = firstline_data
        if '/' not in namelist_list:
            # We haven't found the end of the namelist yet.  Read more lines as necessary.
            end_not_found = True
            while end_not_found:
                # Grab the next line of data
                data = self._parse_next_line()
                # Extend the list of all data by the current line read
                namelist_list.extend(data)
                # Check if we've got the end of the namelist yet
                if '/' in data:
                    end_not_found = False

        # Now, we have the entire namelist as a list of values.  Parse into cards/commands and values
        values = []
        add_values = False
        for i in range(len(namelist_list)):
            # Get the list elements on either side of each =
            if namelist_list[i] == '=':
                add_values = True
            elif add_values and namelist_list[i] != '/' and namelist_list[i] != ',':
                # Add all values in the order read, as long as they aren't / or a comma, and remove trailing commas
                values.append(namelist_list[i].rstrip(','))

        # Return the values found throughout the namelist read
        return values

    def _read_namelists_nml_file(self):
        reading = True
        while reading:
            data = self._parse_next_line()
            if data and data != ['END', 'OF', 'NAMELISTS']:
                if '&FLX3' in data[0].strip():
                    self._read_stresses_flx3_namelist(data)
                elif '&FLX4' in data[0].strip():
                    self._read_stresses_flx4_namelist(data)
                elif '&SLN1' in data[0].strip():
                    self._read_linear_input_sln1_namelist(data)
                elif '&SIN1' in data[0].strip():
                    self._read_exponential_input_sin1_namelist(data)
                elif '&SIN2' in data[0].strip():
                    self._read_exponential_input_sin2_namelist(data)
                elif '&SIN3' in data[0].strip():
                    self._read_exponential_input_sin3_namelist(data)
                elif '&SIN6' in data[0].strip():
                    self._read_exponential_input_sin6_namelist(data)
                elif '&SNL1' in data[0].strip():
                    self._read_nonlinear_interactions_snl1_namelist(data)
                elif '&SNL2' in data[0].strip():
                    self._read_nonlinear_interactions_snl2_namelist(data)
                elif '&ANL2' in data[0].strip():
                    self._read_nonlinear_interactions_anl2_namelist(data)
                elif '&SNL3' in data[0].strip():
                    self._read_nonlinear_interactions_snl3_namelist(data)
                elif '&ANL3' in data[0].strip():
                    self._read_nonlinear_interactions_anl3_namelist(data)
                elif '&SNL4' in data[0].strip():
                    self._read_nonlinear_interactions_snl4_namelist(data)
                elif '&SNLS' in data[0].strip():
                    self._read_nonlinear_filter_snls_namelist(data)
                elif '&SDS1' in data[0].strip():
                    self._read_whitecapping_sds1_namelist(data)
                elif '&SDS2' in data[0].strip():
                    self._read_whitecapping_sds2_namelist(data)
                elif '&SDS3' in data[0].strip():
                    self._read_whitecapping_sds3_namelist(data)
                elif '&SDS6' in data[0].strip():
                    self._read_whitecapping_sds6_namelist(data)
                elif '&SWL6' in data[0].strip():
                    self._read_whitecapping_swl6_namelist(data)
                elif '&SBT1' in data[0].strip():
                    self._read_bottom_friction_sbt1_namelist(data)
                elif '&SDB1' in data[0].strip():
                    self._read_surf_breaking_sdb1_namelist(data)
                elif '&SIC2' in data[0].strip():
                    self._read_dissipation_in_ice_sic2_namelist(data)
                elif '&SIS2' in data[0].strip():
                    self._read_scattering_in_ice_sis2_namelist(data)
                elif '&SIC4' in data[0].strip():
                    self._read_dissipation_by_ice_sic4_namelist(data)
                elif '&STR1' in data[0].strip():
                    self._read_triad_nonlinear_str1_namelist(data)
                elif '&REF1' in data[0].strip():
                    self._read_shoreline_reflections_ref1_namelist(data)
                elif '&SIG1' in data[0].strip():
                    self._read_free_ig_sig1_namelist(data)
                elif '&PRO1' in data[0].strip():
                    self._read_propagation_pro1_namelist(data)
                elif '&PRO2' in data[0].strip():
                    self._read_propagation_pro2_namelist(data)
                elif '&PRO3' in data[0].strip():
                    self._read_propagation_pro3_namelist(data)
                elif '&UNST' in data[0].strip():
                    self._read_unstructured_grids_unst_namelist(data)
                elif '&PSMC' in data[0].strip():
                    self._read_unstructured_grids_psmc_namelist(data)
                elif '&OUTS' in data[0].strip():
                    self._read_output_outs_namelist(data)
                elif '&MISC' in data[0].strip():
                    self._read_miscellaneous_misc_namelist(data)
                elif '&FLD1' in data[0].strip():
                    self._read_sea_state_fld1_namelist(data)
                elif '&FLD2' in data[0].strip():
                    self._read_sea_state_fld2_namelist(data)
                else:
                    raise ValueError(f'Unrecognized namelist {data}')
            else:
                reading = False

    def _read_stresses_flx3_namelist(self, data):
        """Read the stresses FLX3 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading FLX3 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CDMAX' in cards:
            consts.parameter('FLX3CDMAX').value = float(values[cards.index('CDMAX')])
        if 'CTYPE' in cards:
            consts.parameter('FLX3CTYPE').value = int(float(values[cards.index('CTYPE')]))
        consts.parameter('stresses').value = 0

    def _read_stresses_flx4_namelist(self, data):
        """Read the stresses FLX4 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading FLX4 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CDFAC' in cards:
            consts.parameter('FLX4CDFAC').value = float(values[cards.index('CDFAC')])
        consts.parameter('stresses').value = 1

    def _read_linear_input_sln1_namelist(self, data):
        """Read the stresses SLN1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SLN1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CLIN' in cards:
            consts.parameter('SLN1CLIN').value = float(values[cards.index('CLIN')])
        if 'RFPM' in cards:
            consts.parameter('SLN1RFPM').value = float(values[cards.index('RFPM')])
        if 'RFHF' in cards:
            consts.parameter('SLN1RFPM').value = float(values[cards.index('RFHF')])

    def _read_exponential_input_sin1_namelist(self, data):
        """Read the exponential input SIN1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SIN1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CINP' in cards:
            consts.parameter('SIN1CINP').value = float(values[cards.index('CINP')])
        consts.parameter('exponential_input').value = 0

    def _read_exponential_input_sin2_namelist(self, data):
        """Read the exponential input SIN2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SIN2 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'ZWND' in cards:
            consts.parameter('ZWND').value = float(values[cards.index('ZWND')])
        if 'SWELLF' in cards:
            consts.parameter('SIN2SWELLF').value = float(values[cards.index('SWELLF')])
        if 'STABSH' in cards:
            consts.parameter('SIN2STABSH').value = float(values[cards.index('STABSH')])
        if 'STABOF' in cards:
            consts.parameter('SIN2STABOF').value = float(values[cards.index('STABOF')])
        if 'CNEG' in cards:
            consts.parameter('SIN2CNEG').value = float(values[cards.index('CNEG')])
        if 'CPOS' in cards:
            consts.parameter('SIN2CPOS').value = float(values[cards.index('CPOS')])
        if 'FNEG' in cards:
            consts.parameter('SIN2FNEG').value = float(values[cards.index('FNEG')])
        consts.parameter('exponential_input').value = 1

    def _read_exponential_input_sin3_namelist(self, data):
        """Read the exponential input SIN3 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SIN3 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'ZWND' in cards:
            consts.parameter('ZWND').value = float(values[cards.index('ZWND')])
        if 'ALPHA0' in cards:
            consts.parameter('ALPHA0').value = float(values[cards.index('ALPHA0')])
        if 'Z0MAX' in cards:
            consts.parameter('Z0MAX').value = float(values[cards.index('Z0MAX')])
        if 'BETAMAX' in cards:
            consts.parameter('BETAMAX').value = float(values[cards.index('BETAMAX')])
        if 'SINTHP' in cards:
            consts.parameter('SINTHP').value = float(values[cards.index('SINTHP')])
        if 'ZALP' in cards:
            consts.parameter('ZALP').value = float(values[cards.index('ZALP')])
        if 'TAUWSHELTER' in cards:
            consts.parameter('TAUWSHELTER').value = float(values[cards.index('TAUWSHELTER')])
        if 'SWELLFPAR' in cards:
            consts.parameter('SWELLFPAR').value = int(float(values[cards.index('SWELLFPAR')]))
        if 'SWELLF' in cards:
            consts.parameter('SWELLF').value = float(values[cards.index('SWELLF')])
        if 'SWELLF2' in cards:
            consts.parameter('SWELLF2').value = float(values[cards.index('SWELLF2')])
        if 'SWELLF3' in cards:
            consts.parameter('SWELLF3').value = float(values[cards.index('SWELLF3')])
        if 'SWELLF4' in cards:
            consts.parameter('SWELLF4').value = float(values[cards.index('SWELLF4')])
        if 'SWELLF5' in cards:
            consts.parameter('SWELLF5').value = float(values[cards.index('SWELLF5')])
        if 'Z0RAT' in cards:
            consts.parameter('Z0RAT').value = float(values[cards.index('Z0RAT')])
        consts.parameter('exponential_input').value = 2

    def _read_exponential_input_sin6_namelist(self, data):
        """Read the exponential input SIN6 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SIN6 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'SINA0' in cards:
            consts.parameter('SIN6SINA0').value = float(values[cards.index('SINA0')])
        if 'SINWS' in cards:
            consts.parameter('SIN6SINU10').value = float(values[cards.index('SINWS')])
        consts.parameter('exponential_input').value = 3

    def _read_nonlinear_interactions_snl1_namelist(self, data):
        """Read the nonlinear interactions SNL1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SNL1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'LAMBDA' in cards:
            consts.parameter('SNL1LAMBDA').value = float(values[cards.index('LAMBDA')])
        if 'NLPROP' in cards:
            consts.parameter('SNL1NLPROP').value = float(values[cards.index('NLPROP')])
        if 'KDCONV' in cards:
            consts.parameter('SNL1KDCONV').value = float(values[cards.index('KDCONV')])
        if 'KDMIN' in cards:
            consts.parameter('SNL1KDMIN').value = float(values[cards.index('KDMIN')])
        if 'SNLCS1' in cards:
            consts.parameter('SNL1SNLCS1').value = float(values[cards.index('SNLCS1')])
        if 'SNLCS2' in cards:
            consts.parameter('SNL1SNLCS2').value = float(values[cards.index('SNLCS2')])
        if 'SNLCS3' in cards:
            consts.parameter('SNL1SNLCS3').value = float(values[cards.index('SNLCS2')])
        consts.parameter('non_linear_interactions').value = 0

    def _read_nonlinear_interactions_snl2_namelist(self, data):
        """Read the nonlinear interactions SNL2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SNL2 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'IQTYPE' in cards:
            consts.parameter('SNL2IQTYPE').value = int(float(values[cards.index('IQTYPE')]))
        if 'TAILNL' in cards:
            consts.parameter('SNL2TAILNL').value = float(values[cards.index('TAILNL')])
        consts.parameter('non_linear_interactions').value = 1

    def _read_nonlinear_interactions_snl3_namelist(self, data):
        """Read the nonlinear interactions SNL3 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SNL3 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'MSC' in cards:
            consts.parameter('SNL3MSC').value = float(values[cards.index('MSC')])
        if 'NSC' in cards:
            consts.parameter('SNL3NSC').value = float(values[cards.index('NSC')])
        if 'KDFD' in cards:
            consts.parameter('SNL3KDFD').value = float(values[cards.index('KDFD')])
        if 'KDFS' in cards:
            consts.parameter('SNL3KDFS').value = float(values[cards.index('KDFS')])
        consts.parameter('non_linear_interactions').value = 2

    def _read_nonlinear_interactions_snl4_namelist(self, data):
        """Read the nonlinear interactions SNL4 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SNL4 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'INDTSA' in cards:
            consts.parameter('SNL4INDTSA').value = int(float(values[cards.index('INDTSA')]))
        if 'ALTLP' in cards:
            consts.parameter('SNL4ALTLP').value = int(float(values[cards.index('ALTLP')]))
        consts.parameter('non_linear_interactions').value = 3

    def _read_nonlinear_interactions_anl2_namelist(self, data):
        """Read the nonlinear interactions ANL2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        self._logger.info('Reading ANL2 namelist...')
        values = self._get_namelist_table_values(data)
        anl2_depths = []
        for value in values:
            anl2_depths.append(float(value))

        # Turn in to an xarray Dataset and store
        anl2depths = {
            'Depth': xr.DataArray(data=np.array([anl2_depths], dtype=np.float64)),
        }
        self._sim_data.anl2depths = xr.Dataset(data_vars=anl2depths)

    def _read_nonlinear_interactions_anl3_namelist(self, data):
        """Read the nonlinear interactions ANL3 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        self._logger.info('Reading ANL3 namelist...')
        values = self._get_namelist_table_values(data)
        anl3_lambda = []
        anl3_mu = []
        anl3_dt12 = []
        anl3_cdeep = []
        anl3_cshal = []
        for i in range(0, len(values), 5):
            anl3_lambda.append(float(values[i]))
            anl3_mu.append(float(values[i + 1]))
            anl3_dt12.append(float(values[i + 2]))
            anl3_cdeep.append(float(values[i + 3]))
            anl3_cshal.append(float(values[i + 4]))

        # Turn in to an xarray Dataset and store
        anl3qparms = {
            'LAMBDA': xr.DataArray(data=np.array([anl3_lambda], dtype=np.float64)),
            'MU': xr.DataArray(data=np.array([anl3_mu], dtype=np.float64)),
            'DT12': xr.DataArray(data=np.array([anl3_dt12], dtype=np.float64)),
            'Cdeep': xr.DataArray(data=np.array([anl3_cdeep], dtype=np.float64)),
            'Cshal': xr.DataArray(data=np.array([anl3_cshal], dtype=np.float64)),
        }
        self._sim_data.anl3qparms = xr.Dataset(data_vars=anl3qparms)

    def _read_nonlinear_filter_snls_namelist(self, data):
        """Read the nonlinear filter SNLS namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SNLS namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'A34' in cards:
            consts.parameter('SNLSA34').value = float(values[cards.index('A34')])
        if 'FHFC' in cards:
            consts.parameter('SNLSFHFC').value = float(values[cards.index('FHFC')])
        if 'DMN' in cards:
            consts.parameter('SNLSDMN').value = float(values[cards.index('DMN')])
        if 'FC1-3' in cards:
            consts.parameter('SNLSFC13').value = float(values[cards.index('FC1-3')])

    def _read_whitecapping_sds1_namelist(self, data):
        """Read the whitecapping dissipation SDS1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SDS1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CDIS' in cards:
            consts.parameter('SDS1CDIS').value = float(values[cards.index('CDIS')])
        if 'APM' in cards:
            consts.parameter('SDS1APM').value = float(values[cards.index('APM')])
        consts.parameter('whitecapping_dissipation').value = 0

    def _read_whitecapping_sds2_namelist(self, data):
        """Read the whitecapping dissipation SDS2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SDS2 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'SDSA0' in cards:
            consts.parameter('SDS2SDSA0').value = float(values[cards.index('SDSA0')])
        if 'SDSA1' in cards:
            consts.parameter('SDS2SDSA1').value = float(values[cards.index('SDSA1')])
        if 'SDSA2' in cards:
            consts.parameter('SDS2SDSA2').value = float(values[cards.index('SDSA2')])
        if 'SDSB0' in cards:
            consts.parameter('SDS2SDSB0').value = float(values[cards.index('SDSB0')])
        if 'SDSB1' in cards:
            consts.parameter('SDS2SDSB1').value = float(values[cards.index('SDSB1')])
        if 'PHIMIN' in cards:
            float(values[cards.index('PHIMIN')])
        consts.parameter('whitecapping_dissipation').value = 1

    def _read_whitecapping_sds3_namelist(self, data):
        """Read the whitecapping dissipation SDS3 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SDS3 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'SDSC1' in cards:
            consts.parameter('SDS3SDSC1').value = float(values[cards.index('SDSC1')])
        if 'MNMEANP' in cards:
            consts.parameter('SDS3MNMEANP').value = float(values[cards.index('MNMEANP')])
        if 'WNMEANPTAIL' in cards:
            consts.parameter('SDS3WNMEANPTAIL').value = float(values[cards.index('WNMEANPTAIL')])
        if 'SDSDELTA1' in cards:
            consts.parameter('SDS3SDSDELTA1').value = float(values[cards.index('SDSDELTA1')])
        if 'SDSDELTA2' in cards:
            consts.parameter('SDS3SDSDELTA2').value = float(values[cards.index('SDSDELTA2')])
        if 'SDSLF' in cards:
            consts.parameter('SDS3SDSLF').value = float(values[cards.index('SDSLF')])
        if 'SDSHF' in cards:
            consts.parameter('SDS3SDSHF').value = float(values[cards.index('SDSHF')])
        if 'SDSC2' in cards:
            consts.parameter('SDS3SDSC2').value = float(values[cards.index('SDSC2')])
        if 'SDSC4' in cards:
            consts.parameter('SDS3SDSC4').value = float(values[cards.index('SDSC4')])
        if 'SDSBR' in cards:
            consts.parameter('SDS3SDSBR').value = float(values[cards.index('SDSBR')])
        if 'SDSP' in cards:
            consts.parameter('SDS3SDSP').value = float(values[cards.index('SDSP')])
        if 'SDSBR2' in cards:
            consts.parameter('SDS3SDSBR2').value = float(values[cards.index('SDSBR2')])
        if 'SDSC5' in cards:
            consts.parameter('SDS3SDSC5').value = float(values[cards.index('SDSC5')])
        if 'SDSC6' in cards:
            consts.parameter('SDS3SDSC6').value = float(values[cards.index('SDSC6')])
        if 'SDSDTH' in cards:
            consts.parameter('SDS3SDSDTH').value = float(values[cards.index('SDSDTH')])
        consts.parameter('whitecapping_dissipation').value = 2

    def _read_whitecapping_sds6_namelist(self, data):
        """Read the whitecapping dissipation SDS6 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SDS6 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'SDSET' in cards:
            consts.parameter('SDS6SDSET').value = float(values[cards.index('SDSET')])
        if 'SDSA1' in cards:
            consts.parameter('SDS6SDSA1').value = float(values[cards.index('SDSA1')])
        if 'SDSA2' in cards:
            consts.parameter('SDS6SDSA2').value = float(values[cards.index('SDSA2')])
        if 'SDSP1' in cards:
            consts.parameter('SDS6SDSP1').value = float(values[cards.index('SDSP1')])
        if 'SDSP2' in cards:
            consts.parameter('SDS6SDSP2').value = float(values[cards.index('SDSP2')])
        consts.parameter('whitecapping_dissipation').value = 3

    def _read_whitecapping_swl6_namelist(self, data):
        """Read the whitecapping dissipation SWL6 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SWL6 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'SWLB1' in cards:
            consts.parameter('SWL6SWLB1').value = float(values[cards.index('SWLB1')])
        consts.parameter('whitecapping_dissipation').value = 4

    def _read_bottom_friction_sbt1_namelist(self, data):
        """Read the bottom friction SBT1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SBT1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'GAMMA' in cards:
            consts.parameter('SBT1GAMMA').value = float(values[cards.index('GAMMA')])

    def _read_surf_breaking_sdb1_namelist(self, data):
        """Read the surf breaking SDB1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SDB1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        consts.parameter('surf_breaking').value = True
        if 'BJALFA' in cards:
            consts.parameter('BJALFA').value = float(values[cards.index('BJALFA')])
        if 'BJGAM' in cards:
            consts.parameter('BJGAM').value = float(values[cards.index('BJGAM')])
        if 'BJFLAG' in cards:
            # Special case where we store true as 0, false as 1
            val = _get_true_false_as_int(values[cards.index('BJFLAG')])
            val = 'Use Hmax/d Ratio Only' if val == 1 else 'Use Hmax/d in Miche Formulation'
            consts.parameter('BJFLAG').value = val

    def _read_dissipation_in_ice_sic2_namelist(self, data):
        """Read the dissipation in the ice SIC2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SIC2 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'IC2DISPER' in cards:
            # Special case where True is 0, False is 1 (combo box)
            val = _get_true_false_as_int(values[cards.index('IC2DISPER')])
            val = 0 if val == 1 else 1
            consts.parameter('SIC2IC2DISPER').value = val
        if 'IC2TURB' in cards:
            consts.parameter('SIC2IC2TURB').value = float(values[cards.index('IC2TURB')])
        if 'IC2ROUGH' in cards:
            consts.parameter('SIC2IC2ROUGH').value = float(values[cards.index('IC2ROUGH')])
        if 'IC2REYNOLDS' in cards:
            consts.parameter('SIC2IC2REYNOLDS').value = float(values[cards.index('IC2REYNOLDS')])
        if 'IC2SMOOTH' in cards:
            consts.parameter('SIC2IC2SMOOTH').value = float(values[cards.index('IC2SMOOTH')])
        if 'IC2VISC' in cards:
            consts.parameter('SIC2IC2VISC').value = float(values[cards.index('IC2VISC')])

    def _read_scattering_in_ice_sis2_namelist(self, data):
        """Read the scattering in the ice SIS2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SIS2 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'ISC1' in cards:
            consts.parameter('SIS2ISC1').value = float(values[cards.index('ISC1')])
        if 'IS2BACKSCAT' in cards:
            consts.parameter('SIS2IS2BACKSCAT').value = float(values[cards.index('IS2BACKSCAT')])
        if 'IS2BREAK' in cards:
            val = _get_true_false_as_int(values[cards.index('IS2BREAK')])
            consts.parameter('SIS2IS2BREAK').value = val
        if 'IS2C1' in cards:
            consts.parameter('SIS2IS2C1').value = float(values[cards.index('IS2C1')])
        if 'IS2C2' in cards:
            consts.parameter('SIS2IS2C2').value = float(values[cards.index('IS2C2')])
        if 'IS2C3' in cards:
            consts.parameter('SIS2IS2C3').value = float(values[cards.index('IS2C3')])
        if 'ISBACKSCAT' in cards:
            consts.parameter('SIS2ISBACKSCAT').value = float(values[cards.index('ISBACKSCAT')])
        if 'IS2DISP' in cards:
            val = _get_true_false_as_int(values[cards.index('IS2DISP')])
            consts.parameter('SIS2IS2DISP').value = val
        if 'FRAGILITY' in cards:
            consts.parameter('SIS2FRAGILITY').value = float(values[cards.index('FRAGILITY')])
        if 'IS2DMIN' in cards:
            consts.parameter('SIS2IS2DMIN').value = float(values[cards.index('IS2DMIN')])
        if 'IS2DAMP' in cards:
            consts.parameter('SIS2IS2DAMP').value = float(values[cards.index('IS2DAMP')])
        if 'IS2UPDATE' in cards:
            val = _get_true_false_as_int(values[cards.index('IS2UPDATE')])
            val = 0 if val == 1 else 1
            consts.parameter('SIS2').value = val

    def _read_dissipation_by_ice_sic4_namelist(self, data):
        """Read the dissipation by sea ice SIC4 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        self._logger.info('Reading SIC4 namelist...')
        _, _ = self._get_namelist_cards_and_values(data)

    def _read_triad_nonlinear_str1_namelist(self, data):
        """Read the triad nonlinear interactions STR1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading STR1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'PTRIAD1' in cards:
            consts.parameter('STR1PTRIAD1').value = float(values[cards.index('PTRIAD1')])
        if 'PTRIAD2' in cards:
            consts.parameter('STR1PTRIAD2').value = float(values[cards.index('PTRIAD2')])
        if 'PTRIAD3' in cards:
            consts.parameter('STR1PTRIAD3').value = float(values[cards.index('PTRIAD3')])
        if 'PTRIAD4' in cards:
            consts.parameter('STR1PTRIAD4').value = float(values[cards.index('PTRIAD4')])
        if 'PTRIAD5' in cards:
            consts.parameter('STR1PTRIAD5').value = float(values[cards.index('PTRIAD5')])

    def _read_shoreline_reflections_ref1_namelist(self, data):
        """Read the shoreline reflections REF1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading REF1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'REFCOAST' in cards:
            consts.parameter('REF1REFCOAST').value = float(values[cards.index('REFCOAST')])
        if 'REFFREQ' in cards:
            consts.parameter('REF1REFFREQ').value = float(values[cards.index('REFFREQ')])
        if 'REFMAP' in cards:
            consts.parameter('REF1REFMAP').value = float(values[cards.index('REFMAP')])
        if 'REFRMAX' in cards:
            consts.parameter('REF1REFRMAX').value = float(values[cards.index('REFRMAX')])
        if 'REFFREQPOW' in cards:
            consts.parameter('REF1REFFREQPOW').value = float(values[cards.index('REFFREQPOW')])
        if 'REFICEBERG' in cards:
            consts.parameter('REF1REFICEBERG').value = float(values[cards.index('REFICEBERG')])
        if 'REFSUBGRID' in cards:
            consts.parameter('REF1REFSUBGRID').value = float(values[cards.index('REFSUBGRID')])
        if 'REFCOSP_STRAIGHT' in cards:
            consts.parameter('REF1REFCOSP_STRAIGHT').value = float(values[cards.index('REFCOSP_STRAIGHT')])

    def _read_free_ig_sig1_namelist(self, data):
        """Read the bound 2nd order spectrum and free IG SIG1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading SIG1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'IGMETHOD' in cards:
            # Limited to 1 or 2, but stored as 0 or 1
            val = int(float(values[cards.index('IGMETHOD')]))
            val = 0 if val == 1 else 1
            consts.parameter('SIG1IGMETHOD').value = val
        if 'IGADDOUTP' in cards:
            consts.parameter('SIG1IGADDOUTP').value = float(values[cards.index('IGADDOUTP')])
        if 'IGSOURCE' in cards:
            # Limited to 1 or 2, but stored as 0 or 1
            val = int(float(values[cards.index('IGSOURCE')]))
            val = 0 if val == 1 else 1
            consts.parameter('SIG1IGSOURCE').value = val
        if 'IGSTERMS' in cards:
            consts.parameter('SIG1IGSTERMS').value = float(values[cards.index('IGSTERMS')])
        if 'IGMAXFREQ' in cards:
            consts.parameter('SIG1IGMAXFREQ').value = float(values[cards.index('IGMAXFREQ')])
        if 'IGEMPERICAL' in cards:
            consts.parameter('SIG1IGEMPIRICAL').value = float(values[cards.index('IGEMPERICAL')])
        if 'IGBCOVERWRITE' in cards:
            val = _get_true_false_as_int(values[cards.index('IGBCOVERWRITE')])
            consts.parameter('SIG1IGBCOVERWRITE').value = val
        if 'IGSWELLMAX' in cards:
            val = _get_true_false_as_int(values[cards.index('IGSWELLMAX')])
            consts.parameter('SIG1IGSWELLMAX').value = val

    def _read_propagation_pro1_namelist(self, data):
        """Read the propagation PRO1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading PRO1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CFLTM' in cards:
            consts.parameter('PROCFLTM').value = float(values[cards.index('CFLTM')])
        consts.parameter('propagation_schemes').value = 0

    def _read_propagation_pro2_namelist(self, data):
        """Read the propagation PRO2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading PRO2 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CFLTM' in cards:
            consts.parameter('PROCFLTM').value = float(values[cards.index('CFLTM')])
        if 'DTIME' in cards:
            consts.parameter('PRO2DTIME').value = float(values[cards.index('DTIME')])
        if 'LATMIN' in cards:
            consts.parameter('PRO2LATMIN').value = float(values[cards.index('LATMIN')])
        consts.parameter('propagation_schemes').value = 1

    def _read_propagation_pro3_namelist(self, data):
        """Read the propagation PRO3 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        self._logger.info('Reading PRO3 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CFLTM' in cards:
            consts.parameter('PROCFLTM').value = float(values[cards.index('CFLTM')])
        if 'WDTHCG' in cards:
            consts.parameter('PRO3WDTHCG').value = float(values[cards.index('WDTHCG')])
        if 'WDTHTH' in cards:
            consts.parameter('PRO3WDTHTH').value = float(values[cards.index('WDTHTH')])
        consts.parameter('propagation_schemes').value = 2

    def _read_unstructured_grids_unst_namelist(self, data):
        """Read the unstructured grids UNST namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        grids = model_parameters.group('grids')
        self._logger.info('Reading UNST namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        # EXPFSN, EXPFSPSI, EXPFSFCT, IMPFSN, EXPTOTAL, IMPREFRACTION, IMPFREQSHIFT and IMPSOURCE are removed since
        # their values are calculated just before export
        if 'UGOBCAUTO' in cards:
            val = _get_true_false_as_int(values[cards.index('UGOBCAUTO')])
            grids.parameter('UGOBCAUTO').value = 'Listed in ww3_grid.nml' if val == 0 \
                else 'Taken From Type 15 Elements'
        if 'UGOBCDEPTH' in cards:
            grids.parameter('UGOBCDEPTH').value = float(values[cards.index('UGOBCDEPTH')])
        if 'IMPTOTAL' in cards:
            val = _get_true_false_as_int(values[cards.index('IMPTOTAL')])
            grids.parameter('method').value = 'Explicit' if val == 0 else 'Implicit'
        if 'JGS_TERMINATE_MAXITER' in cards:
            val = _get_true_false_as_int(values[cards.index('JGS_TERMINATE_MAXITER')])
            grids.parameter('JGS_TERMINATE_MAXITER').value = val
        if 'JGS_TERMINATE_DIFFERENCE' in cards:
            val = _get_true_false_as_int(values[cards.index('JGS_TERMINATE_DIFFERENCE')])
            grids.parameter('JGS_TERMINATE_DIFFERENCE').value = val
        if 'JGS_TERMINATE_NORM' in cards:
            val = _get_true_false_as_int(values[cards.index('JGS_TERMINATE_NORM')])
            grids.parameter('JGS_TERMINATE_NORM').value = val
        if 'JGS_USE_JACOBI' in cards:
            val = _get_true_false_as_int(values[cards.index('JGS_USE_JACOBI')])
            grids.parameter('UNSTJGS_USE_JACOBI').value = val
        if 'JGS_BLOCK_GAUSS_SEIDEL' in cards:
            val = _get_true_false_as_int(values[cards.index('JGS_BLOCK_GAUSS_SEIDEL')])
            grids.parameter('UNSTJGS_BLOCK_GAUSS_SEIDEL').value = val
        if 'JGS_MAXITER' in cards:
            grids.parameter('JGS_MAXITER').value = int(float(values[cards.index('JGS_MAXITER')]))
        if 'JGS_PMIN' in cards:
            grids.parameter('JGS_PMIN').value = float(values[cards.index('JGS_PMIN')])
        if 'JGS_DIFF_THR' in cards:
            grids.parameter('JGS_DIFF_THR').value = float(values[cards.index('JGS_DIFF_THR')])
        if 'JGS_NORM_THR' in cards:
            grids.parameter('JGS_NORM_THR').value = float(values[cards.index('JGS_NORM_THR')])
        if 'SETUP_APPLY_WLV' in cards:
            val = _get_true_false_as_int(values[cards.index('SETUP_APPLY_WLV')])
            grids.parameter('UNSTSETUP_APPLY_WLV').value = val
        if 'SOLVERTHR_SETUP' in cards:
            grids.parameter('UNSTSOLVERTHR_SETUP').value = float(values[cards.index('SOLVERTHR_SETUP')])
        if 'CRIT_DEP_SETUP' in cards:
            grids.parameter('UNSTCRIT_DEP_SETUP').value = float(values[cards.index('CRIT_DEP_SETUP')])
        grids.parameter('UNST').value = 'UNSPT parameters [UNST]'

    def _read_unstructured_grids_psmc_namelist(self, data):
        """Read the unstructured grids PSMC namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        grids = model_parameters.group('grids')
        self._logger.info('Reading PSMC namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CFLTM' in cards:
            consts.parameter('PSMCCFLTM').value = float(values[cards.index('CFLTM')])
        if 'DTIME' in cards:
            consts.parameter('PSMCDTIME').value = float(values[cards.index('DTIME')])
        if 'LATMIN' in cards:
            consts.parameter('PSMCLATMIN').value = float(values[cards.index('LATMIN')])
        if 'RFMAXD' in cards:
            consts.parameter('PSMCRFMAXD').value = float(values[cards.index('RFMAXD')])
        if 'LvSMC' in cards:
            consts.parameter('PSMCLvSMC').value = int(float(values[cards.index('LvSMC')]))
        if 'ISHFT' in cards:
            consts.parameter('PSMCISHFT').value = int(float(values[cards.index('ISHFT')]))
        if 'JEQT' in cards:
            consts.parameter('PSMCJEQT').value = int(float(values[cards.index('JEQT')]))
        if 'NBISMC' in cards:
            consts.parameter('PSMCNBISMC').value = int(float(values[cards.index('NBISMC')]))
        if 'UNO3' in cards:
            val = _get_true_false_as_int(values[cards.index('UNO3')])
            consts.parameter('PSMCUNO3').value = val
        if 'AVERG' in cards:
            val = _get_true_false_as_int(values[cards.index('AVERG')])
            consts.parameter('PSMCUNO3Add').value = val
        # Change this to PSMC once it becomes enabled in the model control
        grids.parameter('UNST').value = 'UNSPT parameters [UNST]'

    def _read_output_outs_namelist(self, data):
        """Read the outptut OUTS namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        consts = model_parameters.group('consts')
        miscellaneous = model_parameters.group('miscellaneous')
        self._logger.info('Reading OUTS namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'E3D' in cards:
            consts.parameter('E3D').value = int(float(values[cards.index('E3D')]))
        if 'I1E3D' in cards:
            consts.parameter('I1E3D').value = int(float(values[cards.index('I1E3D')]))
        if 'I2E3D' in cards:
            consts.parameter('I2E3D').value = int(float(values[cards.index('I2E3D')]))
        if 'TH1MF' in cards:
            consts.parameter('TH1MF').value = int(float(values[cards.index('TH1MF')]))
        if 'I1TH1MF' in cards:
            consts.parameter('I1TH1MF').value = int(float(values[cards.index('I1TH1MF')]))
        if 'I2TH1MF' in cards:
            consts.parameter('I2TH1MF').value = int(float(values[cards.index('I2TH1MF')]))
        if 'STH1MF' in cards:
            consts.parameter('STH1MF').value = int(float(values[cards.index('STH1MF')]))
        if 'I1STH1MF' in cards:
            consts.parameter('I1STH1MF').value = int(float(values[cards.index('I1STH1MF')]))
        if 'I2STH1MF' in cards:
            consts.parameter('I2STH1MF').value = int(float(values[cards.index('I2STH1MF')]))
        if 'TH2MF' in cards:
            consts.parameter('TH2MF').value = int(float(values[cards.index('TH2MF')]))
        if 'I1TH2MF' in cards:
            consts.parameter('I1TH2MF').value = int(float(values[cards.index('I1TH2MF')]))
        if 'I2TH2MF' in cards:
            consts.parameter('I2TH2MF').value = int(float(values[cards.index('I2TH2MF')]))
        if 'STH2MF' in cards:
            consts.parameter('STH2MF').value = int(float(values[cards.index('STH2MF')]))
        if 'I1STH2MF' in cards:
            consts.parameter('I1STH2MF').value = int(float(values[cards.index('I1STH2MF')]))
        if 'I2STH2MF' in cards:
            consts.parameter('I2STH2MF').value = int(float(values[cards.index('I2STH2MF')]))
        if 'P2SF' in cards:
            consts.parameter('P2SFOut').value = int(float(values[cards.index('P2SF')]))
        if 'I1P2SF' in cards:
            miscellaneous.parameter('I1P2SF').value = int(float(values[cards.index('I1P2SF')]))
        if 'I2P2SF' in cards:
            miscellaneous.parameter('I2P2SF').value = int(float(values[cards.index('I2P2SF')]))

    def _read_miscellaneous_misc_namelist(self, data):
        """Read the miscellaneous MISC namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        miscellaneous = model_parameters.group('miscellaneous')
        self._logger.info('Reading MISC namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'CICE0' in cards:
            miscellaneous.parameter('CICE0').value = float(values[cards.index('CICE0')])
        if 'CICEN' in cards:
            miscellaneous.parameter('CICEN').value = float(values[cards.index('CICEN')])
        if 'PMOVE' in cards:
            miscellaneous.parameter('PMOVE').value = float(values[cards.index('PMOVE')])
        if 'XSEED' in cards:
            miscellaneous.parameter('XSEED').value = float(values[cards.index('XSEED')])
        if 'FLAGTR' in cards:
            miscellaneous.parameter('FLAGTR').value = int(float(values[cards.index('FLAGTR')]))
        if 'XP' in cards:
            miscellaneous.parameter('XP').value = float(values[cards.index('XP')])
        if 'XR' in cards:
            miscellaneous.parameter('XR').value = float(values[cards.index('XR')])
        if 'XFILT' in cards:
            miscellaneous.parameter('XFILT').value = float(values[cards.index('XFILT')])
        if 'IHMAX' in cards:
            miscellaneous.parameter('XIHMAX').value = int(float(values[cards.index('IHMAX')]))
        if 'HSPMIN' in cards:
            miscellaneous.parameter('HSPMIN').value = float(values[cards.index('HSPMIN')])
        if 'WSM' in cards:
            miscellaneous.parameter('WSM').value = float(values[cards.index('WSM')])
        if 'WSC' in cards:
            miscellaneous.parameter('WSC').value = float(values[cards.index('WSC')])
        if 'FLC' in cards:
            miscellaneous.parameter('FLC').value = float(values[cards.index('FLC')])
        if 'NOSW' in cards:
            miscellaneous.parameter('NOSW').value = int(float(values[cards.index('NOSW')]))
        if 'FMICHE' in cards:
            miscellaneous.parameter('FMICHE').value = float(values[cards.index('FMICHE')])
        if 'STDX' in cards:
            miscellaneous.parameter('STDX').value = float(values[cards.index('STDX')])
        if 'STDY' in cards:
            miscellaneous.parameter('STDY').value = float(values[cards.index('STDY')])
        if 'STDT' in cards:
            miscellaneous.parameter('STDT').value = float(values[cards.index('STDT')])
        if 'P2SF' in cards:
            miscellaneous.parameter('P2SF').value = int(float(values[cards.index('P2SF')]))

    def _read_sea_state_fld1_namelist(self, data):
        """Read the sea state stress FLD1 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        sea_state_stress = model_parameters.group('sea_state_stress')
        self._logger.info('Reading FLD1 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'TAILTYPE' in cards:
            sea_state_stress.parameter('TAILTYPE').value = int(float(values[cards.index('TAILTYPE')]))
        if 'TAILLEV' in cards:
            sea_state_stress.parameter('TAILLEV').value = float(values[cards.index('TAILLEV')])
        if 'TAILT1' in cards:
            sea_state_stress.parameter('TAILT1').value = float(values[cards.index('TAILT1')])
        if 'TAILT2' in cards:
            sea_state_stress.parameter('TAILT2').value = float(values[cards.index('TAILT2')])
        sea_state_stress.parameter('FLDMethod').value = 0

    def _read_sea_state_fld2_namelist(self, data):
        """Read the sea state stress FLD2 namelist.

        Args:
            data(:obj:`list[str]`): The current line (including the namelist ID) that may contain more data.
        """
        model_parameters = self._global_values
        sea_state_stress = model_parameters.group('sea_state_stress')
        self._logger.info('Reading FLD2 namelist...')
        cards, values = self._get_namelist_cards_and_values(data)
        if 'TAILTYPE' in cards:
            sea_state_stress.parameter('TAILTYPE').value = int(float(values[cards.index('TAILTYPE')]))
        if 'TAILLEV' in cards:
            sea_state_stress.parameter('TAILLEV').value = float(values[cards.index('TAILLEV')])
        if 'TAILT1' in cards:
            sea_state_stress.parameter('TAILT1').value = float(values[cards.index('TAILT1')])
        if 'TAILT2' in cards:
            sea_state_stress.parameter('TAILT2').value = float(values[cards.index('TAILT2')])
        sea_state_stress.parameter('FLDMethod').value = 1

    def read(self):
        """Top-level entry point for the WaveWatch3 namelists nml input file reader."""
        try:
            self._logger.info('Parsing ASCII text from file...')
            with open(self._filename, 'r', buffering=READ_BUFFER_SIZE) as f:
                self._lines = f.readlines()
            self._read_namelists_nml_file()
            self._logger.info('Committing changes....')
            self._sim_data.global_values = self._global_values.extract_values()
            self._sim_data.commit()
            self._logger.info('Finished!')
        except Exception:
            self._logger.exception(
                'Unexpected error in namelists nml preprocessor file '
                f'(line {self._current_line + 1}).'
            )
            raise
