"""Class to write a WaveWatch3 ounf nml file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
from io import StringIO
import logging
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.time_format import string_to_datetime

# 4. Local modules
from xms.wavewatch3.file_io.namelists_defaults_util import WW3NamelistsDefaultsUtil
from xms.wavewatch3.gui.gui_util import get_formatted_date_string, get_output_field_type_string


class WW3OunfNmlWriter:
    """Class to write a WaveWatch3 ounf nml file (related to grid namelist)."""
    def __init__(self, xms_data):
        """Constructor.

        Args:
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
        """
        self._ss = StringIO()
        self._logger = logging.getLogger('xms.wavewatch3')
        self._bound_process = None
        self._grid_process = None
        self._xms_data = xms_data
        self._defaults = WW3NamelistsDefaultsUtil()

    def _write_ounf_nml_file(self):
        """Writes the ounf namelist file."""
        file_w_path = "ww3_ounf.nml"

        self._write_ounf_header()
        self._write_field_namelist()
        self._write_file_namelist()
        self._write_smc_namelist()
        self._write_end_comments()
        self._flush(file_w_path)

    def _write_ounf_header(self):
        """Writes the header comments for the ounf namelist file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - ww3_ounf.nml - Grid output post-processing           !\n'
            '! -------------------------------------------------------------------- !\n'
            '\n\n'
        )

    def _write_field_namelist(self):
        """Writes the FIELD_NML namelist for the output fields to postprocess."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! Define the output fields to postprocess via FIELD_NML namelist\n'
            '!\n'
            '! * the detailed list of field names FIELD%LIST is given in ww3_shel.nml\n'
            '!  DPT CUR WND AST WLV ICE TAU RHO IBG D50 IC1 IC5\n'
            '!  HS LM T02 T0M1 T01 FP DIR SPR DP HIG WNM\n'
            '!  EF TH1M STH1M TH2M STH2M WN\n'
            '!  PHS PTP PLP PDIR PSPR PWS PDP PQP PPE PGW PSW PTM10 PT01 PT02 PEP TWS PNR\n'
            '!  UST CHA CGE FAW TAW TWA WCC WCF WCH WCM FWS\n'
            '!  SXY TWO BHD FOC TUS USS P2S USF P2L TWI FIC TOC\n'
            '!  ABR UBR BED FBB TBB\n'
            '!  MSS MSC WL02 AXT AYT AXY\n'
            '!  DTD FC CFX CFD CFK\n'
            '!  U1 U2\n'
            '!\n'
            '! * namelist must be terminated with /\n'
            '! * definitions & defaults:\n'
            "!     FIELD%TIMESTART            = '19000101 000000'  ! Stop date for the output field\n"
            "!     FIELD%TIMESTRIDE           = '0'                ! Time stride for the output field\n"
            "!     FIELD%TIMECOUNT            = '1000000000'       ! Number of time steps\n"
            '!     FIELD%TIMESPLIT            = 6                  '
            '! [0(nodate),4(yearly),6(monthly),8(daily),10(hourly)]\n'
            "!     FIELD%LIST                 = 'unset'            ! List of output fields\n"
            "!     FIELD%PARTITION            = '0 1 2 3'          "
            "! List of wave partitions ['0 1 2 3 4 5']\n"
            '!     FIELD%SAMEFILE             = T                  ! All the variables in the same file\n'
            '!     FIELD%VECTOR               = T                  ! Vector [T] or dir/magnitude [F] for\n'
            '!                                                     !   directional fields\n'
            '!     FIELD%TYPE                 = 3                  '
            '! [2 = SHORT, 3 = it depends , 4 = REAL]\n'
            '!     FIELD%FCVARS               = F                  ! Generate auxiliary forecast variables\n'
            '!                                                     '
            '!   (forecast_period and forecast_reference_time)\n'
            "!     FIELD%TIMEREF              = 'unset'            "
            "! \"Forecast reference time\" for calculating\n"
            '!                                                     '
            '!   forecast_period; defaults to TIMESTART\n'
            "!     FIELD%TIMEVAR              = 'D'                "
            "! Time var type ['D' = DOUBLE, 'I' = INT64]\n"
            "!     FIELD%TIMEUNIT             = 'D'                "
            "! Time units ['D' = days, 'I' = seconds]\n"
            "!     FIELD%TIMEEPOCH            = '19900101 000000'  "
            "! Epoch used for encoding of NC time variables\n"
            '!     FIELD%NOVAL                = UNDEF              '
            '! Value for wet cells that have an UNDEF value\n'
            '!     FIELD%MAPSTA               = .TRUE.             ! Output MAPSTA field in file?\n'
            '!\n'
            '! Note: If FCVARS = T, the following auxiliary variables will be generated\n'
            '! (see the manual entry for ww3_ounf for more information):\n'
            '!  \n'
            '!    - forecast_reference_time: The time associated with the "analysis" of\n'
            '!      the current forecast. Defaults to TIMESTART if TIMEREF not set.\n'
            '!\n'
            '!    - forecast_period: the time period elapsed (in seconds) since the\n'
            '!      associated "forecast_reference_time".\n'
            '! -------------------------------------------------------------------- !\n'
        )
        ounf = self._xms_data.sim_data_output_fields.group('user_defined')  # OUNF is in the user-defined group
        time_params = self._xms_data.sim_data_output_fields.group('time_parameters')
        self._ss.write('&FIELD_NML\n')
        time_split = [0, 4, 6, 8, 10]
        # field_type = [2, 3, 4]
        # not sure why we aren't using field_timestart here
        time_start = string_to_datetime(time_params.parameter('output_start_date').value)
        self._ss.write(f"    FIELD%TIMESTART      = '{get_formatted_date_string(time_start)}'\n")
        # also not sure why we aren't using field_timestride here
        if time_params.parameter('output_second_interval').value != 0:
            self._ss.write(f"    FIELD%TIMESTRIDE     = '{time_params.parameter('output_second_interval').value}'\n")
        if ounf.parameter('field_timecount').value != 1000000000:
            self._ss.write(f"    FIELD%TIMECOUNT      = '{ounf.parameter('field_timecount').value}'\n")
        if ounf.parameter('field_timesplit').value in time_split:
            self._ss.write(f"    FIELD%SPLIT          = {ounf.parameter('field_timesplit').value}\n")
        output_fields = self._xms_data.sim_data_output_fields
        self._ss.write(f"    FIELD%LIST           = '{get_output_field_type_string(output_fields)}'\n")
        if ounf.parameter('field_samefile').value != 1:
            self._ss.write("    FIELD%SAMEFILE      = F\n")
        if ounf.parameter('field_vector').value != 1:
            self._ss.write("    FIELD%VECTOR         = F\n")
        self._ss.write("    FIELD%TYPE           = 4\n")  # Should always write FIELD%TYPE = 4 for REAL
        self._ss.write('/\n\n\n')

    def _write_file_namelist(self):
        """Writes the FILE_NML namelist for the defining the content of the output file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! Define the content of the output file via FILE_NML namelist\n'
            '!\n'
            '! * namelist must be terminated with /\n'
            '! * definitions & defaults:\n'
            "!     FILE%PREFIX        = 'ww3.'            ! Prefix for output file name\n"
            '!     FILE%NETCDF        = 3                 ! Netcdf version [3|4]\n'
            '!     FILE%IX0           = 1                 ! First X-axis or node index\n'
            '!     FILE%IXN           = 1000000000        ! Last X-axis or node index\n'
            '!     FILE%IY0           = 1                 ! First Y-axis index\n'
            '!     FILE%IYN           = 1000000000        ! Last Y-axis index\n'
            '! -------------------------------------------------------------------- !\n'
        )
        ounf = self._xms_data.sim_data_output_fields.group('user_defined')  # OUNF is in the user-defined group
        self._ss.write('&FILE_NML\n')
        netcdf_version = [3, 4]
        if ounf.parameter('file_prefix').value != 'ww3.':
            self._ss.write(f"    FILE%PREFIX          = '{ounf.parameter('file_prefix').value}'\n")
        if ounf.parameter('file_netcdf').value in netcdf_version:
            self._ss.write(f"    FILE%NETCDF          = {ounf.parameter('file_netcdf').value}\n")
        if ounf.parameter('file_ix0').value != 1:
            self._ss.write(f"    FILE%IX0             = {ounf.parameter('file_ix0').value}\n")
        if ounf.parameter('file_ixn').value != 1000000000:
            self._ss.write(f"    FILE%IXN             = {ounf.parameter('file_ixn').value}\n")
        if ounf.parameter('file_iy0').value != 1:
            self._ss.write(f"    FILE%IY0             = {ounf.parameter('file_iy0').value}\n")
        if ounf.parameter('file_iyn').value != 1000000000:
            self._ss.write(f"    FILE%IYN             = {ounf.parameter('file_iyn').value}\n")
        self._ss.write('/\n\n\n')

    def _write_smc_namelist(self):
        """Writes the SMC_NML namelist for the defining the content of the output file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! Define the content of the output file via SMC_NML namelist\n'
            '!\n'
            '! * For SMC grids, IX0, IXN, IY0 and IYN from FILE_NML are not used.\n'
            '!   Two types of output are available:\n'
            '! *   TYPE=1: Flat 1D "seapoint" array of grid cells.\n'
            '! *   TYPE=2: Re-gridded regular grid with cell sizes being an integer\n'
            '! *           multiple of the smallest SMC grid cells size.\n'
            '!\n'
            '! * Note that the first/last longitudes and latitudes will be adjusted\n'
            '!  to snap to the underlying SMC grid edges. CELFAC is only used for\n'
            '!  type 2 output and defines the output cell sizes as an integer\n'
            '!  multiple of the smallest SMC Grid cell size. CELFAC should be a\n'
            '!  power of 2, e.g: 1,2,4,8,16, etc...\n'
            '!\n'
            '! * namelist must be terminated with /\n'
            '! * definitions & defaults:\n'
            '!     SMC%TYPE          = 1              ! SMC Grid type (1 or 2)\n'
            '!     SMC%SXO           = -999.9         ! First longitude\n'
            '!     SMC%EXO           = -999.9         ! Last longitude\n'
            '!     SMC%SYO           = -999.9         ! First latitude\n'
            '!     SMC%EYO           = -999.9         ! Last latitude\n'
            '!     SMC%CELFAC        = 1              ! Cell size factor (SMCTYPE=2 only)\n'
            '!     SMC%NOVAL         = UNDEF          ! Fill value for wet cells with no data\n'
            '! -------------------------------------------------------------------- !\n'
        )
        self._ss.write('&SMC_NML\n')
        self._ss.write('/\n\n\n')

    def _write_end_comments(self):
        """Writes out the end comments at the end of the file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - end of namelist                                      !\n'
            '! -------------------------------------------------------------------- !\n'
            '\n'
        )

    def _flush(self, file_w_path):
        """Writes the StringIO previously processed to a file.

        Args:
            file_w_path (:obj:`str`):  String of the filename to write to.
        """
        f = open(file_w_path, 'w')
        self._ss.seek(0)
        shutil.copyfileobj(self._ss, f, 100000)
        f.close()

    def write(self):
        """Top-level entry point for the WaveWatch3 ounf file writer."""
        self._write_ounf_nml_file()
