"""Class to read a WaveWatch3 shell namelist file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
import glob
import logging
import os
import re
import shlex
from typing import cast
import uuid

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.coverage_component_builder import CoverageComponentBuilder
from xms.data_objects.parameters import Component, Point, Projection, Simulation
from xms.gmi.data_bases.coverage_base_data import CoverageBaseData
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.wavewatch3.components.bc_component import BcComponent
from xms.wavewatch3.components.output_points_component import OutputPointsComponent
from xms.wavewatch3.components.sim_component import SIM_DATA_MAINFILE, SimData
from xms.wavewatch3.data.model import get_model
from xms.wavewatch3.data.output_fields import get_output_field_model
from xms.wavewatch3.file_io.bounc_nml_reader import BouncNmlReader
from xms.wavewatch3.file_io.gmsh_reader import GmshReader
from xms.wavewatch3.file_io.grid_nml_reader import GridNmlReader
from xms.wavewatch3.file_io.io_util import (create_component_folder, GEOGRAPHIC_WKT, LOCAL_METERS_WKT, READ_BUFFER_SIZE)
from xms.wavewatch3.file_io.namelists_nml_reader import NamelistsNmlReader
from xms.wavewatch3.file_io.netcdf_dataset_reader import NetcdfDatasetReader
from xms.wavewatch3.file_io.ounf_nml_reader import OunfNmlReader
from xms.wavewatch3.file_io.ounp_nml_reader import OunpNmlReader
from xms.wavewatch3.file_io.prnc_nml_reader import PrncNmlReader
from xms.wavewatch3.file_io.spec_list_reader import WW3SpecListReader
from xms.wavewatch3.gui.gui_util import get_date_time_string, get_output_field_type_list


class ShellNmlReader:
    """Class to read a WaveWatch3 shell Namelist file."""
    def __init__(self, filename=''):
        """Constructor.

        Args:
            filename (:obj:`str`): Path to the mesh file. If not provided (not testing or control file read),
                will retrieve from Query.
        """
        self._filename = filename
        self._mapped_bc_dir = ''
        self._mapped_data = None
        self._query = None
        self._sim_data = None
        self._sim_comp_uuid = str(uuid.uuid4())
        self._setup_query()
        self._lines = []
        self._current_line = 0
        self._logger = logging.getLogger('xms.wavewatch3')
        self._output_points = []
        self._output_point_names = []
        self._inbound_points = {}
        self._lateral_points = []
        self._grid_name = ''
        self._is_geographic = True
        self._build_data = {
            'sim': None,
            'sim_comp': None,
            'links': [],
            'output_pts_cov': None,
            'output_pts_cov_comp': None,
            'output_pts_keywords': None,
            'bc_cov': None,
            'bc_cov_comp': None,
            'bc_cov_keywords': None,
            'do_ugrid': None,
            'co_ugrid': None,
            'spectral_coverages': [],
        }
        self._num_ic1 = 0
        self._num_ic2 = 0
        self._num_ic3 = 0
        self._num_ic4 = 0
        self._num_ic5 = 0
        self._num_mdn = 0
        self._num_mth = 0
        self._num_mvs = 0
        self._num_lev = 0
        self._num_cur = 0
        self._num_wnd = 0
        self._output_fields_section = get_output_field_model().model_parameters
        self._global_values = get_model().global_parameters

    def _setup_query(self):
        """Setup the xmsapi Query for sending data to SMS and get the import filename."""
        if not self._filename:  # pragma: no cover
            self._query = Query()
            self._filename = self._query.read_file

            comp_dir = os.path.join(self._query.xms_temp_directory, 'Components')
            self._mapped_bc_dir = os.path.join(comp_dir, str(uuid.uuid4()))
            os.makedirs(self._mapped_bc_dir, exist_ok=True)

    def _parse_next_line(self, shell=False):
        """Parse the next line of text from the file.

        Skips empty and comment lines.

        Args:
            shell (:obj:`bool`): If True will parse line using shlex. Slower but convenient for quoted tokens.

        Returns:
            (:obj:`list[str]`): The next line of text, split on whitespace
        """
        line = None
        while not line or line.startswith('!'):  # blank lines and control file identifier
            if self._current_line >= len(self._lines):
                # raise RuntimeError('Unexpected end of file.')
                return None
            line = self._lines[self._current_line].strip()
            self._current_line += 1
        if shell:
            return shlex.split(line, posix=False)
        return line.split()

    def _read_shell_nml_file(self):
        reading = True
        while reading:
            data = self._parse_next_line()
            if data:
                if '&DOMAIN_NML' in data[0].strip():
                    self._read_domain_namelist()
                elif '&INPUT_NML' in data[0]:
                    self._read_input_namelist()
                elif '&OUTPUT_TYPE_NML' in data[0]:
                    self._read_output_type_namelist()
                elif '&OUTPUT_DATE_NML' in data[0]:
                    self._read_output_date_namelist()
                elif '&OUTPUT_PATH_NML' in data[0]:
                    self._read_output_path_namelist()
                elif '&HOMOG_COUNT_NML' in data[0]:
                    self._read_homogenous_count_namelist()
                elif '&HOMOG_INPUT_NML' in data[0]:
                    self._read_homogenous_input_namelist()
                else:
                    raise ValueError(f'Unrecognized namelist {data}')
            else:
                reading = False
            self._sim_data.global_values = self._global_values.extract_values()
            self._sim_data.output_field_values = self._output_fields_section.extract_values()

    def _read_domain_namelist(self):
        """Read the name list containing the top level parameters."""
        parameters = self._global_values
        self._logger.info('Reading domain namelist...')
        end_not_found = True
        while end_not_found:
            data = self._parse_next_line()
            if '/' in data[0]:
                end_not_found = False
            elif 'DOMAIN%IOSTYP' in data[0].strip():
                parameters.group('output').parameter('IOSType').value = int(data[2])
            elif 'DOMAIN%START' in data[0].strip():
                datetime = get_date_time_string(data[2], data[3])
                parameters.group('run_control').parameter('starting_date').value = datetime
            elif 'DOMAIN%STOP' in data[0].strip():
                parameters.group('run_control').parameter('end_date').value = get_date_time_string(data[2], data[3])
            else:
                raise ValueError(f'Unrecognized entry in domain namelist {data[0]}')

    def _read_input_namelist(self):
        """Read the name list containing the forcing parameters."""
        parameters = self._global_values
        self._logger.info('Reading input namelist...')
        end_not_found = True
        while end_not_found:
            data = self._parse_next_line()
            if '/' in data[0]:
                end_not_found = False
            elif 'INPUT%FORCING%WATER_LEVELS' in data[0].strip():
                parameters.group('run_control').parameter('water_levels').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%CURRENTS' in data[0].strip():
                parameters.group('run_control').parameter('currents').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%WINDS' in data[0].strip():
                parameters.group('run_control').parameter('winds').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%ATM_MOMENTUM' in data[0].strip():
                flag = self._get_use_flag(data[2][1:-1])
                parameters.group('run_control').parameter('define_atm_momentum').value = flag
            elif 'INPUT%FORCING%AIR_DENSITY' in data[0].strip():
                parameters.group('run_control').parameter('air_density').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%ICE_CONC' in data[0].strip():
                parameters.group('ice_and_mud').parameter('concentration').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%ICE_PARAM1' in data[0].strip():
                parameters.group('ice_and_mud').parameter('param_1').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%ICE_PARAM2' in data[0].strip():
                parameters.group('ice_and_mud').parameter('param_2').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%ICE_PARAM3' in data[0].strip():
                parameters.group('ice_and_mud').parameter('param_3').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%ICE_PARAM4' in data[0].strip():
                parameters.group('ice_and_mud').parameter('param_4').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%ICE_PARAM5' in data[0].strip():
                parameters.group('ice_and_mud').parameter('param_5').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%MUD_DENSITY' in data[0].strip():
                parameters.group('ice_and_mud').parameter('mud_density').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%MUD_THICKNESS' in data[0].strip():
                parameters.group('ice_and_mud').parameter('mud_thickness').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%FORCING%MUD_VISCOSITY' in data[0].strip():
                parameters.group('ice_and_mud').parameter('mud_viscosity').value = self._get_use_flag(data[2][1:-1])
            elif 'INPUT%ASSIM%MEAN' in data[0].strip():
                flag = self._get_use_flag(data[2][1:-1])
                parameters.group('ice_and_mud').parameter('assimilation_data_mean_para').value = flag
            elif 'INPUT%ASSIM%SPEC1D' in data[0].strip():
                flag = self._get_use_flag(data[2][1:-1])
                parameters.group('ice_and_mud').parameter('assimilation_data_1D_spectra').value = flag
            elif 'INPUT%ASSIM%SPEC2D' in data[0].strip():
                flag = self._get_use_flag(data[2][1:-1])
                parameters.group('ice_and_mud').parameter('assimilation_data_2D_spectra').value = flag
            else:
                raise ValueError(f'Unrecognized entry in input namelist {data[0]}')

    def _read_output_type_namelist(self):
        """Read the name list containing the output types points parameters."""
        self._logger.info('Reading output type namelist...')
        expected_fields = {
            'DPT': 'forcing_fields',
            'CUR': 'forcing_fields',
            'WND': 'forcing_fields',
            'AST': 'forcing_fields',
            'WLV': 'forcing_fields',
            'ICE': 'forcing_fields',
            'IBG': 'forcing_fields',
            'D50': 'forcing_fields',
            'IC1': 'forcing_fields',
            'IC5': 'forcing_fields',
            'HS': 'mean_wave',
            'LM': 'mean_wave',
            'T02': 'mean_wave',
            'T0M1': 'mean_wave',
            'T01': 'mean_wave',
            'FP': 'mean_wave',
            'DIR': 'mean_wave',
            'SPR': 'mean_wave',
            'DP': 'mean_wave',
            'HIG': 'mean_wave',
            'MXE': 'mean_wave',
            'MXES': 'mean_wave',
            'MXH': 'mean_wave',
            'MXHC': 'mean_wave',
            'SDMH': 'mean_wave',
            'SDMHC': 'mean_wave',
            'WBT': 'mean_wave',
            'TP': 'mean_wave',
            'EF': 'spectral_parameters',
            'TH1M': 'spectral_parameters',
            'STH1M': 'spectral_parameters',
            'TH2M': 'spectral_parameters',
            'STH2M': 'spectral_parameters',
            'WN': 'spectral_parameters',
            'PHS': 'partition_params',
            'PTP': 'partition_params',
            'PLP': 'partition_params',
            'PDIR': 'partition_params',
            'PSPR': 'partition_params',
            'PWS': 'partition_params',
            'PDP': 'partition_params',
            'PQP': 'partition_params',
            'PPE': 'partition_params',
            'PGW': 'partition_params',
            'PSW': 'partition_params',
            'PTM10': 'partition_params',
            'PT01': 'partition_params',
            'PT02': 'partition_params',
            'PEP': 'partition_params',
            'TWS': 'partition_params',
            'PNR': 'partition_params',
            'UST': 'atmosphere_waves',
            'CHA': 'atmosphere_waves',
            'CGE': 'atmosphere_waves',
            'FAW': 'atmosphere_waves',
            'TAW': 'atmosphere_waves',
            'TWA': 'atmosphere_waves',
            'WCC': 'atmosphere_waves',
            'WCF': 'atmosphere_waves',
            'WCH': 'atmosphere_waves',
            'WCM': 'atmosphere_waves',
            'FWS': 'atmosphere_waves',
            'SXY': 'wave_ocean',
            'TWO': 'wave_ocean',
            'BHD': 'wave_ocean',
            'FOC': 'wave_ocean',
            'TUS': 'wave_ocean',
            'USS': 'wave_ocean',
            'P2S': 'wave_ocean',
            'USF': 'wave_ocean',
            'P2L': 'wave_ocean',
            'TWI': 'wave_ocean',
            'FIC': 'wave_ocean',
            'USP': 'wave_ocean',
            'ABR': 'wave_bottom',
            'UBR': 'wave_bottom',
            'BED': 'wave_bottom',
            'FBB': 'wave_bottom',
            'TBB': 'wave_bottom',
            'MSS': 'spectrum_parameters',
            'MSC': 'spectrum_parameters',
            'WL02': 'spectrum_parameters',
            'AXT': 'spectrum_parameters',
            'AYT': 'spectrum_parameters',
            'AXY': 'spectrum_parameters',
            'DTD': 'numerical_diagnostics',
            'FC': 'numerical_diagnostics',
            'CFX': 'numerical_diagnostics',
            'CFD': 'numerical_diagnostics',
            'CFK': 'numerical_diagnostics',
            'U1': 'user_defined',
            'U2': 'user_defined'
        }
        parameters = self._output_fields_section
        end_not_found = True
        while end_not_found:
            data = self._parse_next_line()
            if '/' in data[0]:
                end_not_found = False
            elif 'TYPE%FIELD%LIST' in data[0].strip():
                for i in range(2, len(data)):
                    value = data[i].replace("'", "")
                    value = value.replace('"', '')
                    if value in expected_fields:
                        parameters.group(expected_fields[value]).parameter(value).value = True
            elif 'TYPE%POINT%FILE' in data[0].strip():
                self._read_output_point_file(data[2])
            else:
                raise ValueError(f'Unrecognized entry in output type namelist {data[0]}')

    def _read_output_date_namelist(self):
        """Read the name list containing the output date parameters."""
        output_fields = self._output_fields_section
        time_parameters = output_fields.group('time_parameters')
        model_parameters = self._global_values
        output = model_parameters.group('output')
        end_not_found = True
        while end_not_found:
            data = self._parse_next_line()
            if '/' in data[0]:
                end_not_found = False
            elif 'DATE%FIELD%START' in data[0].strip():
                time_parameters.parameter('output_start_date').value = get_date_time_string(data[2], data[3])
            elif 'DATE%FIELD%STRIDE' in data[0].strip():
                time_parameters.parameter('output_second_interval').value = int(data[2][1:-1])
            elif 'DATE%FIELD%STOP' in data[0].strip():
                time_parameters.parameter('output_end_date').value = get_date_time_string(data[2], data[3])
            elif 'DATE%FIELD' in data[0].strip():
                time_parameters.parameter('output_start_date').value = get_date_time_string(data[2], data[3])
                time_parameters.parameter('output_second_interval').value = int(data[4][1:-1])
                time_parameters.parameter('output_end_date').value = get_date_time_string(data[5], data[6])
            elif 'DATE%POINT%START' in data[0].strip():
                output.parameter('output_start_date_point').value = get_date_time_string(data[2], data[3])
            elif 'DATE%POINT%STRIDE' in data[0].strip():
                output.parameter('output_second_interval_point').value = int(data[2][1:-1])
                output.parameter('OType2').value = 1 if int(data[2][1:-1]) else 0
            elif 'DATE%POINT%STOP' in data[0].strip():
                output.parameter('output_end_date_point').value = get_date_time_string(data[2], data[3])
            elif 'DATE%POINT' in data[0].strip():
                output.parameter('output_start_date_point').value = get_date_time_string(data[2], data[3])
                output.parameter('output_second_interval_point').value = int(data[4][1:-1])
                output.parameter('OType2').value = 1 if int(data[4][1:-1]) else 0
                output.parameter('output_end_date_point').value = get_date_time_string(data[5], data[6])
            elif 'DATE%TRACK%START' in data[0].strip():
                output.parameter('output_start_date_track').value = get_date_time_string(data[2], data[3])
            elif 'DATE%TRACK%STRIDE' in data[0].strip():
                output.parameter('output_second_interval_track').value = int(data[2][1:-1])
                output.parameter('OType3').value = 1 if int(data[2][1:-1]) else 0
            elif 'DATE%TRACK%STOP' in data[0].strip():
                output.parameter('output_end_date_track').value = get_date_time_string(data[2], data[3])
            elif 'DATE%TRACK' in data[0].strip():
                output.parameter('output_start_date_track').value = get_date_time_string(data[2], data[3])
                output.parameter('output_second_interval_track').value = int(data[4][1:-1])
                output.parameter('OType3').value = 1 if int(data[4][1:-1]) else 0
                output.parameter('output_end_date_track').value = get_date_time_string(data[5], data[6])
            elif 'DATE%RESTART%START' in data[0].strip():
                output.parameter('output_start_date_restart').value = get_date_time_string(data[2], data[3])
            elif 'DATE%RESTART%STRIDE' in data[0].strip():
                output.parameter('output_second_interval_restart').value = int(data[2][1:-1])
                output.parameter('OType4').value = 1 if int(data[2][1:-1]) else 0
            elif 'DATE%RESTART%STOP' in data[0].strip():
                output.parameter('output_end_date_restart').value = get_date_time_string(data[2], data[3])
            elif 'DATE%RESTART' in data[0].strip():
                output.parameter('output_start_date_restart').value = get_date_time_string(data[2], data[3])
                output.parameter('output_second_interval_restart').value = int(data[4][1:-1])
                output.parameter('OType4').value = 1 if int(data[4][1:-1]) else 0
                output.parameter('output_end_date_restart').value = get_date_time_string(data[5], data[6])
            elif 'DATE%BOUNDARY%START' in data[0].strip():
                output.parameter('output_start_date_boundary').value = get_date_time_string(data[2], data[3])
            elif 'DATE%BOUNDARY%STRIDE' in data[0].strip():
                output.parameter('output_second_interval_boundary').value = int(data[2][1:-1])
                output.parameter('OType5').value = 1 if int(data[2][1:-1]) else 0
            elif 'DATE%BOUNDARY%STOP' in data[0].strip():
                output.parameter('output_end_date_boundary').value = get_date_time_string(data[2], data[3])
            elif 'DATE%BOUNDARY' in data[0].strip():
                output.parameter('output_start_date_boundary').value = get_date_time_string(data[2], data[3])
                output.parameter('output_second_interval_boundary').value = int(data[4][1:-1])
                output.parameter('OType5').value = 1 if int(data[4][1:-1]) else 0
                output.parameter('output_end_date_boundary').value = get_date_time_string(data[5], data[6])
            elif 'DATE%PARTITION%START' in data[0].strip():
                output.parameter('output_start_date_separated').value = get_date_time_string(data[2], data[3])
            elif 'DATE%PARTITION%STRIDE' in data[0].strip():
                output.parameter('output_second_interval_separated').value = int(data[2][1:-1])
                output.parameter('OType6').value = 1 if int(data[2][1:-1]) else 0
            elif 'DATE%PARTITION%STOP' in data[0].strip():
                output.parameter('output_end_date_separated').value = get_date_time_string(data[2], data[3])
            elif 'DATE%PARTITION' in data[0].strip():
                output.parameter('output_start_date_separated').value = get_date_time_string(data[2], data[3])
                output.parameter('output_second_interval_separated').value = int(data[4][1:-1])
                output.parameter('OType6').value = 1 if int(data[4][1:-1]) else 0
                output.parameter('output_end_date_separated').value = get_date_time_string(data[5], data[6])
            elif 'DATE%COUPLING%START' in data[0].strip():
                output.parameter('output_start_date_coupling').value = get_date_time_string(data[2], data[3])
            elif 'DATE%COUPLING%STRIDE' in data[0].strip():
                output.parameter('output_second_interval_coupling').value = int(data[2][1:-1])
                output.parameter('OType7').value = 1 if int(data[2][1:-1]) else 0
            elif 'DATE%COUPLING%STOP' in data[0].strip():
                output.parameter('output_end_date_coupling').value = get_date_time_string(data[2], data[3])
            elif 'DATE%COUPLING' in data[0].strip():
                output.parameter('output_start_date_coupling').value = get_date_time_string(data[2], data[3])
                output.parameter('output_second_interval_coupling').value = int(data[4][1:-1])
                output.parameter('OType7').value = 1 if int(data[4][1:-1]) else 0
                output.parameter('output_end_date_coupling').value = get_date_time_string(data[5], data[6])
            else:
                raise ValueError(f'Unrecognized entry in output type namelist {data[0]}')

    def _read_output_path_namelist(self):
        """Read the name list containing the output path parameters."""
        # This is currently unimplemented
        end_not_found = True
        while end_not_found:
            data = self._parse_next_line()
            if '/' in data[0]:
                end_not_found = False

    def _read_homogenous_count_namelist(self):
        """Read the name list containing the homogenous count parameters."""
        end_not_found = True
        while end_not_found:
            data = self._parse_next_line()
            if '/' in data[0]:
                end_not_found = False
            elif 'HOMOG_COUNT%N_IC1' in data[0].strip():
                self._num_ic1 = int(data[2])
            elif 'HOMOG_COUNT%N_IC2' in data[0].strip():
                self._num_ic2 = int(data[2])
            elif 'HOMOG_COUNT%N_IC3' in data[0].strip():
                self._num_ic3 = int(data[2])
            elif 'HOMOG_COUNT%N_IC4' in data[0].strip():
                self._num_ic4 = int(data[2])
            elif 'HOMOG_COUNT%N_IC5' in data[0].strip():
                self._num_ic5 = int(data[2])
            elif 'HOMOG_COUNT%N_MDN' in data[0].strip():
                self._num_mdn = int(data[2])
            elif 'HOMOG_COUNT%N_MTH' in data[0].strip():
                self._num_mth = int(data[2])
            elif 'HOMOG_COUNT%N_MVS' in data[0].strip():
                self._num_mvs = int(data[2])
            elif 'HOMOG_COUNT%N_LEV' in data[0].strip():
                self._num_lev = int(data[2])
            elif 'HOMOG_COUNT%N_CUR' in data[0].strip():
                self._num_cur = int(data[2])
            elif 'HOMOG_COUNT%N_WND' in data[0].strip():
                self._num_wnd = int(data[2])
            else:
                raise ValueError(f'Unrecognized entry in homogeneous count namelist {data[0]}')

    def _read_homogenous_input_namelist(self):
        """Read the name list containing the homogenous input parameters."""
        model_parameters = self._global_values
        ice_and_mud = model_parameters.group('ice_and_mud')
        run_control = model_parameters.group('run_control')
        homogeneous_dict = {}
        end_not_found = True

        while end_not_found:
            data = self._parse_next_line()
            if '/' in data[0]:
                end_not_found = False
                continue

            # Normalize line if it's a HOMOG_INPUT line
            if 'HOMOG_INPUT' in data[0].strip():
                # Join tokens into a single string and split on '='
                line = ' '.join(data).strip()
                lhs, rhs = map(str.strip, line.split('=', 1))

                # Special case for DATE field
                if '%DATE' in lhs:
                    rhs_clean = rhs.strip("'")
                    date_part, time_part = rhs_clean.split()
                    data = [lhs, '=', date_part, time_part]
                else:
                    data = [lhs, '=', rhs.strip()]

                index = int(re.search(r"\((.*?)\)", data[0].strip()).group(1))
                if index not in homogeneous_dict:
                    ival = -99999999999
                    homogeneous_dict[index] = {'name': '', 'date': ival, 'value1': ival, 'value2': ival, 'value3': ival}

                if '%NAME' in data[0].strip():
                    homogeneous_dict[index]['name'] = data[2][1:-1].upper()  # strip quotes
                elif '%DATE' in data[0].strip():
                    date_time_val = self._get_date_time(data[2], data[3])
                    homogeneous_dict[index]['date'] = date_time_val
                elif '%VALUE1' in data[0].strip():
                    homogeneous_dict[index]['value1'] = data[2]
                elif '%VALUE2' in data[0].strip():
                    homogeneous_dict[index]['value2'] = data[2]
                elif '%VALUE3' in data[0].strip():
                    homogeneous_dict[index]['value3'] = data[2]

        # Convert the dictionary into individual lists

        # IC1:
        ic1 = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'IC1':
                ic1.append([values_dict['date'], values_dict['value1']])
        if len(ic1) > 0:
            ice_and_mud.parameter('ice_param_1').value = ic1

        # IC2:
        ic2 = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'IC2':
                ic2.append([values_dict['date'], values_dict['value1']])
        if len(ic2) > 0:
            ice_and_mud.parameter('ice_param_2').value = ic2

        # IC3:
        ic3 = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'IC3':
                ic3.append([values_dict['date'], values_dict['value1']])
        if len(ic3) > 0:
            ice_and_mud.parameter('ice_param_3').value = ic3

        # IC4:
        ic4 = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'IC4':
                ic4.append([values_dict['date'], values_dict['value1']])
        if len(ic4) > 0:
            ice_and_mud.parameter('ice_param_4').value = ic4

        # IC5:
        ic5 = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'IC5':
                ic5.append([values_dict['date'], values_dict['value1']])
        if len(ic5) > 0:
            ice_and_mud.parameter('ice_param_5').value = ic5

        # MDN:
        mdn = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'MDN':
                mdn.append([values_dict['date'], values_dict['value1']])
        if len(mdn) > 0:
            ice_and_mud.parameter('mud_density_table').value = mdn

        # MTH:
        mth = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'MTH':
                mth.append([values_dict['date'], values_dict['value1']])
        if len(mth) > 0:
            ice_and_mud.parameter('mud_thickness_table').value = mth

        # MVS:
        mvs = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'MVS':
                mvs.append([values_dict['date'], values_dict['value1']])
        if len(mvs) > 0:
            ice_and_mud.parameter('mud_viscosity_table').value = mvs

        # LEV:
        lev = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'LEV':
                lev.append([values_dict['date'], values_dict['value1']])
        if len(lev) > 0:
            run_control.parameter('water_level_homogeneous').value = lev

        # CUR:
        cur = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'CUR':
                cur.append([values_dict['date'], values_dict['value1'], values_dict['value2']])
        if len(cur) > 0:
            run_control.parameter('currents_homogeneous').value = cur

        # WND:
        wnd = []
        for _, values_dict in homogeneous_dict.items():
            if values_dict['name'] == 'WND':
                wnd.append([values_dict['date'], values_dict['value1'], values_dict['value2'], values_dict['value3']])
        if len(wnd) > 0:
            run_control.parameter('winds_homogeneous').value = wnd

    def _get_date_time(self, date_part, time_part):
        """Create a correctly formatted datetime string from two string parts.

        Args:
            date_part (:obj:`str`): The part of the string containing the date, ex:  20080225.
            time_part (:obj:`str`): The part of the string containing the time, ex:  105500.

        Returns:
            The formatted string
        """
        year = date_part[0:4]
        month = date_part[4:6]
        day = date_part[6:8]

        hour = time_part[0:2]
        minute = time_part[2:4]
        second = time_part[4:6]

        dt = f'{year}-{month}-{day} {hour}:{minute}:{second}'
        return dt

    def _get_use_flag(self, val_string):
        """Convert the string from the namelist file to a flag value.

        Args:
            val_string (:obj:`str`): string to be converted to a flag value.

        Returns:
            (:obj:`int`): flag value
        """
        if val_string.upper() == 'F':
            flag_value = 'F: no forcing'
        elif val_string.upper() == 'T':
            flag_value = 'T: external forcing file'
        elif val_string.upper() == 'H':
            flag_value = 'H: homogeneous forcing input'
        elif val_string.upper() == 'C':
            flag_value = 'C: coupled forcing field'
        else:
            raise ValueError(f'Unrecognized flag value {val_string}')
        return flag_value

    def _read_output_point_file(self, point_file_name):
        """Read the output point locations.

        Args:
            point_file_name (:obj:`str`): name of the output point file.
        """
        with open(os.path.join(os.path.dirname(self._filename), point_file_name.replace("'", ''))) as fp:
            lines = fp.readlines()
            for line in lines:
                data = shlex.split(line, posix=False)
                pt_x = float(data[0])
                pt_y = float(data[1])
                pt_name = data[2].strip('"\'').strip()
                pt = Point(pt_x, pt_y)
                pt.id = len(self._output_points) + 1
                self._output_points.append(pt)
                self._output_point_names.append(pt_name)

    def _create_output_points_coverage(self, sim):
        """Create the output points coverage from any output points read in the Type 2 section.

        Args:
            sim (:obj:`Simulation`):  The simulation.
        """
        if not self._output_points:
            return

        section = get_model().point_parameters
        section.group('output_point').is_active = True
        builder = CoverageComponentBuilder(OutputPointsComponent, 'Output Points')
        data = cast(CoverageBaseData, builder.data)
        values = []
        for name in self._output_point_names:
            section.group('output_point').parameter('name').value = name
            values.append(section.extract_values())
        types = ['output_point'] * len(values)
        comp_ids = data.add_features(TargetType.point, values, types)
        for point, comp_id in zip(self._output_points, comp_ids):
            x, y, z, feature_id = point.x, point.y, point.z, point.id
            builder.add_point(x, y, z, comp_id, feature_id)

        coverage, component, keywords = builder.build()

        coverage.projection = Projection(wkt=GEOGRAPHIC_WKT if self._is_geographic else LOCAL_METERS_WKT)

        cov_comp = Component(
            main_file=component.main_file,
            model_name='WaveWatch3',
            unique_name='OutputPointsComponent',
            comp_uuid=component.uuid
        )
        self._build_data['output_pts_cov_comp'] = cov_comp
        self._build_data['output_pts_cov'] = coverage
        self._build_data['output_pts_keywords'] = keywords
        self._build_data['links'].append((sim.uuid, coverage.uuid))

    def _build_inbound_nodestrings(self, builder):
        """Build nodestrings for the inbound arc type."""
        if not self._inbound_points:
            return

        # Sort keys to preserve order
        sorted_points = sorted(self._inbound_points.items())
        nodestring = []

        for i, (_key, (pt_id, _, _cont_flag)) in enumerate(sorted_points):
            # Always add first point
            if i == 0:
                nodestring.append(pt_id - 1)
            else:
                prev_pt_id = sorted_points[i - 1][1][0]
                # Check for gap
                if pt_id - prev_pt_id > 1 and sorted_points[i - 1][1][2]:  # prev.cont_flag == True
                    # Fill in missing intermediate points
                    for p in range(prev_pt_id + 1, pt_id):
                        nodestring.append(p - 1)
                nodestring.append(pt_id - 1)

        # Draw the nodestring
        section = get_model().arc_parameters
        section.group('input').is_active = True
        values = section.extract_values()
        comp_id = builder.data.add_feature(TargetType.arc, values, 'input')
        builder.add_node_string(nodestring, comp_id)

    def _build_lateral_nodestrings(self, builder):
        """Build nodestrings for the lateral arc type."""
        unprocessed_nodes = [point[0][1] for point in self._lateral_points]  # Indexes of all nodes used by lateral arcs
        all_nodes = set(unprocessed_nodes)
        ugrid = self._build_data['co_ugrid'].ugrid
        section = get_model().arc_parameters
        section.group('lateral').is_active = True
        values = section.extract_values()
        for node in unprocessed_nodes:
            for neighbor in ugrid.get_point_adjacent_points(node):
                if neighbor > node and neighbor in all_nodes:
                    comp_id = builder.data.add_feature(TargetType.arc, values, 'lateral')
                    builder.add_node_string([node, neighbor], comp_id)

    def _build_bc_coverage(self, builder):
        """Build the BC Coverage."""
        coverage, component, keywords = builder.build()
        if len(coverage.arcs) < 1:
            return
        cov_comp = Component(
            main_file=component.main_file,
            model_name='WaveWatch3',
            unique_name='Bc_Component',
            comp_uuid=component.uuid
        )
        self._build_data['bc_cov'] = coverage
        self._build_data['bc_cov_comp'] = cov_comp
        self._build_data['bc_cov_keywords'] = keywords

        sim = self._build_data['sim']
        self._build_data['links'].append((sim.uuid, coverage.uuid))

    def _build_xms_data(self):
        """Add all the imported data to the xmsapi Query to send back to SMS."""
        # Add the simulation
        sim = Simulation(model='WaveWatch3', sim_uuid=str(uuid.uuid4()))
        sim.name = os.path.basename(os.path.dirname(self._filename)) if self._filename else 'Sim'
        sim_comp = Component(
            name='',
            comp_uuid=self._sim_comp_uuid,
            main_file=self._sim_data._filename,
            model_name='WaveWatch3',
            unique_name='SimComponent',
            locked=False
        )
        # self._query.add_simulation(sim, [sim_comp])
        self._build_data['sim'] = sim
        self._build_data['sim_comp'] = sim_comp

        # Link the mesh to the simulation
        if self._build_data['do_ugrid']:
            self._build_data['links'].append((sim.uuid, self._build_data['do_ugrid'].uuid))

        # Link the spectral coverages to the simulation
        for spectral_cov in self._build_data['spectral_coverages']:
            self._build_data['links'].append((sim.uuid, spectral_cov.m_cov.uuid))

        # Add the output points coverage
        self._create_output_points_coverage(sim)

        # Add the BC coverage
        builder = CoverageComponentBuilder(BcComponent, 'Boundary Conditions', self._build_data['co_ugrid'])
        self._build_inbound_nodestrings(builder)
        self._build_lateral_nodestrings(builder)
        self._build_bc_coverage(builder)

    def _add_build_data(self):
        """Adds build data at the end of reading everything."""
        if not self._query:
            return

        # Add the main simuation
        self._query.add_simulation(self._build_data['sim'], components=[self._build_data['sim_comp']])

        # Add the UGrid
        if self._build_data['do_ugrid']:
            self._query.add_ugrid(self._build_data['do_ugrid'])

        # Add the coverage(s)
        if self._build_data['output_pts_cov'] is not None and self._build_data['output_pts_cov_comp'] is not None:
            self._query.add_coverage(
                self._build_data['output_pts_cov'],
                model_name='WaveWatch3',
                coverage_type='Output Points',
                components=[self._build_data['output_pts_cov_comp']],
                component_keywords=self._build_data['output_pts_keywords']
            )
        if self._build_data['bc_cov'] is not None and self._build_data['bc_cov_comp'] is not None:
            self._query.add_coverage(
                self._build_data['bc_cov'],
                model_name='WaveWatch3',
                coverage_type='Boundary Conditions',
                components=[self._build_data['bc_cov_comp']],
                component_keywords=self._build_data['bc_cov_keywords']
            )
        for spectral_cov in self._build_data['spectral_coverages']:
            self._query.add_coverage(spectral_cov)

        # Link items
        for taker_uuid, taken_uuid in self._build_data['links']:
            self._query.link_item(taker_uuid=taker_uuid, taken_uuid=taken_uuid)

    def send(self):
        """Send all the imported data back to SMS."""
        if self._query:
            self._build_xms_data()
            self._add_build_data()
            self._query.send()

    def get_netcdf_datasets_files(self, cur_dirpath):
        """Finds any netcdf dataset type files to use to read onto the mesh.

        Looks in the same directory only, not subdirectories.

        Args:
            cur_dirpath (:obj:`str`):  Path to look in.

        Returns:
            (:obj:`list`):  List of possible files.
        """
        files = set()
        for root, _, filenames in os.walk(cur_dirpath):
            for filename in filenames:
                if re.match(r'ww3.\d{4,6}.nc', filename) and root == cur_dirpath:  # noqa: W605
                    # Only want this in the top directory
                    files.add(filename)
        return list(files)

    def read(self):
        """Top-level entry point for the WaveWatch3 shell_nml input file reader."""
        try:
            self._logger.info('Parsing ASCII text from file...')
            with open(self._filename, 'r', buffering=READ_BUFFER_SIZE) as f:
                self._lines = f.readlines()

            self._sim_data = SimData(os.path.join(create_component_folder(self._sim_comp_uuid), SIM_DATA_MAINFILE))
            self._read_shell_nml_file()

            # Next, try reading the grid namelist file:
            grid_filename = os.path.join(
                os.path.dirname(self._filename),
                os.path.basename(self._filename).replace('_shel', '_grid')
            )
            if os.path.isfile(grid_filename):
                self._logger.info('Reading grid namelist....')
                geom_uuid = ''
                grid_reader = GridNmlReader(filename=grid_filename, sim_data=self._sim_data)
                grid_reader.read()
                self._inbound_points = grid_reader.get_inbound_points()

                # Next, try reading the namelists namelist file
                namelists_nml_file = os.path.join(
                    os.path.dirname(grid_filename),
                    grid_reader._grid_namelist.lstrip("'").rstrip("'")
                )
                if os.path.isfile(namelists_nml_file):
                    self._logger.info('Reading namelists namelist....')
                    namelists_reader = NamelistsNmlReader(filename=namelists_nml_file, sim_data=self._sim_data)
                    namelists_reader.read()

                # Next, try reading the mesh file
                msh_file = os.path.join(
                    os.path.dirname(grid_filename),
                    grid_reader._gmsh_filename.lstrip("'").rstrip("'")
                )
                if os.path.isfile(msh_file):
                    self._logger.info('Reading mesh....')
                    mesh_reader = GmshReader(
                        filename=msh_file, grid_name=grid_reader._grid_name.lstrip("'").rstrip("'")
                    )
                    mesh_reader.read()
                    self._build_data['do_ugrid'] = mesh_reader.do_ugrid
                    self._build_data['co_ugrid'] = mesh_reader.co_ugrid
                    geom_uuid = mesh_reader.do_ugrid.uuid
                    self._global_values.group('miscellaneous').parameter('domain_uuid').value = geom_uuid

                    # Try and read netCDF datasets from a solution if there
                    mesh_netcdf_files = self.get_netcdf_datasets_files(os.path.dirname(self._filename))
                    for mesh_netcdf_file in mesh_netcdf_files:
                        dt = datetime.datetime(1990, 1, 1)
                        output_types = get_output_field_type_list(self._output_fields_section)
                        netcdf_file = os.path.join(os.path.dirname(msh_file), mesh_netcdf_file)
                        temp_dir = self._query.xms_temp_directory if self._query else '.'
                        netcdf_reader = NetcdfDatasetReader(
                            filename=netcdf_file,
                            reftime=dt,
                            query=self._query,
                            temp_dir=temp_dir,
                            output_types=output_types,
                            geom_uuid=mesh_reader.do_ugrid.uuid
                        )
                        netcdf_reader.read()

                # Next, try the meshbnd.msh file for lateral BC arcs
                meshbnd_file = os.path.join(os.path.dirname(grid_filename), 'meshbnd.msh')
                if os.path.isfile(meshbnd_file):
                    self._logger.info('Reading meshbnd.msh....')
                    meshbnd_reader = GmshReader(filename=meshbnd_file, grid_name='meshbnd')
                    self._lateral_points = meshbnd_reader.read(meshbnd=True)

                # Next, try reading the OUNF namelist file
                ounf_filename = os.path.join(
                    os.path.dirname(self._filename),
                    os.path.basename(self._filename).replace('_shel', '_ounf')
                )
                if os.path.isfile(ounf_filename):
                    self._logger.info('Reading ounf namelist....')
                    ounf_reader = OunfNmlReader(filename=ounf_filename, sim_data=self._sim_data)
                    ounf_reader.read()

                # Next, try reading the OUNP namelist file
                ounp_filename = os.path.join(
                    os.path.dirname(self._filename),
                    os.path.basename(self._filename).replace('_shel', '_ounp')
                )
                if os.path.isfile(ounp_filename):
                    self._logger.info('Reading ounp namelist....')
                    ounp_reader = OunpNmlReader(filename=ounp_filename, sim_data=self._sim_data)
                    ounp_reader.read()

                # Next, try reading the BOUNC namelist file
                bounc_filename = os.path.join(
                    os.path.dirname(self._filename),
                    os.path.basename(self._filename).replace('_shel', '_bounc')
                )
                if os.path.isfile(bounc_filename):
                    self._logger.info('Reading bounc namelist....')
                    bounc_reader = BouncNmlReader(filename=bounc_filename, sim_data=self._sim_data)
                    bounc_reader.read()

                    # Try to read the spectral list file
                    bound_file = self._global_values.group('run_control').parameter('bound_file').value
                    spec_list_filename = os.path.join(os.path.dirname(self._filename), bound_file)
                    if os.path.isfile(spec_list_filename):
                        self._logger.info('Reading spectral spec.list....')
                        spec_list_reader = WW3SpecListReader(filename=spec_list_filename)
                        spec_list_reader.read()
                        self._build_data['spectral_coverages'] = spec_list_reader.spectral_coverages

                # Next, try reading any PRNC namelist files
                prnc_list = glob.glob(os.path.join(os.path.dirname(self._filename), 'ww3_prnc*.nml'))
                for prnc_file in prnc_list:
                    self._logger.info(f'Reading prnc file {prnc_file}...')
                    prnc_reader = PrncNmlReader(filename=prnc_file, sim_data=self._sim_data)
                    prnc_reader.read()
                    prnc_reader.read_netcdf_dataset(self._query, geom_uuid)

                self._logger.info('Committing changes....')
            else:
                self._logger.warning(f'Could not find grid namelist file {grid_filename}')

            self._sim_data.commit()
            self._logger.info('Finished!')
        except Exception:
            self._logger.exception(f'Unexpected error in shell nml preprocessor file (line {self._current_line + 1}).')
            raise
