"""Class to write a WaveWatch3 shell namelist file."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules
import datetime
from io import StringIO
import logging
import shutil

# 2. Third party modules

# 3. Aquaveo modules
from xms.data_objects.parameters import FilterLocation
from xms.guipy.data.target_type import TargetType
from xms.guipy.time_format import string_to_datetime

# 4. Local modules
from xms.wavewatch3.data.model import get_model
from xms.wavewatch3.gui.gui_util import get_formatted_date_string, get_output_field_type_string


class WW3ShellNamelistWriter:
    """Class to write a WaveWatch3 shell namelist file."""
    def __init__(self, xms_data):
        """Constructor.

        Args:
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
        """
        self._ss = StringIO()
        self._logger = logging.getLogger('xms.wavewatch3')
        self._xms_data = xms_data

    # def _start_grid_process(self):
    #     """Start the parallel process for running the ww3_grid.nml exporter."""
    #     self._logger.info('Starting shell namelist export in parallel process...')
    #     self._grid_process = subprocess.Popen([
    #         sys.executable,  # Python executable
    #         os.path.normpath(os.path.join(os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points',
    #                                       'grid_parallel.py')),  # Path to script
    #         os.path.normpath(self._xms_data.sim_data._filename),  # Simulation component main file
    #         os.path.normpath(self._xms_data.do_ugrid.cogrid_file),  # CoGrid file
    #         self._xms_data.do_ugrid.name,  # Name of the domain mesh
    #     ], env=os.environ)

    def _write_shell_namelist_file(self):
        """Write the ww3_shel.nml file."""
        file_w_path = "ww3_shel.nml"
        self._write_shell_header()
        self._write_domain_namelist()
        self._write_input_namelist()
        self._write_define_output_point_types_namelist()
        self._write_output_dates_namelist()
        self._write_homogeneous_namelists()
        self._write_end_comments()
        self._flush(file_w_path)

    def _write_shell_header(self):
        """Writes out the header to the shell input file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - ww3_shel.nml - single-grid model                     !\n'
            '! -------------------------------------------------------------------- !\n\n'
        )

    def _write_domain_namelist(self):
        """Writes out the output server mode section."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! Define top-level model parameters via DOMAIN_NML namelist\n'
            '!\n'
            '! * IOSTYP defines the output server mode for parallel implementation.\n'
            '!             0 : No data server processes, direct access output from\n'
            '!                 each process (requires true parallel file system).\n'
            '!             1 : No data server process. All output for each type \n'
            '!                 performed by process that performs computations too.\n'
            '!             2 : Last process is reserved for all output, and does no\n'
            '!                 computing.\n'
            '!             3 : Multiple dedicated output processes.\n'
            '!\n'
            '! * namelist must be terminated with /\n'
            '! * definitions & defaults:\n'
            '!     DOMAIN%IOSTYP =  1                 ! Output server type\n'
            '!     DOMAIN%START  = \'19680606 000000\'  ! Start date for the entire model \n'
            '!     DOMAIN%STOP   = \'19680607 000000\'  ! Stop date for the entire model\n'
            '! -------------------------------------------------------------------- !\n'
        )
        self._ss.write('&DOMAIN_NML\n')
        output = self._xms_data.sim_data_model_control.group('output')
        # IOSType is currently hidden in the model control
        iostype = output.parameter('IOSType').value if output.parameter('IOSType') else 0
        if iostype != 0:
            self._ss.write('     DOMAIN%IOSTYP = ' + str(iostype) + '\n')
        run_control = self._xms_data.sim_data_model_control.group('run_control')
        start_date = run_control.parameter('starting_date').value
        start_date = datetime.datetime.strptime(start_date.replace('T', ' '), '%Y-%m-%d %H:%M:%S')
        date_str = f"'{get_formatted_date_string(start_date)}'"
        self._ss.write(f'     DOMAIN%START  = {date_str}\n')
        end_date = run_control.parameter('end_date').value
        end_date = datetime.datetime.strptime(end_date.replace('T', ' '), '%Y-%m-%d %H:%M:%S')
        date_str = f"'{get_formatted_date_string(end_date)}'"
        self._ss.write(f'     DOMAIN%STOP   = {date_str}\n')
        self._ss.write('/\n\n\n\n')

    def _write_input_namelist(self):
        """Writes out the compiler switches and flags section."""
        self._ss.write(
            "! -------------------------------------------------------------------- !\n"
            "! Define each forcing via the INPUT_NML namelist\n"
            "!\n"
            "! * The FORCING flag can be  : 'F' for \"no forcing\"\n"
            "!                              'T' for \"external forcing file\"\n"
            "!                              'H' for \"homogeneous forcing input\"\n"
            "!                              'C' for \"coupled forcing field\"\n"
            "!\n"
            "! * homogeneous forcing is not available for ICE_CONC\n"
            "!\n"
            "! * The ASSIM flag can :  'F' for \"no forcing\"\n"
            "!                         'T' for \"external forcing file\"\n"
            "!\n"
            "! * namelist must be terminated with /\n"
            "! * definitions & defaults:\n"
            "!     INPUT%FORCING%WATER_LEVELS  = 'F'\n"
            "!     INPUT%FORCING%CURRENTS      = 'F'\n"
            "!     INPUT%FORCING%WINDS         = 'F'\n"
            "!     INPUT%FORCING%ATM_MOMENTUM  = 'F'\n"
            "!     INPUT%FORCING%AIR_DENSITY   = 'F'\n"
            "!     INPUT%FORCING%ICE_CONC      = 'F'\n"
            "!     INPUT%FORCING%ICE_PARAM1    = 'F'\n"
            "!     INPUT%FORCING%ICE_PARAM2    = 'F'\n"
            "!     INPUT%FORCING%ICE_PARAM3    = 'F'\n"
            "!     INPUT%FORCING%ICE_PARAM4    = 'F'\n"
            "!     INPUT%FORCING%ICE_PARAM5    = 'F'\n"
            "!     INPUT%FORCING%MUD_DENSITY   = 'F'\n"
            "!     INPUT%FORCING%MUD_THICKNESS = 'F'\n"
            "!     INPUT%FORCING%MUD_VISCOSITY = 'F'\n"
            "!     INPUT%ASSIM%MEAN            = 'F'\n"
            "!     INPUT%ASSIM%SPEC1D          = 'F'\n"
            "!     INPUT%ASSIM%SPEC2D          = 'F'\n"
            "! -------------------------------------------------------------------- !\n"
        )
        self._ss.write('&INPUT_NML\n')
        run_control = self._xms_data.sim_data_model_control.group('run_control')
        ice_and_mud = self._xms_data.sim_data_model_control.group('ice_and_mud')
        if run_control.parameter('water_levels').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%WATER_LEVELS  = {self._use_flag(run_control.parameter('water_levels').value)}\n"
            )
        if run_control.parameter('currents').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%CURRENTS      = {self._use_flag(run_control.parameter('currents').value)}\n"
            )
        if run_control.parameter('winds').value != 'F: no forcing':
            self._ss.write(f"  INPUT%FORCING%WINDS         = {self._use_flag(run_control.parameter('winds').value)}\n")
        #  This option is currently hidden in the Model Control
        if run_control.parameter('define_atm_momentum').value != 'F: no forcing':
            param = 'define_atm_momentum'
            self._ss.write(f"  INPUT%FORCING%ATM_MOMENTUM  = {self._use_flag(run_control.parameter(param).value)}\n")
        if run_control.parameter('air_density').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%AIR_DENSITY   = {self._use_flag(run_control.parameter('air_density').value)}\n"
            )
        if ice_and_mud.parameter('concentration').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%ICE_CONC      = {self._use_flag(ice_and_mud.parameter('concentration').value)}\n"
            )
        if ice_and_mud.parameter('param_1').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%ICE_PARAM1    = {self._use_flag(ice_and_mud.parameter('param_1').value)}\n"
            )
        if ice_and_mud.parameter('param_2').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%ICE_PARAM2    = {self._use_flag(ice_and_mud.parameter('param_2').value)}\n"
            )
        if ice_and_mud.parameter('param_3').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%ICE_PARAM3    = {self._use_flag(ice_and_mud.parameter('param_3').value)}\n"
            )
        if ice_and_mud.parameter('param_4').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%ICE_PARAM4    = {self._use_flag(ice_and_mud.parameter('param_4').value)}\n"
            )
        if ice_and_mud.parameter('param_5').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%ICE_PARAM5    = {self._use_flag(ice_and_mud.parameter('param_5').value)}\n"
            )
        if ice_and_mud.parameter('mud_density').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%MUD_DENSITY   = {self._use_flag(ice_and_mud.parameter('mud_density').value)}\n"
            )
        if ice_and_mud.parameter('mud_thickness').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%MUD_THICKNESS = {self._use_flag(ice_and_mud.parameter('mud_thickness').value)}\n"
            )
        if ice_and_mud.parameter('mud_viscosity').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%FORCING%MUD_VISCOSITY = {self._use_flag(ice_and_mud.parameter('mud_viscosity').value)}\n"
            )
        #  These 3 options are currently hidden in the Model Control
        if ice_and_mud.parameter('assimilation_data_mean_para').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%ASSIM%MEAN            = "
                f"{self._use_flag(ice_and_mud.parameter('assimilation_data_mean_para').value)}\n"
            )
        if ice_and_mud.parameter('assimilation_data_1D_spectra').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%ASSIM%SPEC1D          = "
                f"{self._use_flag(ice_and_mud.parameter('assimilation_data_1D_spectra').value)}\n"
            )
        if ice_and_mud.parameter('assimilation_data_2D_spectra').value != 'F: no forcing':
            self._ss.write(
                f"  INPUT%ASSIM%SPEC2D          = "
                f"{self._use_flag(ice_and_mud.parameter('assimilation_data_2D_spectra').value)}\n"
            )
        self._ss.write('/\n\n\n\n\n\n')

    def _use_flag(self, value):
        """Returns a letter code F, T, H, or C for the value flag passed in.

        Args:
            value (:obj:`int`):  The forcing or assim value to convert to a string.

        Returns:
            (:obj:`str`):  F, T, H, or C
        """
        # if value == 'F: no forcing': It should be impossible for value to be 'F'.
        #     return "'F'"
        if value == 'T: external forcing file':
            return "'T'"
        elif value == 'H: homogeneous forcing input':
            return "'H'"
        elif value == 'C: coupled forcing field':
            return "'C'"
        else:
            raise ValueError("Invalid forcing code.")

    def _write_define_output_point_types_header(self):
        """Writes out the header for the output types section."""
        self._ss.write(
            "! -------------------------------------------------------------------- !\n"
            "! Define the output types point parameters via OUTPUT_TYPE_NML namelist\n"
            "!\n"
            "! * the point file is a space separated values per line :\n"
            "!   longitude latitude 'name' (C*40)\n"
            "!\n"
            "! * the full list of field names is : \n"
            "!  All parameters listed below are available in output file of the types\n"
            "!  ASCII and NetCDF. If selected output file types are grads or grib, \n"
            "!  some parameters may not be available. The first two columns in the\n"
            "!  table below identify such cases by flags, cols 1 (GRB) and 2 (GXO)\n"
            "!  refer to grib (ww3_grib) and grads (gx_outf), respectively.\n"
            "!\n"
            "! Columns 3 and 4 provide group and parameter numbers per group.\n"
            "! Columns 5, 6 and 7 provide:\n"
            "!   5 - code name (internal)\n"
            "!   6 - output tags (names used is ASCII file extensions, NetCDF \n"
            "!       variable names and namelist-based selection\n"
            "!   7 - Long parameter name/definition\n"
            "!\n"
            "!  G  G\n"
            "!  R  X Grp  Param Code     Output  Parameter/Group\n"
            "!  B  O Numb Numbr Name        Tag  Definition \n"
            "!  --------------------------------------------------\n"
            "!        1                          Forcing Fields\n"
            "!   -------------------------------------------------\n"
            "!  T  T  1     1   DW         DPT   Water depth.\n"
            "!  T  T  1     2   C[X,Y]     CUR   Current velocity.\n"
            "!  T  T  1     3   UA         WND   Wind speed.\n"
            "!  T  T  1     4   AS         AST   Air-sea temperature difference.\n"
            "!  T  T  1     5   WLV        WLV   Water levels.\n"
            "!  T  T  1     6   ICE        ICE   Ice concentration.\n"
            "!  T  T  1     7   IBG        IBG   Iceberg-induced damping.\n"
            "!  T  T  1     8   D50        D50   Median sediment grain size.\n"
            "!  T  T  1     9   IC1        IC1   Ice thickness.\n"
            "!  T  T  1    10   IC5        IC5   Ice flow diameter.\n"
            "!   -------------------------------------------------\n"
            "!        2                          Standard mean wave Parameters\n"
            "!   -------------------------------------------------\n"
            "!  T  T  2     1   HS         HS    Wave height.\n"
            "!  T  T  2     2   WLM        LM    Mean wave length.\n"
            "!  T  T  2     3   T02        T02   Mean wave period (Tm0,2).\n"
            "!  T  T  2     4   T0M1       T0M1  Mean wave period (Tm0,-1).\n"
            "!  T  T  2     5   T01        T01   Mean wave period (Tm0,1).\n"
            "!  T  T  2     6   FP0        FP    Peak frequency.\n"
            "!  T  T  2     7   THM        DIR   Mean wave direction.\n"
            "!  T  T  2     8   THS        SPR   Mean directional spread.\n"
            "!  T  T  2     9   THP0       DP    Peak direction.\n"
            "!  T  T  2    10   HIG        HIG   Infragravity height\n"
            "!  T  T  2    11   STMAXE     MXE   Max surface elev (STE)\n"
            "!  T  T  2    12   STMAXD     MXES  St Dev of max surface elev (STE)\n"
            "!  T  T  2    13   HMAXE      MXH   Max wave height (STE)\n"
            "!  T  T  2    14   HCMAXE     MXHC  Max wave height from crest (STE)\n"
            "!  T  T  2    15   HMAXD      SDMH  St Dev of MXC (STE)\n"
            "!  T  T  2    16   HCMAXD     SDMHC St Dev of MXHC (STE)\n"
            "!  F  T  2    17   WBT        WBT   Dominant wave breaking probability bT\n"
            "!  F  F  2    18   FP0        TP    Peak period (from peak freq)\n"
            "!   -------------------------------------------------\n"
            "!        3                          Spectral Parameters (first 5)\n"
            "!   -------------------------------------------------\n"
            "!  F  F  3     1   EF         EF    Wave frequency spectrum\n"
            "!  F  F  3     2   TH1M       TH1M  Mean wave direction from a1,b2\n"
            "!  F  F  3     3   STH1M      STH1M Directional spreading from a1,b2\n"
            "!  F  F  3     4   TH2M       TH2M  Mean wave direction from a2,b2\n"
            "!  F  F  3     5   STH2M      STH2M Directional spreading from a2,b2\n"
            "!  F  F  3     6   WN         WN    Wavenumber array\n"
            "!   -------------------------------------------------\n"
            "!        4                          Spectral Partition Parameters \n"
            "!   -------------------------------------------------\n"
            "!  T  T  4     1   PHS        PHS   Partitioned wave heights.\n"
            "!  T  T  4     2   PTP        PTP   Partitioned peak period.\n"
            "!  T  T  4     3   PLP        PLP   Partitioned peak wave length.\n"
            "!  T  T  4     4   PDIR       PDIR  Partitioned mean direction.\n"
            "!  T  T  4     5   PSI        PSPR  Partitioned mean directional spread.\n"
            "!  T  T  4     6   PWS        PWS   Partitioned wind sea fraction.\n"
            "!  T  T  4     7   PDP        PDP   Peak wave direction of partition.\n"
            "!  T  T  4     8   PQP        PQP   Goda peakdedness parameter of partition.\n"
            "!  T  T  4     9   PPE        PPE   JONSWAP peak enhancement factor of partition.\n"
            "!  T  T  4    10   PGW        PGW   Gaussian frequency width of partition.\n"
            "!  T  T  4    11   PSW        PSW   Spectral width of partition.\n"
            "!  T  T  4    12   PTM1       PTM10 Mean wave period (m-1,0) of partition.\n"
            "!  T  T  4    13   PT1        PT01  Mean wave period (m0,1) of partition.\n"
            "!  T  T  4    14   PT2        PT02  Mean wave period (m0,2) of partition.\n"
            "!  T  T  4    15   PEP        PEP   Peak spectral density of partition.\n"
            "!  T  T  4    16   PWST       TWS   Total wind sea fraction.\n"
            "!  T  T  4    17   PNR        PNR   Number of partitions.\n"
            "!   -------------------------------------------------\n"
            "!        5                          Atmosphere-waves layer\n"
            "!   -------------------------------------------------\n"
            "!  T  T  5     1   UST        UST   Friction velocity.\n"
            "!  F  T  5     2   CHARN      CHA   Charnock parameter\n"
            "!  F  T  5     3   CGE        CGE   Energy flux\n"
            "!  F  T  5     4   PHIAW      FAW   Air-sea energy flux\n"
            "!  F  T  5     5   TAUWI[X,Y] TAW   Net wave-supported stress\n"
            "!  F  T  5     6   TAUWN[X,Y] TWA   Negative part of the wave-supported stress\n"
            "!  F  F  5     7   WHITECAP   WCC   Whitecap coverage\n"
            "!  F  F  5     8   WHITECAP   WCF   Whitecap thickness\n"
            "!  F  F  5     9   WHITECAP   WCH   Mean breaking height\n"
            "!  F  F  5    10   WHITECAP   WCM   Whitecap moment\n"
            "!  F  F  5    11   FWS        FWS   Wind sea mean period\n"
            "!   -------------------------------------------------\n"
            "!        6                          Wave-ocean layer \n"
            "!   -------------------------------------------------\n"
            "!  F  F  6     1   S[XX,YY,XY] SXY  Radiation stresses.\n"
            "!  F  F  6     2   TAUO[X,Y]  TWO   Wave to ocean momentum flux\n"
            "!  F  F  6     3   BHD        BHD   Bernoulli head (J term) \n"
            "!  F  F  6     4   PHIOC      FOC   Wave to ocean energy flux\n"
            "!  F  F  6     5   TUS[X,Y]   TUS   Stokes transport\n"
            "!  F  F  6     6   USS[X,Y]   USS   Surface Stokes drift\n"
            "!  F  F  6     7   [PR,TP]MS  P2S   Second-order sum pressure \n"
            "!  F  F  6     8   US3D       USF   Spectrum of surface Stokes drift\n"
            "!  F  F  6     9   P2SMS      P2L   Micro seism  source term\n"
            "!  F  F  6    10   TAUICE     TWI   Wave to sea ice stress\n"
            "!  F  F  6    11   PHICE      FIC   Wave to sea ice energy flux\n"
            "!  F  F  6    12   USSP       USP   Partitioned surface Stokes drift\n"
            "!   -------------------------------------------------\n"
            "!        7                          Wave-bottom layer \n"
            "!   -------------------------------------------------\n"
            "!  F  F  7     1   ABA        ABR   Near bottom rms amplitides.\n"
            "!  F  F  7     2   UBA        UBR   Near bottom rms velocities.\n"
            "!  F  F  7     3   BEDFORMS   BED   Bedforms\n"
            "!  F  F  7     4   PHIBBL     FBB   Energy flux due to bottom friction \n"
            "!  F  F  7     5   TAUBBL     TBB   Momentum flux due to bottom friction\n"
            "!   -------------------------------------------------\n"
            "!        8                          Spectrum parameters\n"
            "!   -------------------------------------------------\n"
            "!  F  F  8     1   MSS[X,Y]   MSS   Mean square slopes\n"
            "!  F  F  8     2   MSC[X,Y]   MSC   Spectral level at high frequency tail\n"
            "!  F  F  8     3   WL02[X,Y]  WL02  East/X North/Y mean wavelength compon\n"
            "!  F  F  8     4   ALPXT      AXT   Correl sea surface gradients (x,t)\n"
            "!  F  F  8     5   ALPYT      AYT   Correl sea surface gradients (y,t)\n"
            "!  F  F  8     6   ALPXY      AXY   Correl sea surface gradients (x,y)\n"
            "!   -------------------------------------------------\n"
            "!        9                          Numerical diagnostics  \n"
            "!   -------------------------------------------------\n"
            "!  T  T  9     1   DTDYN      DTD   Average time step in integration.\n"
            "!  T  T  9     2   FCUT       FC    Cut-off frequency.\n"
            "!  T  T  9     3   CFLXYMAX   CFX   Max. CFL number for spatial advection. \n"
            "!  T  T  9     4   CFLTHMAX   CFD   Max. CFL number for theta-advection. \n"
            "!  F  F  9     5   CFLKMAX    CFK   Max. CFL number for k-advection. \n"
            "!   -------------------------------------------------\n"
            "!        10                         User defined          \n"
            "!   -------------------------------------------------\n"
            "!  F  F  10    1              U1    User defined #1. (requires coding ...)\n"
            "!  F  F  10    2              U2    User defined #1. (requires coding ...)\n"
            "!   -------------------------------------------------\n"
            "!\n"
            "!     Section 4 consist of a set of fields, index 0 = wind sea, index\n"
            "!     1:NOSWLL are first NOSWLL swell fields.\n"
            "!\n"
            "!\n"
            "! * output track file formatted (T) or unformated (F)\n"
            "!\n"
            "! * coupling fields exchanged list is :\n"
            "!   - Sent fields by ww3:\n"
            "!       - Ocean model : T0M1 OCHA OHS DIR BHD TWO UBR FOC TAW TUS USS LM DRY\n"
            "!       - Atmospheric model : ACHA AHS TP (or FP) FWS\n"
            "!       - Ice model : IC5 TWI\n"
            "!   - Received fields by ww3:\n"
            "!       - Ocean model : SSH CUR\n"
            "!       - Atmospheric model : WND\n"
            "!       - Ice model : ICE IC1 IC5\n"
            "!\n"
            "! * Coupling at T+0 (extra fields in the restart needed)\n"
            "!\n"
            "! * extra fields to be written to the restart:\n"
            "!   - The list includes all fields sent by coupling exchange only\n"
            "!\n"
            "! * namelist must be terminated with /\n"
            "! * definitions & defaults:\n"
            "!     TYPE%FIELD%LIST         =  'unset'\n"
            "!     TYPE%POINT%FILE         =  'points.list'\n"
            "!     TYPE%TRACK%FORMAT       =  T\n"
            "!     TYPE%PARTITION%X0       =  0\n"
            "!     TYPE%PARTITION%XN       =  0\n"
            "!     TYPE%PARTITION%NX       =  0\n"
            "!     TYPE%PARTITION%Y0       =  0\n"
            "!     TYPE%PARTITION%YN       =  0\n"
            "!     TYPE%PARTITION%NY       =  0\n"
            "!     TYPE%PARTITION%FORMAT   =  T\n"
            "!     TYPE%COUPLING%SENT      = 'unset'\n"
            "!     TYPE%COUPLING%RECEIVED  = 'unset'\n"
            "!     TYPE%COUPLING%COUPLET0  = .FALSE.\n"
            "!     TYPE%RESTART%EXTRA      = 'unset'\n"
            "!\n"
            "! -------------------------------------------------------------------- !\n"
        )

    def _handle_output_points_data(self, filename='./points.list'):
        """Handles writing out any output points to a point file, and returns the filename if created.

        Args:
            filename(:obj:`str`):  The filename to write to.

        Returns:
            (:obj:`str`):  The path of the points file, if created, else and empty string.
        """
        # Check for the existence of any points
        coverage = self._xms_data.output_points_coverage
        points = coverage.get_points(FilterLocation.PT_LOC_DISJOINT) if coverage else []
        if not points:
            # No output points, return empty string
            return ''

        # Read the points along with their names, and write to a file
        data = self._xms_data.output_points_data
        section = get_model().point_parameters

        with open(filename, 'w') as f:
            for point in points:
                values = data.feature_values(TargetType.point, feature_id=point.id)
                section.restore_values(values)
                name = section.group('output_point').parameter('name').value
                # Write X Y 'point_name'
                f.write(f'{point.x: .10f} {point.y: .10f} \'{name}\'\n')
        return filename

    def _write_define_output_point_types_namelist(self):
        """Writes the output points namelist."""
        self._write_define_output_point_types_header()
        self._ss.write('&OUTPUT_TYPE_NML\n')
        field_list_str = get_output_field_type_string(self._xms_data.sim_data_output_fields)
        if len(field_list_str) > 0:
            self._ss.write(f"    TYPE%FIELD%LIST          = '{field_list_str}'\n")
        point_list_str = self._handle_output_points_data()
        if len(point_list_str) > 0:
            self._ss.write(f"    TYPE%POINT%FILE          = '{point_list_str}'\n")
        self._ss.write('/\n\n\n\n')

    def _write_output_dates_namelist(self):
        """Writes out the output dates namelist."""
        self._ss.write(
            "! -------------------------------------------------------------------- !\n"
            "! Define output dates via OUTPUT_DATE_NML namelist\n"
            "!\n"
            "! * start and stop times are with format 'yyyymmdd hhmmss'\n"
            "! * if time stride is equal '0', then output is disabled\n"
            "! * time stride is given in seconds\n"
            "!\n"
            "! * namelist must be terminated with /\n"
            "! * definitions & defaults:\n"
            "!     DATE%FIELD%START         =  '19680606 000000'\n"
            "!     DATE%FIELD%STRIDE        =  '0'\n"
            "!     DATE%FIELD%STOP          =  '19680607 000000'\n"
            "!     DATE%POINT%START         =  '19680606 000000'\n"
            "!     DATE%POINT%STRIDE        =  '0'\n"
            "!     DATE%POINT%STOP          =  '19680607 000000'\n"
            "!     DATE%TRACK%START         =  '19680606 000000'\n"
            "!     DATE%TRACK%STRIDE        =  '0'\n"
            "!     DATE%TRACK%STOP          =  '19680607 000000'\n"
            "!     DATE%RESTART%START       =  '19680606 000000'\n"
            "!     DATE%RESTART%STRIDE      =  '0'\n"
            "!     DATE%RESTART%STOP        =  '19680607 000000'\n"
            "!     DATE%BOUNDARY%START      =  '19680606 000000'\n"
            "!     DATE%BOUNDARY%STRIDE     =  '0'\n"
            "!     DATE%BOUNDARY%STOP       =  '19680607 000000'\n"
            "!     DATE%PARTITION%START     =  '19680606 000000'\n"
            "!     DATE%PARTITION%STRIDE    =  '0'\n"
            "!     DATE%PARTITION%STOP      =  '19680607 000000'\n"
            "!     DATE%COUPLING%START      =  '19680606 000000'\n"
            "!     DATE%COUPLING%STRIDE     =  '0'\n"
            "!     DATE%COUPLING%STOP       =  '19680607 000000'\n"
            "!\n"
            "!     DATE%RESTART             =  '19680606 000000' '0' '19680607 000000'\n"
            "! -------------------------------------------------------------------- !\n"
        )
        param_names = [
            'output_start_date',
            'output_second_interval',
            'output_end_date',
            'output_start_date_point',
            'output_second_interval_point',
            'output_end_date_point',
            'output_start_date_track',
            'output_second_interval_track',
            'output_end_date_track',
            'output_start_date_boundary',
            'output_second_interval_boundary',
            'output_end_date_boundary',
            'output_start_date_separated',
            'output_second_interval_separated',
            'output_end_date_separated',
            'output_start_date_coupling',
            'output_second_interval_coupling',
            'output_end_date_coupling',
            'output_start_date_restart',
            'output_second_interval_restart',
            'output_end_date_restart',
        ]

        output = self._xms_data.sim_data_model_control.group('output')
        fields_output = self._xms_data.sim_data_output_fields.group('time_parameters')
        attrs = {
            var:
                (
                    output.parameter(var).value if hasattr(output, 'parameter') and var in output.parameter_names else
                    fields_output.parameter(var).value
                )
            for var in param_names
        }

        self._ss.write('&OUTPUT_DATE_NML\n')
        self._print_dates_and_stride(
            'DATE%FIELD', attrs['output_start_date'], attrs['output_second_interval'], attrs['output_end_date']
        )
        self._print_dates_and_stride(
            'DATE%POINT', attrs['output_start_date_point'], attrs['output_second_interval_point'],
            attrs['output_end_date_point']
        )
        self._print_dates_and_stride(
            'DATE%TRACK', attrs['output_start_date_track'], attrs['output_second_interval_track'],
            attrs['output_end_date_track']
        )
        self._print_dates_and_stride(
            'DATE%BOUNDARY', attrs['output_start_date_boundary'], attrs['output_second_interval_boundary'],
            attrs['output_end_date_boundary']
        )
        self._print_dates_and_stride(
            'DATE%PARTITION', attrs['output_start_date_separated'], attrs['output_second_interval_separated'],
            attrs['output_end_date_separated']
        )
        self._print_dates_and_stride(
            'DATE%COUPLING', attrs['output_start_date_coupling'], attrs['output_second_interval_coupling'],
            attrs['output_end_date_coupling']
        )
        self._print_dates_and_stride(
            'DATE%RESTART',
            attrs['output_start_date_restart'],
            attrs['output_second_interval_restart'],
            attrs['output_end_date_restart'],
            one_line=True
        )
        self._ss.write('/\n\n\n\n\n')

    def _print_dates_and_stride(self, prefix, start_date_str, stride, end_date_str, one_line=False):
        """Prints out start, stride, and end date strings into the file, if the stride values is non-zero.

        Args:
            prefix(:obj:`str`):  The prefix to print out for the lines in the namelist.
            start_date_str(:obj:`str`):  The start date string, in format "Thu Jul 20 00:00:00 1989"
            stride(:obj:`int`):  The stride (interval) in seconds.
            end_date_str(:obj:`str`):  The end date string, in format "Thu Jul 20 00:00:00 1989"
            one_line(:obj:`bool`):  Whether or not to print all three time strings on one line.
        """
        if stride != 0:
            start_field = f'     {prefix}%START'
            start_field = start_field.ljust(30)
            stride_field = f'     {prefix}%STRIDE'
            stride_field = stride_field.ljust(30)
            stop_field = f'     {prefix}%STOP'
            stop_field = stop_field.ljust(30)
            prefix_field = f'     {prefix}'
            prefix_field = prefix_field.ljust(30)

            start_date = start_date_str
            if not start_date:
                start_date = self._xms_data.global_time
            else:
                start_date = string_to_datetime(start_date)
            end_date = end_date_str
            if not end_date:
                end_date = self._xms_data.global_time
            else:
                end_date = string_to_datetime(end_date)

            # Restart wants an extra blank line
            if prefix == 'DATE%RESTART':
                self._ss.write('\n')

            # Write out start, stride, and stop
            # Could be on three separate lines, or one line
            if not one_line:
                # Three lines of printing, each with its own prefix
                self._ss.write(f"{start_field}= '{get_formatted_date_string(start_date)}'\n")
                self._ss.write(f"{stride_field}= '{stride}'\n")
                self._ss.write(f"{stop_field}= '{get_formatted_date_string(end_date)}'\n")
            else:
                # One line of printing, no special prefixes
                self._ss.write(
                    f"{prefix_field}= '{get_formatted_date_string(start_date)}' '{stride}' "
                    f"'{get_formatted_date_string(end_date)}'\n"
                )

    def _write_homogeneous_namelists(self):
        """Writes out the homogeneous namelists."""
        self._ss.write(
            "! -------------------------------------------------------------------- !\n"
            "! Define homogeneous input via HOMOG_COUNT_NML and HOMOG_INPUT_NML namelist\n"
            "!\n"
            "! * the number of each homogeneous input is defined by HOMOG_COUNT\n"
            "! * the total number of homogeneous input is automatically calculated\n"
            "! * the homogeneous input must start from index 1 to N\n"
            "! * if VALUE1 is equal 0, then the homogeneous input is desactivated\n"
            "! * NAME can be IC1, IC2, IC3, IC4, IC5, MDN, MTH, MVS, LEV, CUR, WND, ICE, MOV\n"
            "! * each homogeneous input is defined over a maximum of 3 values detailled below :\n"
            "!     - IC1 is defined by thickness\n"
            "!     - IC2 is defined by viscosity\n"
            "!     - IC3 is defined by density\n"
            "!     - IC4 is defined by modulus\n"
            "!     - IC5 is defined by floe diameter\n"
            "!     - MDN is defined by density\n"
            "!     - MTH is defined by thickness\n"
            "!     - MVS is defined by viscosity\n"
            "!     - LEV is defined by height\n"
            "!     - CUR is defined by speed and direction\n"
            "!     - WND is defined by speed, direction and airseatemp\n"
            "!     - ICE is defined by concentration\n"
            "!     - MOV is defined by speed and direction\n"
            "!\n"
            "! * namelist must be terminated with /\n"
            "! * definitions & defaults:\n"
            "!     HOMOG_COUNT%N_IC1            =  0\n"
            "!     HOMOG_COUNT%N_IC2            =  0\n"
            "!     HOMOG_COUNT%N_IC3            =  0\n"
            "!     HOMOG_COUNT%N_IC4            =  0\n"
            "!     HOMOG_COUNT%N_IC5            =  0\n"
            "!     HOMOG_COUNT%N_MDN            =  0\n"
            "!     HOMOG_COUNT%N_MTH            =  0\n"
            "!     HOMOG_COUNT%N_MVS            =  0\n"
            "!     HOMOG_COUNT%N_LEV            =  0\n"
            "!     HOMOG_COUNT%N_CUR            =  0\n"
            "!     HOMOG_COUNT%N_WND            =  0\n"
            "!     HOMOG_COUNT%N_ICE            =  0\n"
            "!     HOMOG_COUNT%N_MOV            =  0\n"
            "!\n"
            "!     HOMOG_INPUT(I)%NAME           =  'unset'\n"
            "!     HOMOG_INPUT(I)%DATE           =  '19680606 000000'\n"
            "!     HOMOG_INPUT(I)%VALUE1         =  0\n"
            "!     HOMOG_INPUT(I)%VALUE2         =  0\n"
            "!     HOMOG_INPUT(I)%VALUE3         =  0\n"
            "! -------------------------------------------------------------------- !\n"
        )
        # Homogeneous counts
        num_ic1 = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('ice_param_1').value
        num_ic2 = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('ice_param_2').value
        num_ic3 = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('ice_param_3').value
        num_ic4 = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('ice_param_4').value
        num_ic5 = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('ice_param_5').value
        num_mdn = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('mud_density_table').value
        num_mth = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('mud_thickness_table').value
        num_mvs = self._xms_data.sim_data_model_control.group('ice_and_mud').parameter('mud_viscosity_table').value
        num_lev = self._xms_data.sim_data_model_control.group('run_control').parameter('water_level_homogeneous').value
        num_cur = self._xms_data.sim_data_model_control.group('run_control').parameter('currents_homogeneous').value
        num_wnd = self._xms_data.sim_data_model_control.group('run_control').parameter('winds_homogeneous').value
        self._ss.write('&HOMOG_COUNT_NML\n')
        if num_ic1:
            self._ss.write(f"    HOMOG_COUNT%N_IC1 = {len(num_ic1)}\n")
        if num_ic2:
            self._ss.write(f"    HOMOG_COUNT%N_IC2 = {len(num_ic2)}\n")
        if num_ic3:
            self._ss.write(f"    HOMOG_COUNT%N_IC3 = {len(num_ic3)}\n")
        if num_ic4:
            self._ss.write(f"    HOMOG_COUNT%N_IC4 = {len(num_ic4)}\n")
        if num_ic5:
            self._ss.write(f"    HOMOG_COUNT%N_IC5 = {len(num_ic5)}\n")
        if num_mdn:
            self._ss.write(f"    HOMOG_COUNT%N_MDN = {len(num_mdn)}\n")
        if num_mth:
            self._ss.write(f"    HOMOG_COUNT%N_MTH = {len(num_mth)}\n")
        if num_mvs:
            self._ss.write(f"    HOMOG_COUNT%N_MVS = {len(num_mvs)}\n")
        if num_lev:
            self._ss.write(f"    HOMOG_COUNT%N_LEV = {len(num_lev)}\n")
        if num_cur:
            self._ss.write(f"    HOMOG_COUNT%N_CUR = {len(num_cur)}\n")
        if num_wnd:
            self._ss.write(f"    HOMOG_COUNT%N_WND = {len(num_wnd)}\n")
        self._ss.write('/\n\n')

        self._ss.write('&HOMOG_INPUT_NML\n')
        counter = 1
        for i in num_ic1:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'IC1'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_ic2:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'IC2'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_ic3:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'IC3'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_ic4:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'IC4'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_ic5:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'IC5'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_mdn:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'MDN'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_mth:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'MTH'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_mvs:
            dt = parse_datetime(i[0])
            cur_val1 = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'MVS'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_val1}\n")
            counter = counter + 1
        for i in num_lev:
            dt = parse_datetime(i[0])
            cur_height = float(i[1])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'LEV'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_height}\n")
            counter = counter + 1
        for i in num_cur:
            dt = parse_datetime(i[0])
            cur_speed = float(i[1])
            cur_dir = float(i[2])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'CUR'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_speed}\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE2 = {cur_dir}\n")
            counter = counter + 1
        for i in num_wnd:
            dt = parse_datetime(i[0])
            cur_speed = float(i[1])
            cur_dir = float(i[2])
            cur_ast = float(i[3])
            date_str = get_formatted_date_string(dt)
            self._ss.write(f"    HOMOG_INPUT({counter})%NAME = 'WND'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%DATE = '{date_str}'\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE1 = {cur_speed}\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE2 = {cur_dir}\n")
            self._ss.write(f"    HOMOG_INPUT({counter})%VALUE3 = {cur_ast}\n")
            counter = counter + 1
        self._ss.write('/\n\n\n\n')

    def _write_end_comments(self):
        """Writes out the comments at the end of the namelist file."""
        self._ss.write(
            '! -------------------------------------------------------------------- !\n'
            '! WAVEWATCH III - end of namelist                                      !\n'
            '! -------------------------------------------------------------------- !\n'
        )

    def _flush(self, file_w_path):
        """Writes the StringIO previously processed to a file.

        Args:
            file_w_path (:obj:`str`):  String of the filename to write to.
        """
        f = open(file_w_path, 'w')
        self._ss.seek(0)
        shutil.copyfileobj(self._ss, f, 100000)
        f.close()

    def write(self):
        """Top-level entry point for the WaveWatch3 shell input file writer."""
        # self._start_grid_process()
        self._write_shell_namelist_file()

        # self._logger.info('Waiting for parallel export processes to complete...')
        # if self._grid_process is not None:
        #     self._grid_process.wait()
        # if self._bound_process is not None:
        #     self._bound_process.wait()


def parse_datetime(datetime_str):
    """Ensures the datetime string contains both date and time information.

    Args:
        datetime_str (:obj:`str`): String containing the date and possibly the time.
    """
    if len(datetime_str) == 10:  # yyy-mm-dd
        datetime_str += ' 00:00:00'
    return datetime.datetime.strptime(datetime_str, '%Y-%m-%d %H:%M:%S')
