"""Classes to handle the GUI side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import List
import uuid


# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class CommandItem:
    """Data class to hold commands that are executed. Used to keep an undo/redo history."""

    def __init__(self, command, uuid, new_value, prev_value, index=None, uuid_index=None):
        """Initialize the CommandItem class.

        Args:
            command (string): Tells which function is called to undo/redo command
            uuid (string): Unique identifier for the command
            new_value (varies): Value after the command
            prev_value (varies): Value before the command
            index (int): Index of the set value (for list type variables)
            uuid_index (string): Unique identifier for the index of the set value
        """
        self.command = command
        self.uuid = uuid
        self.prev_value = prev_value
        self.new_value = new_value
        self.index = index
        self.uuid_index = uuid_index  # Needed for list type variables
        # self.name = name


class BatchCommandItem:
    """Data class to hold a batch of commands that are executed. Used to keep an undo/redo history."""

    def __init__(self, command: str = None, commands: List[CommandItem] = None, batch_uuid: uuid.UUID = None,
                 new_value=None, prev_value=None, index: int = None,
                 name: str = '', uuid_index: int = None):
        """Initialize the BatchCommandItem class.

        Specify command new_value, prev_value, index=None, or commands.

        Args:
            command (CommandItem): The command to include in the batch
            commands (list): List of CommandItem instances
            batch_uuid (string): Unique identifier for the batch command
            new_value (varies): Value after the command
            prev_value (varies): Value before the command
            index (int): Index of the set value (for list type variables)
            name (string): Text shown to the user in a display of undo/redo history
            uuid_index (string): Unique identifier for the index of the set value
        """
        if command and not commands:
            command = CommandItem(command=command, uuid=uuid.uuid4(), new_value=new_value, prev_value=prev_value,
                                  index=index, uuid_index=uuid_index)
            commands = [command]
        self.commands = commands if commands is not None else []
        self.uuid = batch_uuid if batch_uuid is not None else uuid.uuid4()
        self.name = name
        self.uuid_index = uuid_index

    def append_command(self, command):
        """Append a command to the batch.

        Args:
            command (CommandItem): The command to append.
        """
        self.commands.append(command)


class CommandManager:
    """Class to manage the undo/redo history of commands."""

    def __init__(self, max_history=100):
        """Initialize the CommandManager class."""
        self.log_file = None
        self._undo_stack = []
        self._redo_stack = []
        self._max_history = max_history

    def add_new_undo_command(self, new_command, max_history=None):
        """Adds a new undo command to our history list.

        Args:
            new_command (CommandItem): the command that should be added to the history
            max_history (int): the maximum number of commands to keep in the history
        """
        if max_history is not None:
            self._max_history = max_history
        # add as a list, so we can have a batch of commanditems
        # TODO: refactor copy/paste to create a batch of commands, instead of a bunch of individual commands
        # TODO: refactor a change calc to be a batch operation? (So we could undo outside the dialog, in the main)
        self._undo_stack.append(new_command)
        if len(self._undo_stack) > self._max_history:
            self._undo_stack.pop(0)
        self._redo_stack = []

    def _move_undo_command_to_redo(self):
        """Moves the last undo command from undo history to redo history."""
        self._redo_stack.append(self._undo_stack[-1])
        self._undo_stack.pop(-1)

    def _move_redo_command_to_undo(self):
        """Moves the last redo command from undo history to undo history."""
        self._undo_stack.append(self._redo_stack[-1])
        self._redo_stack.pop(-1)

    def undo(self, operate_command_func):
        """Undo - Perform an Undo for the last item in undo history.

        Args:
            operate_command_func (function): function to operate the command
        """
        if len(self._undo_stack) > 0:
            batch_commands = self._undo_stack[-1]
            for command_item in batch_commands.commands:
                operate_command_func(command_item, new_value=command_item.prev_value, prev_value=command_item.new_value)
            self._move_undo_command_to_redo()

    def redo(self, operate_command_func):
        """Redo - Perform an Redo for the last item in redo history.

        Args:
            operate_command_func (function): function to operate the command
        """
        if len(self._redo_stack) > 0:
            batch_commands = self._redo_stack[-1]
            for command_item in batch_commands.commands:
                operate_command_func(command_item, new_value=command_item.new_value, prev_value=command_item.prev_value)
            self._move_redo_command_to_undo()

    def clear_redo(self):
        """Clear the redo history."""
        self._redo_stack = []

    def clear_undo(self):
        """Clear the undo history."""
        self._undo_stack = []

    def clear_history(self):
        """Clear the undo and redo history."""
        self.clear_undo()
        self.clear_redo()

    def get_undo_list(self):
        """Returns the undo list."""
        return self._undo_stack

    def get_redo_list(self):
        """Returns the redo list."""
        return self._redo_stack
