"""Perform Unit Conversion."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class ConversionViscosityCalc:
    """Perform viscosity unit conversion computations."""

    # Lists to determine units from strings
    # ----------------------------------------------------------------------
    # Dynamic Viscosity
    # ----------------------------------------------------------------------
    kns_m2 = ['kNs/m^2', 'kN*s/m^2']
    ns_m2 = ['Ns/m^2', 'N*s/m^2']
    pas_m2 = ['pas', 'pa*s']
    lbs_ft2 = ['lbs/ft^2', 'lbfs/ft^2', 'lb*s/ft^2', 'lbf*s/ft^2']
    # ----------------------------------------------------------------------
    # Kinematic Viscosity
    # ----------------------------------------------------------------------
    m2_s = ['m^2/s',]
    mm2_s = ['mm^2/s',]
    ft2_s = ['ft^2/s']

    def __init__(self):
        """Initialize the ConversionCalc Class.

        Returns:
            True if converted, False if units not found
        """

    def get_si_complementary_unit(self, from_unit):
        """Get the complementary (similar) si unit as given US unit.

        Args:
            from_unit (string): US unit

        Returns:
            Successful (bool): If the functions suceeded
            SI Unit (str): The complementary unit
        """
        if from_unit in self.lbs_ft2:
            return True, 'pas'
        elif from_unit in self.ft2_s:
            return True, 'm^2/s'
        return False, ''

    # ----------------------------------------------------------------------
    # viscosity
    # ----------------------------------------------------------------------

    def convert_units(self, from_unit, to_unit, value):
        """Convert viscosity.

        Args:
            from_unit (string): unit that value is in currently
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if from_unit in self.kns_m2:
            return self.convert_from_kns_m2(to_unit, value)
        elif from_unit in self.ns_m2 or from_unit in self.pas_m2:
            return self.convert_from_pa_m2(to_unit, value)
        elif from_unit in self.lbs_ft2:
            return self.convert_from_lbs_ft2(to_unit, value)

        if from_unit in self.m2_s:
            return self.convert_from_m2_s(to_unit, value)
        elif from_unit in self.mm2_s:
            return self.convert_from_mm2_s(to_unit, value)
        elif from_unit in self.ft2_s:
            return self.convert_from_ft2_s(to_unit, value)

        return False, value

    def convert_from_kns_m2(self, to_unit, value):
        """Convert from the kN*s/m2 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.kns_m2:
            return True, value
        elif to_unit in self.ns_m2 or to_unit in self.pas_m2:
            return True, value * 1000.0
        elif to_unit in self.lbs_ft2:
            return True, value * 671.968994813

        else:
            return False, value

    def convert_from_pa_m2(self, to_unit, value):
        """Convert from the pas/m2 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.kns_m2:
            return True, value / 1000.0
        elif to_unit in self.ns_m2 or to_unit in self.pas_m2:
            return True, value
        elif to_unit in self.lbs_ft2:
            return True, value * 0.671968994813

        else:
            return False, value

    def convert_from_lbs_ft2(self, to_unit, value):
        """Convert from the lbs/ft^2 unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.kns_m2:
            return True, value * 0.001488163900000009
        elif to_unit in self.ns_m2 or to_unit in self.pas_m2:
            return True, value * 1.488163900000009
        elif to_unit in self.lbs_ft2:
            return True, value

        else:
            return False, value

    def convert_from_m2_s(self, to_unit, value):
        """Convert from the m^2/s unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.m2_s:
            return True, value
        elif to_unit in self.mm2_s:
            return True, value * 1000000.0
        elif to_unit in self.ft2_s:
            return True, value * 10.76391041671

        else:
            return False, value

    def convert_from_mm2_s(self, to_unit, value):
        """Convert from the mm^2/s unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.m2_s:
            return True, value / 1000000.0
        elif to_unit in self.mm2_s:
            return True, value
        elif to_unit in self.ft2_s:
            return True, value * 1.076391041671e-5

        else:
            return False, value

    def convert_from_ft2_s(self, to_unit, value):
        """Convert from the ft^2/s unit.

        Args:
            to_unit (string): unit that value needs to be converted to
            value (float): value to convert

        Returns:
            True if converted, False if units not found
            value (float): converted value
        """
        if to_unit in self.m2_s:
            return True, value / 10.76391041671
        elif to_unit in self.mm2_s:
            return True, value / 1.076391041671e-5
        elif to_unit in self.ft2_s:
            return True, value

        else:
            return False, value
