"""Classes to handle the GUI side of variables."""
__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
from dataclasses import dataclass, field, InitVar
from pathlib import Path
from typing import Any, List, Optional
from uuid import UUID, uuid4

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.FhwaVariable.interface_adapters.view_model.main.tree_data import TreeData
# from xms.FhwaVariable.core_data.calculator.metadata import Metadata


@dataclass
class _TreeItem(TreeData):
    """Dataclass representing a tree item in the GUI."""

    # Class-level paths/icons
    current_file_directory = Path(__file__).parent
    two_levels_up_directory = current_file_directory.parent.parent.parent
    icon_folder = two_levels_up_directory / 'core_data' / 'resources' / 'icons'

    # Back-compat input alias for external callers
    tree_item_uuid: Optional[UUID] = field(default=None, repr=False, compare=False)

    # Identity and relations
    uuid: UUID = field(default_factory=uuid4)
    name: str = ''
    item_class: Optional[str] = None
    parent: InitVar[Any] = None
    parent_uuid: Optional[UUID] = None

    # Presentation and state
    icon: Any = None
    is_checked: bool = False
    is_partially_checked: bool = False
    is_enabled: bool = True
    is_expanded: bool = False
    is_selected: bool = False
    tool_tip: Optional[str] = None

    # Item and project IDs
    item_uuid: Optional[UUID] = None
    project_uuid: Optional[UUID] = None

    # Children/drag-drop
    children: List[Any] = field(default_factory=list)
    accepted_drop_types: List[Any] = field(default_factory=list)
    is_visible: bool = True

    def __post_init__(self, parent):
        # Call base initializer to keep behavior consistent with previous code
        super().__init__(icon=self.icon, is_checked=self.is_checked, is_expanded=self.is_expanded,
                         is_selected=self.is_selected, parent_uuid=self.parent_uuid)
        # Preserve legacy UUID precedence: tree_item_uuid > item_uuid > generated
        if self.tree_item_uuid is not None:
            self.uuid = self.tree_item_uuid
            self.tree_item_uuid = None
        elif self.item_uuid is not None:
            self.uuid = self.item_uuid

        # Store parent as a non-dataclass attribute to avoid asdict recursion
        object.__setattr__(self, 'parent', parent)

    def parent_to_uuid_recursive(self, clear_parent: bool = False) -> None:
        """Propagate and set parent_uuid for the subtree, optionally clearing parent refs.

        - Sets each child's `parent_uuid` to this node's `uuid` if not already set.
        - Recurses into children.
        - If `clear_parent` is True, sets `child.parent = None` to avoid cyclic refs.
        """
        for child in self.children:
            # Set child's parent_uuid from this node
            if getattr(child, 'parent_uuid', None) is None:
                child.parent_uuid = self.uuid
            # Optionally clear the back-reference
            if clear_parent and hasattr(child, 'parent'):
                child.parent = None
            # Recurse
            if hasattr(child, 'parent_to_uuid_recursive'):
                child.parent_to_uuid_recursive(clear_parent=clear_parent)


@dataclass
class FolderItem(_TreeItem):
    """Dataclass to organize the data in the project data tree in folders."""

    # Class-level icon paths and names
    file_icon = _TreeItem.icon_folder / 'File_Save.ico'
    settings_icon = _TreeItem.icon_folder / 'GreyGear.ico'
    root_icon = _TreeItem.icon_folder / 'root.ico'
    folder_icon = _TreeItem.icon_folder / 'Folder.ico'
    proposed_icon = _TreeItem.icon_folder / 'BlueFolder.ico'
    existing_icon = _TreeItem.icon_folder / 'RedFolder.ico'

    folder_list = ['root', 'Model', 'Project', 'Folder', 'Proposed', 'Existing', 'File', 'Settings',
                   'Geometry Folder', 'Coverage Folder', 'GIS Folder', 'Model Folder']

    # Folder-specific state
    class_name: str = 'Folder'
    type: str = 'FolderItem'
    is_drop_enabled: bool = True

    # Instance icon references (set in __post_init__)
    model_icon: Any = None
    project_icon: Any = None
    folder_icon_inst: Any = None
    proposed_icon_inst: Any = None
    existing_icon_inst: Any = None

    def __post_init__(self, parent):
        """Finalize FolderItem initialization."""
        # Initialize base, forwarding InitVar
        super().__post_init__(parent)

        # Match previous behavior: default item_uuid to uuid if unset
        if self.item_uuid is None:
            self.item_uuid = self.uuid

        # Set instance icon references from class-level icons
        self.model_icon = FolderItem.root_icon
        self.project_icon = FolderItem.root_icon
        self.folder_icon_inst = FolderItem.folder_icon
        self.proposed_icon_inst = FolderItem.proposed_icon
        self.existing_icon_inst = FolderItem.existing_icon

        # Set the standard icon based on class_name
        self.set_standard_icon(False)

    def set_standard_icon(self, override=True):
        """Sets the icon for the folder."""
        if override is False and self.icon is not None:
            return
        if self.class_name == FolderItem.folder_list[0]:
            self.icon = self.root_icon
        elif self.class_name == FolderItem.folder_list[1]:
            self.icon = self.model_icon
        elif self.class_name == FolderItem.folder_list[2]:
            self.icon = self.project_icon
        elif self.class_name == FolderItem.folder_list[3]:
            self.icon = self.folder_icon
        elif self.class_name == FolderItem.folder_list[4]:
            self.icon = self.proposed_icon
        elif self.class_name == FolderItem.folder_list[5]:
            self.icon = self.existing_icon
        elif self.class_name == FolderItem.folder_list[6]:
            self.icon = self.file_icon
        elif self.class_name == FolderItem.folder_list[7]:
            self.icon = self.settings_icon
        elif self.class_name == FolderItem.folder_list[8:11]:  # Main project folders
            self.icon = self.folder_icon


@dataclass
class Ugrid(_TreeItem):
    """Dataclass holding Ugrid data."""

    # Ugrid-specific fields
    ugrid: Any = None
    filename: Optional[str] = None
    class_name: str = 'Geometry'
    is_drop_enabled: bool = False
    is_drag_enabled: bool = True

    def __post_init__(self, parent):
        """Finalize Ugrid initialization."""
        super().__post_init__(parent)
        # Default icon if not set
        if self.icon is None:
            self.icon = _TreeItem.icon_folder / 'UGrid_Module_Icon.svg'
