"""CalcData for performing Pier Scour calculations."""
__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import copy
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.FhwaVariable.core_data.calculator.calculator_list import CalcOrVarlist
from xms.FhwaVariable.core_data.calculator.table_data import TableData
from xms.FhwaVariable.core_data.calculator.variable_group import VariableGroup
from xms.FhwaVariable.core_data.variables.variable import Variable

# 4. Local modules
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.abutment_scour_data import AbutmentScourData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.contraction_scour_data import ContractionScourData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.long_term_data import LongTermDegradationData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.pier_scour_data import PierScourData
from xms.HydraulicToolboxCalc.hydraulics.bridge_scour.scenario.scenario_calc import ScenarioCalc


class Scenario(VariableGroup):
    """A class that defines a pier scour at a bridge contraction."""

    def __init__(self, clear_my_own_results=True, app_data=None, model_name=None, project_uuid=None):
        """Initializes the GVF calculator.

        Args:
            clear_my_own_results (bool): Should the class clean up its results?  If Standalone, yet. otherwise, no.
            app_data (AppData): The application data (settings).
            model_name (str): The name of the model.
            project_uuid (str): The project UUID.
        """
        super().__init__(app_data=app_data, model_name=model_name, project_uuid=project_uuid)

        self.name = "Bridge Scour Scenario"
        self.type = "BridgeScourScenario"

        self.calculator = ScenarioCalc()

        self.theme = self.get_theme()

        # Input
        # Scenario

        self.input['Compute shear decay'] = Variable('Compute shear decay', 'bool', False)

        wse = {
            'Station': Variable('Station', 'float_list', 0, [0.0], precision=2,
                                limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length),
            'Elevation': Variable('Elevation', 'float_list', 0, [0.0], precision=2,
                                  limits=(-sys.float_info.max, sys.float_info.max), unit_type=['length'],
                                  native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)
        }
        self.input['WSE'] = Variable(
            'WSE', 'table', TableData(self.theme, name='WSE', plot_names=['WSE'], input=wse, min_items=1,
                                      app_data=app_data, model_name=model_name, project_uuid=project_uuid,))

        water_color = self.theme['Plot WSP color']
        water_fill_color = self.theme['Plot WSP fill color']

        self.input['WSE'].get_val().set_plot_series_options(
            'WSE', related_index=0, index=0, name='WSE', x_axis='Station', y_axis='Elevation',
            line_color=water_color, linetype='solid', line_width=1.5, fill_below_line=False,
            fill_color=water_fill_color)

        #   Longterm Degradation
        self.input['Long-term degradation'] = Variable(
            'Long-term degradation', 'calc',
            LongTermDegradationData(clear_my_own_results=False, app_data=app_data, model_name=model_name,
                                    project_uuid=project_uuid))
        self.input['Long-term degradation'].set_row_as_header = True

        #   Contraction Scour
        self.input['Contraction scour'] = {}
        self.input['Contraction scour']['Left overbank contraction scour'] = Variable(
            'Left overbank contraction scour', 'calc',
            ContractionScourData(clear_my_own_results=False, app_data=app_data, model_name=model_name,
                                 project_uuid=project_uuid))
        self.input['Contraction scour']['Main channel contraction scour'] = Variable(
            'Main channel contraction scour', 'calc',
            ContractionScourData(clear_my_own_results=False, app_data=app_data, model_name=model_name,
                                 project_uuid=project_uuid))
        self.input['Contraction scour']['Right overbank contraction scour'] = Variable(
            'Right overbank contraction scour', 'calc',
            ContractionScourData(clear_my_own_results=False, app_data=app_data, model_name=model_name,
                                 project_uuid=project_uuid))
        #   Pier Scour
        self.input['Pier scour'] = Variable(
            'Pier scour', 'calc_list',
            CalcOrVarlist(PierScourData, show_define_btn=True, default_name='Pier scour',
                          default_plural_name='Pier scour', select_one=False, min_number_of_items=0, show_number=False,
                          show_name=False, app_data=app_data, model_name=model_name, project_uuid=project_uuid))
        self.input['Pier scour'].value.input['Number of items'].set_val(0)

        #   Abutment Scour
        self.input['Abutment scour'] = {}
        self.input['Abutment scour']['Left abutment scour'] = Variable(
            'Left abutment scour', 'calc',
            AbutmentScourData(clear_my_own_results=False, app_data=app_data, model_name=model_name,
                              project_uuid=project_uuid))

        self.input['Abutment scour']['Right abutment scour'] = Variable(
            'Right abutment scour', 'calc',
            AbutmentScourData(clear_my_own_results=False, app_data=app_data, model_name=model_name,
                              project_uuid=project_uuid))

        self.banks_specified = False
        self.bridge_geometry = None

        self.unknown = None

        # Intermediate
        # self.compute_prep_functions = []  # Functions to call before compute_data
        # Define variables to copy to the calculator
        self.intermediate_to_copy.extend(
            ['name', 'd50', 'upstream_d50', 'left_channel_bank_station', 'right_channel_bank_station',
             'left_abutment_toe_station', 'right_abutment_toe_station',])  # 'gradations']
        self.theme = self.get_theme()

        self.d50 = 0.0
        self.upstream_d50 = 0.0

        self.left_channel_bank_station = 0.0
        self.right_channel_bank_station = 0.0
        self.left_abutment_toe_station = 0.0
        self.right_abutment_toe_station = 0.0

        self.gradations = None

        # Results
        self.clear_my_own_results = clear_my_own_results

        self.results['Average WSE'] = Variable(
            'Average WSE',
            'float_list',
            0.0, [], limits=(-sys.float_info.max, sys.float_info.max),
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Long-term degradation'] = Variable(
            'Long-term degradation',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Contraction scour'] = {}
        self.results['Contraction scour']['Main channel contraction scour'] = {}
        self.results['Contraction scour']['Main channel contraction scour']['Applied contraction scour depth'] = \
            Variable(
                'Applied contraction scour depth',
                'float_list',
                0.0, [],
                precision=2,
                unit_type=['length'],
                native_unit='ft',
                us_units=self.us_mid_length,
                si_units=self.si_mid_length)

        self.results['Contraction scour']['Main channel contraction scour'][
            'Contraction scour depth and long-term degradation'] = Variable(
                'Contraction scour depth and long-term degradation',
                'float_list',
                0.0, [],
                precision=2,
                unit_type=['length'],
                native_unit='ft',
                us_units=self.us_mid_length,
                si_units=self.si_mid_length)

        self.results['Contraction scour']['Main channel contraction scour']['Clear-water contraction scour depth'] = \
            Variable('Clear-water contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Main channel contraction scour']['Live-bed contraction scour depth'] = \
            Variable('Live-bed contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Main channel contraction scour']['Pressure scour depth'] = Variable(
            'Pressure scour depth', 'float', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Main channel contraction scour']['Streambed thalweg elevation'] = Variable(
            'Streambed thalweg elevation', 'float_list', 0.0, [], precision=2, unit_type=['length'], native_unit='ft',
            us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Main channel contraction scour'][
            'Applied contraction scour elevation with LTD'] = Variable(
                'Applied contraction scour elevation with LTD', 'float', 0.0, [], precision=2, unit_type=['length'],
                native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Left overbank contraction scour'] = {}
        self.results['Contraction scour']['Left overbank contraction scour']['Applied contraction scour depth'] = \
            Variable('Applied contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Left overbank contraction scour']['Clear-water contraction scour depth'] = \
            Variable('Clear-water contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Left overbank contraction scour']['Live-bed contraction scour depth'] = \
            Variable('Live-bed contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Left overbank contraction scour']['Pressure scour depth'] = Variable(
            'Pressure scour depth',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Contraction scour']['Right overbank contraction scour'] = {}
        self.results['Contraction scour']['Right overbank contraction scour']['Applied contraction scour depth'] = \
            Variable('Applied contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Right overbank contraction scour']['Clear-water contraction scour depth'] = \
            Variable('Clear-water contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Right overbank contraction scour']['Live-bed contraction scour depth'] = \
            Variable('Live-bed contraction scour depth', 'float_list', 0.0, [], precision=2, unit_type=['length'],
                     native_unit='ft', us_units=self.us_mid_length, si_units=self.si_mid_length)

        self.results['Contraction scour']['Right overbank contraction scour']['Pressure scour depth'] = Variable(
            'Pressure scour depth',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Pier'] = {}
        self.results['Pier']['Computation method'] = Variable(
            'Computation method',
            "string_list",
            '', [])

        self.results['Pier']['Pier scour depth'] = Variable(
            'Pier scour depth',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Pier']['Max flow depth including pier scour depth'] = Variable(
            'Max flow depth including pier scour depth',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Pier']['Total scour at pier'] = Variable(
            'Total scour at pier',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Pier']['Local streambed elevation at pier'] = Variable(
            'Local streambed elevation at pier',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='local streambed elevation prior to any scour')

        self.results['Pier']['Total scour elevation at pier'] = Variable(
            'Total scour elevation at pier',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Lowest scour elevation'] = Variable(
            'Lowest scour elevation',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour'] = {}
        self.results['Abutment scour']['Left abutment'] = {}
        self.results['Abutment scour']['Left abutment']['Abutment scour depth'] = Variable(
            'Abutment scour depth',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour']['Left abutment']['Max flow depth including abutment scour'] = Variable(
            'Max flow depth including abutment scour',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour']['Left abutment']['Total scour at abutment'] = Variable(
            'Total scour at abutment',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour']['Left abutment']['Local streambed elevation at abutment'] = Variable(
            'Local streambed elevation at abutment',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='local streambed elevation prior to any scour')

        self.results['Abutment scour']['Left abutment']['Total scour elevation at abutment'] = Variable(
            'Total scour elevation at abutment',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour']['Right abutment'] = {}
        self.results['Abutment scour']['Right abutment']['Abutment scour depth'] = Variable(
            'Abutment scour depth',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour']['Right abutment']['Max flow depth including abutment scour'] = Variable(
            'Max flow depth including abutment scour',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour']['Right abutment']['Total scour at abutment'] = Variable(
            'Total scour at abutment',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.results['Abutment scour']['Right abutment']['Local streambed elevation at abutment'] = Variable(
            'Local streambed elevation at abutment',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length,
            note='local streambed elevation prior to any scour')

        self.results['Abutment scour']['Right abutment']['Total scour elevation at abutment'] = Variable(
            'Total scour elevation at abutment',
            'float_list',
            0.0, [],
            precision=2,
            unit_type=['length'],
            native_unit='ft',
            us_units=self.us_mid_length,
            si_units=self.si_mid_length)

        self.warnings = []

        # plot
        # self.plots['profile'] = {}
        # self.plots['profile']['Plot name'] = "Profile"
        # self.plots['profile']['Legend'] = 'best'
        # self.plots['profile']['index'] = 1

    @staticmethod
    def is_left_channel_bank_specified(bridge_geometry):
        """Determines if the left channel bank is specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the left channel bank is specified; otherwise, False
        """
        channel_left_bank_station = bridge_geometry.input['Channel left bank station'].get_val()

        return ScenarioCalc._is_left_channel_bank_specified(channel_left_bank_station)

    @staticmethod
    def is_right_channel_bank_specified(bridge_geometry):
        """Determines if the left channel bank is specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the left channel bank is specified; otherwise, False
        """
        channel_right_bank_station = bridge_geometry.input['Channel right bank station'].get_val()

        return ScenarioCalc._is_right_channel_bank_specified(channel_right_bank_station)

    @staticmethod
    def determine_if_banks_are_specified(bridge_geometry):
        """Determines if the channel banks are specified.

        Args:
            bridge_geometry (BridgeGeometry): The bridge geometry.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        channel_left_bank_station = bridge_geometry.input['Channel left bank station'].get_val()
        channel_right_bank_station = bridge_geometry.input['Channel right bank station'].get_val()

        return ScenarioCalc._determine_if_banks_are_specified(channel_left_bank_station,
                                                              channel_right_bank_station)

    @staticmethod
    def is_left_abutments_specified(bridge_geometry):
        """Determines if the channel banks are specified.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        left_abutment_station = bridge_geometry.input['Left abutment toe station'].get_val()
        return ScenarioCalc._is_left_abutments_specified(left_abutment_station)

    @staticmethod
    def is_right_abutments_specified(bridge_geometry):
        """Determines if the channel banks are specified.

        Returns:
            bool: True if the channel banks are specified; otherwise, False
        """
        left_abutment_station = bridge_geometry.input['Left abutment toe station'].get_val()
        right_abutment_station = bridge_geometry.input['Right abutment toe station'].get_val()
        return ScenarioCalc._is_right_abutments_specified(left_abutment_station, right_abutment_station)

    def get_input_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and left out of the input dictionary)

        Returns:
              input_vars (dictionary of variables): the input variables
        """
        input_vars = {}

        input_vars = copy.deepcopy(self.input)

        if self.bridge_geometry is not None:
            left_bank_specified = Scenario.is_left_channel_bank_specified(self.bridge_geometry)
            right_bank_specified = Scenario.is_right_channel_bank_specified(self.bridge_geometry)
            left_abutment_specified = Scenario.is_left_abutments_specified(self.bridge_geometry)
            right_abutment_specified = Scenario.is_right_abutments_specified(self.bridge_geometry)

            if not left_bank_specified:
                input_vars['Contraction scour'].pop('Left overbank contraction scour')
            if not right_bank_specified:
                input_vars['Contraction scour'].pop('Right overbank contraction scour')
            if not left_abutment_specified and not right_abutment_specified:
                input_vars.pop('Abutment scour')
            elif not left_abutment_specified:
                input_vars['Abutment scour'].pop('Left abutment scour')
            elif not right_abutment_specified:
                input_vars['Abutment scour'].pop('Right abutment scour')

        if self.gradations.input['Contracted gradation input options'].get_val() == 'Single D50':
            self.input['Compute shear decay'].set_val(False)
            input_vars.pop('Compute shear decay')

        if self.clear_my_own_results:
            # This means we are a stand alone calculator and we pop the gradation selections
            pass
        else:
            pass

        return input_vars

    def get_results_tab_group(self, unknown=None):
        """Returns a dictionary of input variables that are needed for current selections.

        Args:
            unknown (string): the variable that is unknown (and included in the result dictionary)

        Returns:
              result_vars (dictionary of variables): the input variables
        """
        results_vars = {}

        # basic_results = {}
        # complete_results = {}

        if not self.get_can_compute():
            return {'Results': results_vars}

        # basic_results['Amplification factor (α)'] = self.results['Amplification factor (α)']
        # complete_results['Amplification factor (α)'] = self.results['Amplification factor (α)']
        # complete_results['q2 / q1'] = self.results['q2 / q1']
        # complete_results['Governing scour method'] = self.results['Governing scour method']

        # basic_results['Flow depth including scour (yc)'] = self.results['Flow depth including scour (yc)']
        # complete_results['Flow depth including scour (yc)'] = self.results['Flow depth including scour (yc)']

        # basic_results['Max flow depth including scour (ymax)'] =
        # self.results['Max flow depth including scour (ymax)']
        # complete_results['Max flow depth including scour (ymax)'] =
        # self.results['Max flow depth including scour (ymax)']

        # contraction_method = self.results['Governing scour method'].get_val()
        # if contraction_method == 'Clear water':
        #     complete_results['Flow depth including scour using clear water method (yc)'] =
        # self.results['Flow depth including scour using clear water method (yc)']
        #     complete_results['Max flow depth including scour using clear water method (ymax)'] =
        # self.results['Max flow depth including scour using clear water method (ymax)']

        # longterm_degradation = 0.0
        # if not self.stand_alone_calc:
        #     self.results['longterm degradation'].set_val(longterm_degradation)

        # basic_results['Scour hole depth (yc)'] = self.results['Scour hole depth (yc)']
        # complete_results['Scour hole depth (yc)'] = self.results['Scour hole depth (yc)']

        # results_vars['Basic Results'] = basic_results
        # results_vars['Complete Results'] = complete_results

        return results_vars

    # def assign_results(self):
    #     """Assigns the results from a computational run to the appropriate locations."""
    #     # Assign the results
    #     _, gamma = self.get_setting('Unit weight of water (γw)')
