"""Perform model checks on an SMS ADCIRC simulation."""

# 1. Standard Python modules
import datetime
import os

# 2. Third party modules
import numpy as np

# 3. Aquaveo modules
from xms.api.dmi import ModelCheckError
from xms.api.tree.tree_util import find_tree_node_by_uuid
from xms.constraint import read_grid_from_file
from xms.core.filesystem import filesystem as io_util
from xms.data_objects.parameters import julian_to_datetime
from xms.grid.ugrid import UGrid
from xms.guipy.time_format import ISO_DATETIME_FORMAT

# 4. Local modules
from xms.adcirc.dmi.model_check_queries import ModelCheckQueries
from xms.adcirc.file_io.grid_crc import compute_grid_crc


def grid_files_match(filename, base_crc):
    """Check if a CoGrid file matches a base CoGrid file CRC.

    Args:
        filename (:obj:`str`): Path to the CoGrid file to compare to the base
        base_crc (:obj:`str`): CRC of the base CoGrid file to compare to

    Returns:
        (:obj:`bool`): True if the CRCs match, False otherwise
    """
    current_crc = compute_grid_crc(filename)
    return base_crc == current_crc


class ADCIRCModelChecker:
    """Runs model checks on an ADCIRC simulation."""
    def __init__(self, xms_data=None):
        """Construct the model checker.

        Args:
            xms_data (:obj:`dict`): Dictionary containing all the data from XMS needed to run model checks. Useful for
                testing. If not provided, XMS will be queried for the data.
                    ::
                        {
                        'sim_data': SimData,
                        'mapped_bc_data': MappedBcData,
                        'projection': str,
                        'horizontal_units': str,
                        'vertical_units': str,
                        'global_time': datetime.datetime,
                        'tidal_data': [MappedTidalData],  # [] if no mapped periodic tidal boundaries
                        'flow_data': MappedFlowData,  # None if no mapped periodic flow boundaries
                        'wind_dsets': dict,  # Key is dataset description, value is data_objects.parameters.Dataset
                        'wind_cov': xms.coverage._windCoverage.WindCoverage,  # Wind coverage dump
                        'current_cogrid': str,  # Filename of the cogrid dump for the current domain mesh
                        'pe_tree': TreeNode,
                        'station_uuid': str
                    }

        """
        self._errors = []
        self._query_helper = None
        self._sim_query_helper = None
        self._run_start_date = None
        self._run_end_date = None  # Timestamp computed from interpolation reference date and ADCIRC run duration.
        self._has_flow_boundaries = False
        self._xms_data = xms_data
        if not self._xms_data:
            self._query_helper = ModelCheckQueries()
            self._xms_data, errors = self._query_helper.get_xms_model_check_data()
            for error in errors:  # Something went wrong getting the data we want to check, propagate errors up to user.
                self._append_error(error[0], error[1], error[2])

    def _append_error(self, problem, description, fix):
        """Append a model check error to the list to be sent back to XMS.

        Args:
            problem (:obj:`str`): The problem text
            description (:obj:`str`): The description text
            fix (:obj:`str`): The fix text
        """
        error = ModelCheckError(problem=problem, description=description, fix=fix)
        self._errors.append(error)

    def _check_mesh(self):
        """Ensure the domain mesh has not been edited since components were mapped."""
        if self._xms_data['current_cogrid']:  # Check if the mesh geometry has changed since mapping
            if not grid_files_match(
                self._xms_data['current_cogrid'], self._xms_data['mapped_bc_data'].info.attrs['grid_crc']
            ):
                self._append_error(
                    'Applied simulation components may be out of date with the ADCIRC domain mesh.',
                    'The ADCIRC domain mesh has been edited after applying simulation components. Any applied items '
                    'under the simulation may not be consistent/valid with the current mesh.',
                    'Reapply the source Boundary Conditions coverage (if it exists) and then any applicable tidal '
                    'constituent components.'
                )
            else:  # Read the CoGrid to do additional checking
                cogrid = read_grid_from_file(self._xms_data['current_cogrid'])
                if not cogrid.check_all_cells_are_of_type(UGrid.cell_type_enum.TRIANGLE):
                    self._append_error(
                        'Non-triangular elements detected in the ADCIRC domain mesh.',
                        'ADCIRC requires all elements in the domain to be triangles.',
                        'Regenerate or modify the domain mesh so that it only contains triangles. Then reapply the '
                        'source Boundary Conditions coverage (if it exists) and then any applicable tidal constituents.'
                    )

    def _check_simulation(self):
        """Ensure run description and ID has been specified."""
        if not self._xms_data['sim_data'].general.attrs['RUNDES']:
            self._append_error(
                'No run description has been specified.',
                'ADCIRC requires an alphanumeric run description with a maximum length of 32 characters.',
                'Right-click on the simulation and select the "Model Control..." menu item. In the "Simulation '
                'description" section of the "General" tab, enter a value for "Project title".'
            )

        if not self._xms_data['sim_data'].general.attrs['RUNID']:
            self._append_error(
                'No run id has been specified.',
                'ADCIRC requires an alphanumeric run identifier with a maximum length of 24 characters.',
                'Right-click on the simulation and select the "Model Control..." menu item. In the "Simulation '
                'description" section of the "General" tab, enter a value for "Run ID".'
            )

    def _check_tidal_constituents(self):
        """Check the applied tidal constituent data."""
        if self._xms_data['tidal_data']:  # Have applied tidal constituents
            # Ensure that each applied tidal forcing/potential constituent has only been defined once.
            all_cons = set()
            reported_cons = set()
            for tidal_data in self._xms_data['tidal_data']:
                con_names = tidal_data.cons.con.data.tolist()
                for con_name in con_names:
                    if con_name not in all_cons:
                        all_cons.add(con_name)
                    elif con_name not in reported_cons:  # Only show message per duplicate constituent.
                        reported_cons.add(con_name)
                        self._append_error(
                            'The same tidal forcing/potential constituent has multiple applied definitions. Only one '
                            'definition per constituent is allowed.',
                            f'The constituent named "{con_name}" has been applied to the simulation multiple times.',
                            'For each applied tidal constituent item under the simulation, right-click on the item and '
                            'select "Tidal Constituents..." to inspect the constituents defined on the item. Ensure '
                            f'that "{con_name}" only appears in one of the applied items. Remove any items with a '
                            f'duplicate definition, reapplying other valid constituents defined in those items as '
                            f'needed.'
                        )

            # Warn if BC coverage was mapped with non-periodic elevation forcing
            if not self._xms_data['mapped_bc_data'].source_data.info.attrs['periodic_tidal']:
                self._append_error(
                    'WARNING! Incompatible elevation forcing options detected.',
                    'Periodic tidal forcing constituents have been applied, but the option to use non-periodic '
                    'elevation forcing was enabled in the applied Boundary Conditions coverage.',
                    'The non-periodic elevation forcing specification will be ignored on export. To use non-periodic '
                    'elevation forcing, remove all applied tidal constituent items from the simulation'
                )
        else:  # No applied tidal constituents.
            # Check if the mapped BC has an ocean boundary.
            ocean_nodes, _ = self._xms_data['mapped_bc_data'].get_ocean_node_ids()
            if ocean_nodes:
                if self._xms_data['mapped_bc_data'].source_data.info.attrs['periodic_tidal']:  # Periodic option enabled
                    self._append_error(
                        'ERROR! Boundary condition specification is incomplete.',
                        'Periodic elevation forcing has been enabled, but no tidal constituents have been applied.',
                        'Right-click in the project explorer pane and select "New Simulation" -> "Tidal '
                        'Constituents". Right-click on the new tidal constituent item and select "Edit Constituents".'
                        ' Choose the desired extraction source for "Source". Set "Reference time" to be the beginning '
                        f'of the ADCIRC run ({self._xms_data["sim_data"].timing.attrs["ref_date"]}). Choose the desired'
                        ' constituents and click "OK". Drag the tidal constituent item under the simulation to apply '
                        'the constituents to the ocean boundary node locations.'
                    )
                else:  # Non-periodic elevation forcing.
                    fort19 = self._xms_data['mapped_bc_data'].source_data.info.attrs['fort.19']
                    if not os.path.isfile(fort19):  # Check for selected fort.19 file
                        self._append_error(
                            'ERROR! Boundary condition specification is incomplete.',
                            'Non-periodic elevation forcing has been enabled, but no fort.19 file has been selected.',
                            'Right-click on the source Boundary Conditions coverage used to apply boundaries to the '
                            'simulation and select "Forcing Options...". If the source Boundary Conditions coverage '
                            'does not exist, right-click on the applied Boundary Conditions item and select "Convert to'
                            ' Coverage". In the "Boundary Conditions Forcing Options" dialog, push the "Select..." '
                            'button in the "Tidal forcing" section. Choose a valid fort.19 elevation forcing data set '
                            'file for the run. Reapply the Boundary Conditions coverage by dragging its item under the '
                            'simulation.'
                        )

    def _check_flow_constituents(self):
        """Check the applied flow constituent data."""
        if not self._has_flow_boundaries:  # Don't do checks if no flow boundaries
            return

        if self._xms_data['flow_data']:  # Have applied flow constituents
            # Ensure that each applied flow forcing constituent has only been defined once.
            all_cons = set()
            reported_cons = set()
            con_names = self._xms_data['flow_data'].cons.con.data.tolist()
            for con_name in con_names:
                if con_name not in all_cons:
                    all_cons.add(con_name)
                elif con_name not in reported_cons:  # Only show message per duplicate constituent.
                    reported_cons.add(con_name)
                    self._append_error(
                        'The same flow forcing constituent has multiple applied definitions. Only one definition per '
                        'constituent is allowed.',
                        f'The flow constituent named "{con_name}" has been applied to the simulation multiple times.',
                        'Right-click on the source Boundary Conditions coverage used to apply boundaries to the '
                        'simulation and select "Forcing Options...". If the source Boundary Conditions coverage '
                        'does not exist, right-click on the applied Boundary Conditions item and select "Convert to '
                        'Coverage". In the "Boundary Conditions Forcing Options" dialog, inspect the constituent table '
                        'in the "Flow forcing" section. Ensure that there is only one definition for the constituent '
                        f'named "{con_name}". Reapply the Boundary Conditions coverage by dragging its item under the '
                        'simulation.'
                    )
        else:  # No applied flow forcing component
            if self._xms_data['mapped_bc_data'].source_data.info.attrs['periodic_flow']:  # Periodic option enabled
                self._append_error(
                    'ERROR! Boundary condition specification is incomplete.',
                    'Periodic flow forcing has been enabled, but no flow constituents have been applied.',
                    'Right-click on the source Boundary Conditions coverage used to apply boundaries to the simulation '
                    'and select "Forcing Options...". If the source Boundary Conditions coverage does not exist, '
                    'right-click on the applied Boundary Conditions item and select "Convert to Coverage". In the '
                    '"Boundary Conditions Forcing Options" dialog, define one or more flow constituents in the "Flow '
                    'forcing" section. Reapply the Boundary Conditions coverage by dragging its item under the '
                    'simulation.'
                )
            # else:  # Non-periodic flow forcing.
            #     fort20 = self._xms_data['mapped_bc_data'].source_data.info.attrs['fort.20']
            #     if not os.path.isfile(fort20):  # Check for selected fort.19 file
            #         self._append_error(
            #             'ERROR! Boundary condition specification is incomplete.',
            #             'Non-periodic flow forcing has been enabled, but no fort.20 file has been selected.',
            #             'Right-click on the source Boundary Conditions coverage used to apply boundaries to the '
            #             'simulation and select "Forcing Options...". If the source Boundary Conditions coverage '
            #             'does not exist, right-click on the applied Boundary Conditions item and select "Convert to'
            #             ' Coverage". In the "Boundary Conditions Forcing Options" dialog, push the "Select..." '
            #            'button in the "Tidal forcing" section. Choose a valid fort.20 flow forcing data set file for '
            #            'the run. Reapply the Boundary Conditions coverage by dragging its item under the simulation.'
            #         )

    def _check_hotstart_files(self):
        """Check that hot start file selections are valid if append option enabled."""
        sim_data = self._xms_data['sim_data']
        ihot_type = sim_data.general.attrs['IHOT']
        if ihot_type != 0:
            proj_dir = sim_data.info.attrs['proj_dir']  # Stored paths will be relative to project if saved.

            # Check the main hot start file
            if not os.path.isfile(os.path.join(proj_dir, sim_data.general.attrs['IHOT_file'])):
                file_desc = "ASCII fort.17"
                if ihot_type == 67:
                    file_desc = "binary fort.67"
                elif ihot_type == 68:
                    file_desc = "binary fort.68"
                elif ihot_type in [367, 567]:
                    file_desc = "NetCDF fort.67.nc"
                elif ihot_type in [368, 569]:
                    file_desc = "NetCDF fort.68.nc"
                self._append_error(
                    'The option to run this simulation as a hot start has been specified, but no hot start file has '
                    'been selected.', 'A hot start file from a previous run must be selected for hot start runs.',
                    'Right-click on the simulation and select the "Model Control..." menu item. In the "Run '
                    f'options" section of the "General" tab, push the "Select" button to choose a valid {file_desc} '
                    'hot start file.'
                )

            # check the NOUTGE fort.63 file
            global_elev = sim_data.output.sel(ParamName='NOUTGE')
            global_elev_format = global_elev['Output'].item()
            if global_elev_format != 0 and global_elev['File'].item() == 0:
                # Output enabled and append to hot start option on - check selected file.
                if not os.path.isfile(os.path.join(proj_dir, global_elev['Hot Start'].item())):
                    file_desc = "ASCII fort.63"
                    if global_elev_format == 2:
                        file_desc = "binary fort.63"
                    elif global_elev_format == 3:
                        file_desc = "NetCDF fort.63"
                    self._append_error(
                        'The option to append global elevation output to a hot start file has been specified, but no '
                        f'{file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the "Hot start" '
                        'column (on the "Global elevation" row) of the "Output" tab, push the button to choose a valid '
                        f'{file_desc} hot start file.'
                    )

            # check the NOUTGV fort.64 file
            global_vel = sim_data.output.sel(ParamName='NOUTGV')
            global_vel_format = global_vel['Output'].item()
            if global_vel_format != 0 and global_vel['File'].item() == 0:
                # Output enabled and append to hot start option on - check selected file.
                if not os.path.isfile(os.path.join(proj_dir, global_vel['Hot Start'].item())):
                    file_desc = "ASCII fort.64"
                    if global_vel_format == 2:
                        file_desc = "binary fort.64"
                    elif global_vel_format == 3:
                        file_desc = "NetCDF fort.64"
                    self._append_error(
                        'The option to append global velocity output to a hot start file has been specified, but no '
                        f'{file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the "Hot start" '
                        'column (on the "Global velocity" row) of the "Output" tab, push the button to choose a valid '
                        f'{file_desc} hot start file.'
                    )

            # check the NOUTGW fort.73 and fort.74 files
            global_wind = sim_data.output.sel(ParamName='NOUTGW')
            global_wind_format = global_wind['Output'].item()
            if global_wind_format != 0 and global_wind['File'].item() == 0 and sim_data.wind.attrs['NWS'] != 0:
                # Output enabled and append to hot start option on - check selected files.
                if not os.path.isfile(os.path.join(proj_dir, global_wind['Hot Start'].item())):
                    file_desc = "ASCII fort.73"
                    if global_wind_format == 2:
                        file_desc = "binary fort.73"
                    elif global_wind_format == 3:
                        file_desc = "NetCDF fort.73"
                    self._append_error(
                        'The option to append global meteorological output to a hot start file has been specified, but '
                        f'no {file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the "Hot start" '
                        'column (on the "Global wind" row) of the "Output" tab, push the button to choose a valid '
                        f'{file_desc} hot start file.'
                    )

                if not os.path.isfile(os.path.join(proj_dir, global_wind['Hot Start (Wind Only)'].item())):
                    file_desc = "ASCII fort.74"
                    if global_wind_format == 2:
                        file_desc = "binary fort.74"
                    elif global_wind_format == 3:
                        file_desc = "NetCDF fort.74"
                    self._append_error(
                        'The option to append global meteorological output to a hot start file has been specified, but '
                        f'no {file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the '
                        '"Hot start (Wind Only)" column (on the "Global wind" row) of the "Output" tab, push the '
                        f'button to choose a valid {file_desc} hot start file.'
                    )

            # check the NOUTE fort.61 file
            station_elev = sim_data.output.sel(ParamName='NOUTE')
            station_elev_format = station_elev['Output'].item()
            if station_elev_format != 0 and station_elev['File'].item() == 0:
                # Output enabled and append to hot start option on - check selected file.
                if not os.path.isfile(os.path.join(proj_dir, station_elev['Hot Start'].item())):
                    file_desc = "ASCII fort.61"
                    if station_elev_format == 2:
                        file_desc = "binary fort.61"
                    elif station_elev_format == 3:
                        file_desc = "NetCDF fort.61"
                    self._append_error(
                        'The option to append station elevation output to a hot start file has been specified, but no '
                        f'{file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the "Hot start" '
                        'column (on the "Station elevation" row) of the "Output" tab, push the button to choose a '
                        f'valid {file_desc} hot start file.'
                    )

            # check the NOUTV fort.62 file
            station_vel = sim_data.output.sel(ParamName='NOUTV')
            station_vel_format = station_vel['Output'].item()
            if station_vel_format != 0 and station_vel['File'].item() == 0:
                # Output enabled and append to hot start option on - check selected file.
                if not os.path.isfile(os.path.join(proj_dir, station_vel['Hot Start'].item())):
                    file_desc = "ASCII fort.62"
                    if station_vel_format == 2:
                        file_desc = "binary fort.62"
                    elif station_vel_format == 3:
                        file_desc = "NetCDF fort.62"
                    self._append_error(
                        'The option to append station velocity output to a hot start file has been specified, but no '
                        f'{file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the "Hot start" '
                        'column (on the "Station velocity" row) of the "Output" tab, push the button to choose a valid '
                        f'{file_desc} hot start file.'
                    )

            # check the NOUTM fort.71 and fort.72 files
            station_wind = sim_data.output.sel(ParamName='NOUTW')
            station_wind_format = station_wind['Output'].item()
            if station_wind_format != 0 and station_wind['File'].item() == 0 and sim_data.wind.attrs['NWS'] != 0:
                # Output enabled and append to hot start option on - check selected files.
                if not os.path.isfile(os.path.join(proj_dir, station_wind['Hot Start'].item())):
                    file_desc = "ASCII fort.71"
                    if station_wind_format == 2:
                        file_desc = "binary fort.71"
                    elif station_wind_format == 3:
                        file_desc = "NetCDF fort.71"
                    self._append_error(
                        'The option to append station meteorological output to a hot start file has been specified, '
                        f'but no {file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the "Hot start" '
                        'column (on the "Station wind" row) of the "Output" tab, push the button to choose a valid '
                        f'{file_desc} hot start file.'
                    )

                if not os.path.isfile(os.path.join(proj_dir, station_wind['Hot Start (Wind Only)'].item())):
                    file_desc = "ASCII fort.72"
                    if station_wind_format == 2:
                        file_desc = "binary fort.72"
                    elif station_wind_format == 3:
                        file_desc = "NetCDF fort.72"
                    self._append_error(
                        'The option to append station meteorological output to a hot start file has been specified, but'
                        f' no {file_desc} has been selected.',
                        'A hot start file from a previous run must be selected to append to.',
                        'Right-click on the simulation and select the "Model Control..." menu item. In the '
                        '"Hot start (Wind Only)" column (on the "Station wind" row) of the "Output" tab, push the '
                        f'button to choose a valid {file_desc} hot start file.'
                    )

    def _check_projection_units(self):
        """Ensure the horizontal and vertical projection units are not in feet."""
        if 'FEET' in self._xms_data['horizontal_units'] or 'FEET' in self._xms_data['vertical_units']:
            self._append_error(
                "STOP! Don't continue with this simulation. Projection units incorrect.",
                'Both horizontal and vertical projection units must be either meters or geographic units.',
                'In the top-level "Display" menu, select the "Display Projection..." menu item. Ensure that the units '
                'option in the "Vertical" section is set to "Meters". If using a non-global projection, also verify '
                'that the units option in the "Horizontal" section is set to "Meters". Remove any applied boundary '
                'condition or tidal constituent items from the simulation and reapply after editing the display '
                'projection.'
            )

    def _check_times(self):
        """Warn user if input dataset time steps do not overlap the ADCIRC run at all."""
        # Build a time duration object representing the length of the ADCIRC run.
        run_duration = datetime.timedelta(days=self._xms_data['sim_data'].timing.attrs['RUNDAY'])
        # Add the ADCIRC run time duration to the ADCIRC reference date timestamp.
        self._run_start_date = datetime.datetime.strptime(
            self._xms_data['sim_data'].timing.attrs['ref_date'], ISO_DATETIME_FORMAT
        )
        self._run_end_date = self._run_start_date + run_duration

        for dset_description, dset in self._xms_data['wind_dsets'].items():
            # Use XMS global zero time if no reference date defined in the dataset.
            ref_time = self._xms_data['global_time'] if dset.ref_time is None else dset.ref_time
            # Get the datetime of the dataset's first time step
            dset_start_time = ref_time + dset.timestep_offset(0)
            # Get the datetime of the dataset's last time step
            dset_end_time = ref_time + dset.timestep_offset(-1)

            if dset_start_time > self._run_start_date or dset_end_time < self._run_end_date:
                self._append_error(
                    f'WARNING - The time steps of the wind input dataset for {dset_description} ({dset.name}) '
                    'do not completely cover the ADCIRC run.',
                    'Wind data must be provided for the entire ADCIRC run. Missing ADCIRC wind time steps will be '
                    'extrapolated from the nearest data set time step when exporting. If using a hotstart, ensure the '
                    'simulation reference matches the hotstart time.',
                    f'Right-click on the simulation and select the "Model Control..." menu item. In the "Wind" tab '
                    f'under "Option - NWS", push the data set selector button for {dset_description} to choose a '
                    f'valid dataset.\n\nAlternatively, change the reference time of the dataset by right-clicking on '
                    f'the dataset in the project explorer and choosing "Time units and reference...".'
                )

    def _check_wind(self):
        """Check various NWS related values."""
        sim_data = self._xms_data['sim_data']
        # If the simulation uses "Use existing wind file" option, make sure they selected a file.
        existing_file = False
        if self._xms_data['sim_data'].wind.attrs['use_existing']:
            existing_file = True
            if not os.path.isfile(os.path.join(sim_data.info.attrs['proj_dir'], sim_data.wind.attrs['existing_file'])):
                self._append_error(
                    'STOP! No fort.22 wind file has been selected.',
                    'The option for using an existing wind file has been enabled, but no fort.22 has been selected.',
                    'Right-click on the simulation and select the "Model Control..." menu item. In the "Option - NWS" '
                    'section of the "Wind" tab, push the "Select" button to choose a valid fort.22 wind file.'
                )

        # Do storm track coverage checks
        nws = sim_data.wind.attrs['NWS']
        if nws in [8, 19, 20]:
            if not existing_file and (not self._xms_data['wind_cov'] or not self._xms_data['wind_cov'].m_nodeWind):
                self._append_error(
                    'No storm track coverage has been linked to the simulation.',
                    'If NWS is equal to 8, 19, or 20, a storm track coverage must be linked to the simulation.',
                    'To create a storm track coverage, right-click on "Map Data" in the project explorer and choose '
                    'the "New Coverage" menu item. In the dialog, select a "Wind" coverage type and click "OK". '
                    'In the project explorer, drag the new coverage under this simulation.\n\nOR\n\nRight-click on the '
                    'simulation and select the "Model Control..." menu item. In the "Wind" tab, choose an NWS '
                    'option that is neither "NWS =    8 - Symmetric cyclonic storm from path", "NWS =   19 - '
                    'Asymmetric cyclonic storm from path", nor "NWS =   20 - GAHM Format cyclonic storm path".'
                )
            elif not existing_file:  # Have a wind coverage, check the times
                storm_start = julian_to_datetime(self._xms_data['wind_cov'].m_nodeWind[0].m_date)
                storm_end = julian_to_datetime(self._xms_data['wind_cov'].m_nodeWind[-1].m_date)

                if nws == 19:  # If NWS=19, storm start time must be same as cold start time
                    year_match = storm_start.year == self._run_start_date.year
                    month_match = storm_start.month == self._run_start_date.month
                    day_match = storm_start.day == self._run_start_date.day
                    hour_match = storm_start.hour == self._run_start_date.hour
                    if not year_match or not month_match or not day_match or not hour_match:
                        self._append_error(
                            'The storm track coverage start time does not match the ADCIRC run start time.',
                            "If NWS=19, the storm track coverage's start time must match the simulation start time.",
                            'Double-click on any of the nodes in the storm track coverage to access the "Storm Track '
                            'Node Attributes" dialog. Adjust the "Storm start time" so that it is equal to '
                            f'{self._run_start_date} to the hour.\n\nOR\n\nRight-click on the '
                            f'simulation and select the "Model Control..." menu item. In the "Timing" tab, adjust the '
                            f'"Interpolation reference date" so that it is equal to {storm_start} '
                            f'to the hour.'
                        )
                elif storm_start > self._run_start_date:  # Beginning of storm is after model run
                    self._append_error(
                        'The storm track coverage time span begins after the end of the ADCIRC run.',
                        'All of the nodes of the storm track have dates after the last ADCIRC time step.',
                        'Double-click on any of the nodes in the storm track coverage to access the "Storm Track '
                        'Node Attributes" dialog. Adjust the "Storm start time" so that it is on or before '
                        f'{self._run_start_date} to the hour.\n\nOR\n\nRight-click on the '
                        f'simulation and select the "Model Control..." menu item. In the "Timing" tab, adjust the '
                        f'"Interpolation reference date" so that it is after {storm_start}.'
                    )

                if self._run_end_date > storm_end:
                    self._append_error(
                        'The storm track coverage time span does not cover the ADCIRC run time span.',
                        'The storm end time is before the simulation end time.',
                        'Double-click on any of the nodes in the storm track coverage to access the "Storm Track Node '
                        'Attributes" dialog. Adjust the "Time offset (hours)" row for the last node so that it is '
                        f'after {self._run_end_date}.\n\nOR\n\nRight-click on the simulation and '
                        'select the "Model Control..." menu item. In the "Timing" tab, adjust the "Interpolation '
                        'reference date" and the "Length of run (days)" so that the "Interpolation reference date" '
                        f'plus "Length of run (days)" is on or before {storm_end}.'
                    )

                # Make sure the storm track does not cross a calendar year boundary.
                if storm_start.year != storm_end.year:
                    self._append_error(
                        'The wind coverage dates cross the boundary of a calendar year.',
                        f'The "{self._xms_data["wind_cov"].m_cov.name}" coverage\'s dates cross the boundary of '
                        'a calendar year. ADCIRC does not allow this.',
                        'Select the wind coverage. With the "Select Feature Point" tool, double-click anywhere in '
                        'the coverage. Modify the dates so that they do not span multiple years.'
                    )
            elif self._xms_data["wind_cov"]:
                # Have a wind coverage linked, but also have the "Use existing wind file" option enabled.
                self._append_error(
                    'NWS wind file "Model Control" option is inconsistent with linked wind coverage.',
                    'A storm track coverage has been linked to the simulation, but the "Use existing wind file" option '
                    'has been enabled.',
                    'Right-click on the simulation\'s linked wind coverage and select "Remove".\n\nOR\n\nRight-click '
                    'on the simulation and select the "Model Control..." menu item. In the "Wind" tab, disable the '
                    '"Use existing wind file" toggle option.'
                )
        elif self._xms_data["wind_cov"]:
            # User linked a wind coverage, but is using a non-wind track NWS type.
            self._append_error(
                'NWS "Model Control" option is inconsistent with linked wind coverage.',
                'A storm track coverage has been linked to the simulation, but the NWS wind option is not set to '
                'NWS = 8, 19, or 20.',
                'Right-click on the simulation\'s linked wind coverage and select "Remove".\n\nOR\n\nRight-click on the'
                ' simulation and select the "Model Control..." menu item. In the "Wind" tab, choose an NWS '
                'option that is either "NWS =    8 - Symmetric cyclonic storm from path", "NWS =   19 - '
                'Asymmetric cyclonic storm from path", or "NWS =   20 - GAHM Format cyclonic storm path".'
            )
        self._check_wind_files()
        self._check_wind_grid()

    def _check_wind_files(self):
        """Check wind files, if there, and if relative or absolute."""
        nws = self._xms_data['sim_data'].wind.attrs['NWS']
        proj_dir = self._xms_data['sim_data'].info.attrs['proj_dir']
        if nws in [10, 11, 12, 15, 16]:
            if nws == 10:
                # Check if we have any NWS files specified
                if self._xms_data['sim_data'].nws10_files.sizes['dim_0'] == 0:
                    self._append_error(
                        'NWS 10 File Error', 'Could not find any NWS 10 files.',
                        'Make sure the files exist and have been specified in the Model Control.'
                    )
                else:
                    # We have files, check if they exist
                    nws_files = self._xms_data['sim_data'].nws10_files['AVN File'].data.tolist()
                    for nws_file in nws_files:
                        if proj_dir == '':
                            # Haven't saved a project yet, so need to have an absolute path
                            cur_file = nws_file
                        else:
                            # We have a project saved, so check via a relative path
                            cur_file = io_util.resolve_relative_path(proj_dir, nws_file)
                        if not os.path.isfile(cur_file):
                            self._append_error(
                                'NWS 10 File Error', f'Could not find NWS 10 AVN file:  {nws_file}',
                                'Ensure that the file path is correct in Model Control.'
                            )
            elif nws == 11:
                # Check if we have any NWS files specified
                if self._xms_data['sim_data'].nws11_files.sizes['dim_0'] == 0:
                    self._append_error(
                        'NWS 11 File Error', 'Could not find any NWS 11 files.',
                        'Make sure the files exist and have been specified in the Model Control.'
                    )
                else:
                    # We have files, check if they exist
                    nws_files = self._xms_data['sim_data'].nws11_files['ETA File'].data.tolist()
                    for nws_file in nws_files:
                        if proj_dir == '':
                            # Haven't saved a project yet, so need to have an absolute path
                            cur_file = nws_file
                        else:
                            # We have a project saved, so check via a relative path
                            cur_file = io_util.resolve_relative_path(proj_dir, nws_file)
                        if not os.path.isfile(cur_file):
                            self._append_error(
                                'NWS 11 File Error', f'Could not find NWS 11 ETA file:  {nws_file}',
                                'Ensure that the file path is correct in Model Control.'
                            )
            elif nws == 12:
                # Check if we have any NWS files specified
                if self._xms_data['sim_data'].nws12_files.sizes['dim_0'] == 0:
                    self._append_error(
                        'NWS 12 File Error', 'Could not find any NWS 12 files.',
                        'Make sure the files exist and have been specified in the Model Control.'
                    )
                else:
                    # We have files, check if they exist
                    nws_files = self._xms_data['sim_data'].nws12_files['Pressure File'].data.tolist()
                    for nws_file in nws_files:
                        if proj_dir == '':
                            # Haven't saved a project yet, so need to have an absolute path
                            cur_file = nws_file
                        else:
                            # We have a project saved, so check via a relative path
                            cur_file = io_util.resolve_relative_path(proj_dir, nws_file)
                        if not os.path.isfile(cur_file):
                            self._append_error(
                                'NWS 12 File Error', f'Could not find NWS 12 Pressure file:  {nws_file}',
                                'Ensure that the file path is correct in Model Control.'
                            )
                    nws_files = self._xms_data['sim_data'].nws12_files['Wind File'].data.tolist()
                    for nws_file in nws_files:
                        if proj_dir == '':
                            # Haven't saved a project yet, so need to have an absolute path
                            cur_file = nws_file
                        else:
                            # We have a project saved, so check via a relative path
                            cur_file = io_util.resolve_relative_path(proj_dir, nws_file)
                        if not os.path.isfile(cur_file):
                            self._append_error(
                                'NWS 12 File Error', f'Could not find NWS 12 Wind file:  {nws_file}',
                                'Ensure that the file path is correct in Model Control.'
                            )
            elif nws == 15:
                # Check if we have any NWS files specified
                if self._xms_data['sim_data'].nws15_files.sizes['dim_0'] == 0:
                    self._append_error(
                        'NWS 15 File Error', 'Could not find any NWS 15 files.',
                        'Make sure the files exist and have been specified in the Model Control.'
                    )
                else:
                    # We have files, check if they exist
                    nws_files = self._xms_data['sim_data'].nws15_files['HWND File'].data.tolist()
                    for nws_file in nws_files:
                        if proj_dir == '':
                            # Haven't saved a project yet, so need to have an absolute path
                            cur_file = nws_file
                        else:
                            # We have a project saved, so check via a relative path
                            cur_file = io_util.resolve_relative_path(proj_dir, nws_file)
                        if not os.path.isfile(cur_file):
                            self._append_error(
                                'NWS 15 File Error', f'Could not find NWS 15 HWND file:  {nws_file}',
                                'Ensure that the file path is correct in Model Control.'
                            )
            elif nws == 16:
                # Check if we have any NWS files specified
                if self._xms_data['sim_data'].nws16_files.sizes['dim_0'] == 0:
                    self._append_error(
                        'NWS 16 File Error', 'Could not find any NWS 16 files.',
                        'Make sure the files exist and have been specified in the Model Control.'
                    )
                else:
                    # We have files, check if they exist
                    nws_files = self._xms_data['sim_data'].nws16_files['GFDL File'].data.tolist()
                    for nws_file in nws_files:
                        if proj_dir == '':
                            # Haven't saved a project yet, so need to have an absolute path
                            cur_file = nws_file
                        else:
                            # We have a project saved, so check via a relative path
                            cur_file = io_util.resolve_relative_path(proj_dir, nws_file)
                        if not os.path.isfile(cur_file):
                            self._append_error(
                                'NWS 16 File Error', f'Could not find NWS 16 GFDL file:  {nws_file}',
                                'Ensure that the file path is correct in Model Control.'
                            )

    def _check_wind_grid(self):
        """Check wind grid for NWS3 and NWS6."""
        nws = self._xms_data['sim_data'].wind.attrs['NWS']
        if nws not in [3, 6] or self._xms_data['sim_data'].wind.attrs['use_existing']:
            return

        # grid must exist
        grid = self._xms_data.get('wind_grid')
        if not grid:
            self._append_error(
                'Missing NWS 3 or NWS 6 Grid ', 'Could not find NWS wind grid',
                'Ensure that the grid is included in the simulation.'
            )
            return

        # grid must not be rotated
        if grid.angle != 0.0:
            self._append_error(
                'Invalid NWS 3 or NWS 6 Grid Orientation', 'Wind grid can not be rotated',
                'Ensure that the grid included in the simulation is not rotated.'
            )

        # grid must have constant width columns and constant height rows
        i_locs = grid.locations_y
        j_locs = grid.locations_x
        i_sizes = np.array([i_locs[i + 1] - i_locs[i] for i in range(len(i_locs) - 1)])
        j_sizes = np.array([j_locs[j + 1] - j_locs[j] for j in range(len(j_locs) - 1)])

        # Check that sizes are constant in both the i and j directions.
        if not np.isclose(i_sizes, i_sizes[0]).all() or not np.isclose(j_sizes, j_sizes[0]).all():
            self._append_error(
                'Invalid NWS 3 or NWS 6 Grid Cell Sizes', 'Wind grid must have constant cell size',
                'Ensure that the grid has constant cell sizes.'
            )

        # grid must cover mesh
        # get extent of mesh
        cogrid = read_grid_from_file(self._xms_data['current_cogrid'])
        mesh_extents = cogrid.ugrid.extents
        # get extent of grid
        grid_extents = grid.ugrid.extents
        out_min = mesh_extents[0][0] < grid_extents[0][0] or mesh_extents[0][1] < grid_extents[0][1]
        out_max = mesh_extents[1][0] > grid_extents[1][0] or mesh_extents[1][1] > grid_extents[1][1]
        if out_min or out_max:
            self._append_error(
                'Invalid NWS 3 or NWS 6 Grid Extents', 'Wind grid must cover mesh',
                'Ensure that the grid covers the mesh.'
            )

    def _check_nodal_atts(self):
        """Ensure that valid datasets have been selected for all enabled nodal attributes."""
        z0_land_dsets = [
            '000',
            '030',
            '060',
            '090',
            '120',
            '150',
            '180',
            '210',
            '240',
            '270',
            '300',
            '330',
        ]
        bridge_dsets = [  # [(dset_name, button_label_text)]
            ('BK', 'BK (bridge pier shape factor)'),
            ('BAlpha', 'BAlpha (fraction of the cross section occupied by all of the piers)'),
            ('BDelX', 'BDelX (approximate node spacing)'),
            ('POAN', 'POAN (bridge pier drag parameter [1 or 2])'),
        ]
        nodal_atts = self._xms_data['sim_data'].nodal_atts.attrs

        # Generic nodal attributes
        if nodal_atts['surface_submergence_state_on']:
            directions = 'under the "Surface submergence state - StartDry (unitless)" checkbox'
            self._check_nodal_att(
                nodal_atts['surface_submergence_state'], 'StartDry', 'StartDry', 'Nodal attributes', directions
            )
        if nodal_atts['wave_refraction_in_swan_on']:
            directions = 'under the "Wave refraction in Swan - SwanWaveRefrac (unitless)" checkbox'
            self._check_nodal_att(
                nodal_atts['wave_refraction_in_swan'], 'SwanWaveRefrac', 'SwanWaveRefrac', 'Nodal attributes',
                directions
            )
        if nodal_atts['bridge_pilings_friction_paramenters_on']:
            for dset in bridge_dsets:
                directions = f'next to the "{dset[0]}" label'
                self._check_nodal_att(
                    nodal_atts[f'{dset[0]}'], 'bridge pilings friction', dset[0], 'Nodal attributes', directions
                )
        if nodal_atts['elemental_slope_limiter_on']:
            directions = 'under the "Elemental slope limiter (m/m)" checkbox'
            self._check_nodal_att(
                nodal_atts['elemental_slope_limiter'], 'elemental slope limiter', 'elemental slope limiter',
                'Nodal attributes', directions
            )
        if nodal_atts['initial_river_elevation_on']:
            directions = 'under the "Initial river elevation (m)" checkbox'
            self._check_nodal_att(
                nodal_atts['initial_river_elevation'], 'initial river elevation', 'initial river elevation',
                'Nodal attributes', directions
            )
        if nodal_atts['average_horizontal_eddy_viscosity_in_sea_water_wrt_depth_on']:
            directions = 'in the "Model type - IM" section'
            self._check_nodal_att(
                nodal_atts['average_horizontal_eddy_viscosity_in_sea_water_wrt_depth'], 'EVC', 'EVC',
                'Model formulation', directions
            )
        if nodal_atts['advection_state_on']:
            directions = 'in the "Nonlinear terms" section'
            self._check_nodal_att(
                nodal_atts['advection_state'], 'advection state', 'advection state', 'Model formulation', directions
            )

        # Wind nodal attributes
        if self._xms_data['sim_data'].wind.attrs['NWS']:  # Wind is enabled
            if nodal_atts['surface_directional_effective_roughness_length_on']:
                for dset in z0_land_dsets:
                    directions = f'next to the "{dset}" label in the "Nodal attributes" section'
                    self._check_nodal_att(
                        nodal_atts[f'z0land_{dset}'], 'z0Land', f'{dset} direction', 'Wind', directions
                    )
            if nodal_atts['surface_canopy_coefficient_on']:
                directions = 'under the "Surface canopy coefficient - VCanopy (unitless)" checkbox'
                self._check_nodal_att(
                    nodal_atts['surface_canopy_coefficient'], 'VCanopy', 'VCanopy', 'Wind', directions
                )

        # Friction nodal attributes
        if self._xms_data['sim_data'].formulation.attrs['TAU0'] == -3:  # -3 = nodal attribute option
            directions = 'in the "Generalized wave continuity equation - GWCE" section'
            self._check_nodal_att(
                nodal_atts['primitive_weighting_in_continuity_equation'], 'TAU0', 'TAU0', 'Model formulation',
                directions
            )
        nolibf = self._xms_data['sim_data'].general.attrs['NOLIBF']
        directions = 'in the "Bottom stress/friction - NOLIBF" section'
        if nolibf == 3:  # 3 = quadratic friction nodal attribute
            self._check_nodal_att(
                nodal_atts['quadratic_friction_coefficient_at_sea_floor'], 'quadratic friction', 'quadratic friction',
                'General', directions
            )
        elif nolibf == 4:  # 4 = Manning's N nodal attribute
            self._check_nodal_att(
                nodal_atts['mannings_n_at_sea_floor'], "Manning's N", "Manning's N", 'General', directions
            )
        elif nolibf == 5:  # 5 = ChezyFric nodal attribute
            self._check_nodal_att(
                nodal_atts['chezy_friction_coefficient_at_sea_floor'], 'Chezy C', 'Chezy C', 'General', directions
            )
        elif nolibf == 6:  # 6 = Z0b_var nodal attribute
            self._check_nodal_att(
                nodal_atts['bottom_roughness_length'], 'bottom roughness length', 'Z0b_var', 'General', directions
            )

    def _check_nodal_att(self, uuid, att_name, dset_name, tab_name, button_directions):
        """Ensure that a valid dataset has been selected for a single enabled nodal attribute.

        Args:
            uuid (:obj:`str`): UUID of the selected dataset
            att_name (:obj:`str`): Name of the nodal attribute
            dset_name (:obj:`str`): Name of the dataset. Often the nodal attribute name.
            tab_name (:obj:`str`): Tab in the Model Control dialog with the nodal attribute dataset selector.
            button_directions (:obj:`str`): Text describing the dataset selector button. See description text
                string in code to know how it will be inserted into the text.
        """
        if not find_tree_node_by_uuid(self._xms_data['pe_tree'], uuid):
            self._append_error(
                'ERROR! Nodal attribute specification is incomplete.',
                f'The option to use the {att_name} nodal attribute has been enabled, but no {dset_name} data set has '
                'been selected.',
                f'Right-click on the simulation and select "Model Control...". Go to the "{tab_name}" tab and press '
                f'the "Select..." button {button_directions}. Select a valid {dset_name} data set. Nodal attributes '
                f'must be scalar, steady-state data sets on the ADCIRC mesh.'
            )

    def run_checks(self):
        """Run model checks on the ADCIRC simulation."""
        if not self._errors:  # Only run checks if we were able to retrieve data from XMS.
            self._has_flow_boundaries = True if self._xms_data['mapped_bc_data'].get_river_node_ids() else False
            self._check_mesh()
            self._check_simulation()
            self._check_tidal_constituents()
            self._check_flow_constituents()
            self._check_hotstart_files()
            self._check_projection_units()
            self._check_times()  # must call before _check_wind to get self._run_end_time
            self._check_wind()
            self._check_nodal_atts()

    def send_errors(self):
        """Send model check warnings and errors back to xms."""
        if self._query_helper:
            self._query_helper.send_model_check_messages(self._errors)

    def get_problems(self):
        """Get a list of the problem strings after running model checks."""
        return [check.problem_text for check in self._errors]

    def get_descriptions(self):
        """Get a list of the description strings after running model checks."""
        return [check.description_text for check in self._errors]

    def get_fixes(self):
        """Get a list of the fix strings after running model checks."""
        return [check.fix_text for check in self._errors]
