"""OutputComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2025'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.guipy.data.plot_and_table_data_srh import PlotsAndTableDataSrh
from xms.guipy.dialogs.plot_and_table_dialog import PlotsAndTableDialog

# 4. Local modules
from xms.adh.components.adh_component import AdHComponent
from xms.adh.data.output_data import OutputData


class OutputComponent(AdHComponent):
    """A Dynamic Model Interface (DMI) component base for the AdH output coverage."""
    def __init__(self, main_file):
        """Constructor.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file.strip('"\''))
        self.data = OutputData(self.main_file)
        self.cov_uuid = self.data.info.attrs['cov_uuid']
        self.point_commands = [
            ('Nodal Output Plots...', 'open_pt_plot'),
        ]

    def create_event(self, lock_state: bool) -> tuple[list, list]:
        """This will be called when the component is created from nothing.

        Args:
            lock_state: True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (:obj:`tuple(list, list)`):
                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being the
                message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        messages = []
        action_requests = [self.get_display_refresh_action()]
        return messages, action_requests

    def save_to_location(self, new_path: str, save_type: str):
        """Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple(list, list)`), (:obj:`list[ActionRequest]`):
                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being
                the message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)

        if save_type == 'DUPLICATE':
            # Reset the component's coverage UUID if a duplicate event
            new_data = OutputData(new_main_file)
            new_data.info.attrs['cov_uuid'] = ''
            new_data.commit()

        return new_main_file, messages, action_requests

    def get_initial_display_options(self, query: Query, params: dict):
        """Set component ids in XMS after a model-native import.

        Args:
            query (:obj:`Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Unused in this case.

        Returns:
            Empty message and ActionRequest lists
        """
        if not self.cov_uuid:  # Store the UUID of the owning coverage
            self.cov_uuid = query.parent_item_uuid()
            self.data.info.attrs['cov_uuid'] = self.cov_uuid
            self.data.commit()
        return [], []

    def handle_merge(self, _merge_list):
        """Method used by coverage component implementations to handle coverage merges.

        Args:
            _merge_list (:obj:`list[tuple]`): tuple containing:

        Returns:
            (:obj:`tuple(list, list)`),(:obj:`list[ActionRequest]`) :
                messages (:obj:`list[tuple(str, str)]`): List of tuples with the first element of the tuple being
                the message level ('DEBUG', 'ERROR', 'WARNING', 'INFO') and the second element being the message text.

                action_requests (:obj:`list[ActionRequest]`): List of actions for XMS to perform.
        """
        return [], []

    def open_pt_plot(self, query: Query, params, win_cont):
        """Opens the Assign Observation dialog and saves component data state on OK.

        Args:
            query (:obj:`xms.api.dmi.Query`): Object for communicating with XMS
            params (:obj:`dict`): Generic map of parameters. Contains selection map and component id files.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (:obj:`list[xms.api.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        f_ids = params[0].get('selection', [])
        if len(f_ids) != 1:
            msg = 'One (and only one) point must be selected to view Nodal Output Plots.'
            return [('INFO', msg)], []

        data = PlotsAndTableDataSrh(
            feature_id=f_ids[0],
            feature_type='Point',
            cov_uuid=self.cov_uuid,
            pe_tree=query.project_tree,
            model_name='AdH'
        )
        dlg = PlotsAndTableDialog(win_cont, data)
        dlg.exec()
        return [], []
