"""This module provides a way to create Qt widgets from param objects."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from adhparam.time_series import TimeSeries
import pandas as pd
import param
from PySide2.QtCore import QDir
from PySide2.QtGui import QIntValidator  # noqa: AQU103
from PySide2.QtWidgets import (
    QCheckBox, QComboBox, QFileDialog, QGroupBox, QHBoxLayout, QLabel, QLineEdit, QPushButton, QSizePolicy, QSpacerItem,
    QVBoxLayout
)

# 3. Aquaveo modules
from xms.guipy.validators.number_corrector import NumberCorrector  # noqa: AQU103
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.data import param_util
from xms.adh.gui.time_series_editor import TimeSeriesEditor
from xms.adh.gui.widgets import adh_table_widget


class ParamQtHelper:
    """A dialog for param."""
    def __init__(self, parent_dialog):
        """Initializes the class, sets up the ui, and writes the model control values.

        Args:
            parent_dialog (QDialog): The dialog to add widgets to.
        """
        self.parent_dialog = parent_dialog
        self.param_dict = dict()
        self.doing_param_widgets = False
        self.param_horizontal_layouts = dict()
        self.param_groups = dict()
        self.time_series_names = []
        self.time_series = dict()
        self.table_rename = dict()
        self.option_rename = dict()
        self.number_corrector = NumberCorrector(self.parent_dialog)

    def add_params_to_layout(self, layout, param_parent):
        """Adds param object widgets to a layout.

        Args:
            layout (QAbstractItemLayout): The layout to add widgets to.
            param_parent (Param): The parent param object.
        """
        # get param classes ordered by original precedence and add them to the vertical layout
        names, params = param_util.names_and_params_from_class(param_parent)
        pdict = param_util.declared_precedence_dict(param_parent)
        lst = []
        for i in range(len(names)):
            lst.append((float(pdict[names[i]]), params[i], names[i]))
        lst.sort()
        for item in lst:
            self.add_param(layout, item[1], item[2], param_parent)

    def add_param(self, layout, param_obj, param_name, parent_class):
        """Add param objects.

        Args:
            layout: The layout to append to
            param_obj: The param object
            param_name: param name of the item
            parent_class: param parent of the item
        """
        layout_in = layout
        ptype = type(param_obj)
        val = getattr(parent_class, param_name)
        if val is None and ptype == param.ClassSelector:
            return
        # widgets = widget_depends[param_name] if widget_depends and param_name in widget_depends else []
        widgets = []

        layout_info = None
        if hasattr(parent_class, 'param_layout') and param_name in parent_class.param_layout:
            layout_info = parent_class.param_layout[param_name]

        if layout_info:
            if layout_info.horizontal_layout:
                if layout_info.horizontal_layout in self.param_horizontal_layouts:
                    layout = self.param_horizontal_layouts[layout_info.horizontal_layout]
                else:
                    horiz_layout = QHBoxLayout()
                    self.param_horizontal_layouts[layout_info.horizontal_layout] = horiz_layout
                    layout.addLayout(horiz_layout)
                    layout = horiz_layout
            if layout_info.group_id is not None:
                label = ''
                if layout_info.group_label is not None:
                    label = layout_info.group_label
                if layout_info.group_id not in self.param_groups:
                    widgets.append(QGroupBox(label))
                    layout.addWidget(widgets[-1])
                    vertical_layout = QVBoxLayout()
                    widgets[-1].setLayout(vertical_layout)
                    self.param_groups[layout_info.group_id] = vertical_layout
                layout = self.param_groups[layout_info.group_id]

        label_str = param_obj.label
        if len(label_str) > 0 and ptype != param.Boolean:
            widgets.append(QLabel(label_str + ':'))
            layout.addWidget(widgets[-1])

        if ptype == param.ObjectSelector:
            widgets.append(QComboBox())
            layout.addWidget(widgets[-1])
            widgets[-1].setObjectName(param_name)
            widgets[-1].addItems(list(param_obj.get_range().keys()))
            # Beware of using lambdas with Qt connect(). Without the ignored '_' argument, this works fine with pure
            # Python but crashes with a Cython build. Should avoid doing this.
            widgets[-1].currentIndexChanged.connect(lambda _: self.do_param_widgets(param_obj))
            value_widget = widgets[-1]
            value_setter = widgets[-1].setCurrentText
            value_getter = widgets[-1].currentText
        elif ptype == param.Integer:
            if param_name in self.time_series_names:
                hor_layout = QHBoxLayout()
                layout.addLayout(hor_layout)
                widgets.append(QPushButton('Edit Series...'))
                widgets[-1].setObjectName(param_name)
                button = widgets[-1]
                hor_layout.addWidget(button)
                button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
                # connect to the button click
                line_edit = QLineEdit()
                hor_layout.addWidget(line_edit)
                hor_layout.addSpacerItem(QSpacerItem(0, 0, QSizePolicy.Expanding))
                line_edit.setVisible(False)
                line_edit.editingFinished.connect(lambda: self.do_param_widgets(param_obj))
                button.clicked.connect(lambda: self.do_time_series(param_obj, line_edit))
                value_widget = line_edit
                value_setter = line_edit.setText
                value_getter = line_edit.text
            else:
                widgets.append(QLineEdit())
                layout.addWidget(widgets[-1])
                widgets[-1].setObjectName(param_name)
                validator = QIntValidator()
                if param_obj.bounds is not None:
                    if param_obj.bounds[0] is not None:
                        validator.setBottom(param_obj.bounds[0])
                    if param_obj.bounds[1] is not None:
                        validator.setTop(param_obj.bounds[1])
                widgets[-1].setValidator(validator)
                widgets[-1].installEventFilter(self.number_corrector)
                widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
                value_widget = widgets[-1]
                value_setter = widgets[-1].setText

                def _int_text():
                    return int(widgets[-1].text())

                value_getter = _int_text
        elif ptype == param.Number:
            widgets.append(QLineEdit())
            layout.addWidget(widgets[-1])
            widgets[-1].setObjectName(param_name)
            validator = QxDoubleValidator()
            validator.setDecimals(NumberCorrector.DEFAULT_PRECISION)
            if param_obj.bounds is not None:
                if param_obj.bounds[0] is not None:
                    validator.setBottom(param_obj.bounds[0])
                if param_obj.bounds[1] is not None:
                    validator.setTop(param_obj.bounds[1])
            widgets[-1].setValidator(validator)
            widgets[-1].installEventFilter(self.number_corrector)
            widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText

            def float_text():
                return float(widgets[-1].text())

            value_getter = float_text
        elif ptype == param.String:
            widgets.append(QLineEdit())
            layout.addWidget(widgets[-1])
            widgets[-1].setObjectName(param_name)
            widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
        elif ptype == param.FileSelector:
            hor_layout = QHBoxLayout()
            layout.addLayout(hor_layout)
            widgets.append(QPushButton('Select File...'))
            button = widgets[-1]
            hor_layout.addWidget(button)
            button.setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            # connect to the button click
            widgets.append(QLineEdit())
            hor_layout.addWidget(widgets[-1])
            widgets[-1].setReadOnly(True)
            widgets[-1].editingFinished.connect(lambda: self.do_param_widgets(param_obj))
            button.clicked.connect(lambda: self.do_file_selector(param_obj, widgets[-1]))
            value_widget = widgets[-1]
            value_setter = widgets[-1].setText
            value_getter = widgets[-1].text
        elif ptype == param.Boolean:
            widgets.append(QCheckBox(label_str))
            layout.addWidget(widgets[-1])
            widgets[-1].setObjectName(param_name)
            widgets[-1].clicked.connect(lambda: self.do_param_widgets(param_obj))
            value_widget = widgets[-1]
            value_setter = widgets[-1].setChecked
            value_getter = widgets[-1].isChecked
        elif ptype == param.DataFrame:
            df = getattr(parent_class, param_name)
            widgets.append(self.setup_ui_table_view(layout, df))
            layout.addWidget(widgets[-1])
            value_widget = widgets[-1]
            # model = widgets[-1].filter_model.sourceModel()
            model = widgets[-1].model
            value_setter = model
            value_getter = model.data_frame
            # if layout_info and layout_info.size_policy_min:
            #     widgets[-1].setSizePolicy(QSizePolicy.Minimum, QSizePolicy.Minimum)
            # add spin box for number of rows
            # hor_layout = QHBoxLayout()
            # layout.addLayout(hor_layout)
            # widgets.append(QLabel('Number of rows:'))
            # hor_layout.addWidget(widgets[-1])
            # widgets.append(QSpinBox())
            # hor_layout.addWidget(widgets[-1])
            # widgets[-1].setMinimum(1)
            # widgets[-1].setMaximum(100000)
            # widgets[-1].setKeyboardTracking(False)
            # widgets[-1].valueChanged.connect(lambda: self.df_numrows_changed(widgets[-1], model))
            # widgets[-1].setSizePolicy(QSizePolicy.Maximum, QSizePolicy.Maximum)
            # hor_layout.addStretch()
        elif ptype == param.ClassSelector:
            layout_to_pass = layout_in
            if layout_info is not None and layout_info.group_id is not None:
                layout_to_pass = layout
            val = getattr(parent_class, param_name)
            if val is not None:
                self.add_params_to_layout(layout_to_pass, val)
            value_widget = None
            value_setter = None
            value_getter = None
        else:
            raise RuntimeError(f'Unsupported "param" parameter type: {ptype}')
            # skip = True
            # val = getattr(parent_class, param_name)
            # if issubclass(type(val), param.Parameterized):
            #     self.add_params_to_layout(layout, val, widget_depends)
            # else:
            #     raise RuntimeError(f'Unsupported "param" parameter type: {ptype}')

        self.param_dict[param_obj] = {
            'param_name': param_name,
            'parent_class': parent_class,
            'value_widget': value_widget,
            'value_getter': value_getter,
            'value_setter': value_setter,
            'widget_list': widgets,
        }

    def get_widget(self, param_name: str):
        """Retrieve the main widget associated with a param name."""
        for param_info in self.param_dict.values():
            if param_info['param_name'] == param_name:
                return param_info['value_widget']
        return None

    def setup_ui_table_view(self, layout, df):
        """Sets up the table veiw for the output times.

        Args:
            layout (QItemLayout): The layout to add the table widget to.
            df (DataFrame): The dataframe that will be used to make a Qt model for a table.

        Returns:
            A QTableWidget.
        """
        widget = adh_table_widget.AdhTableWidget(self.parent_dialog, df, 0, None, {})
        # if not df.index.empty and df.index[0] == 0:
        #     df.index = df.index + 1  # Start index at 1, not 0
        # model = QxPandasTableModel(df)
        # widget.filter_model = QSortFilterProxyModel(self.parent_dialog)
        # widget.filter_model.setSourceModel(model)
        # widget.setModel(widget.filter_model)
        # widget.horizontalHeader().setSectionResizeMode(QHeaderView.ResizeToContents)
        return widget

    def df_numrows_changed(self, spin, model):
        """Called when the number of rows in a dataframe changes.

        Args:
            spin (QSpinBox): The spinbox widget with the number of rows.
            model (QAbstractItemModel): The model to add rows to or remove rows from.
        """
        model_rows = model.rowCount()
        spin_rows = spin.value()
        more_rows = spin_rows - model_rows
        if more_rows > 0:
            model.insertRows(model_rows, more_rows)
        elif more_rows < 0:
            less_rows = -more_rows
            model.removeRows(model_rows - less_rows, less_rows)

    def do_file_selector(self, param_obj, file_edit_field):
        """Opens a file system selector.

        Args:
            param_obj (Param): The param object.
            file_edit_field (QLineEdit): The Qt line edit widget that is normally created.
        """
        p_dict = self.param_dict[param_obj]
        curr_filename = getattr(p_dict['parent_class'], p_dict['param_name'])
        path = param_obj.path
        if curr_filename:
            path = os.path.dirname(curr_filename)
        if not os.path.exists(path):
            path = QDir.homePath()
        file_filter = 'All files (*.*)'
        dlg = QFileDialog(self.parent_dialog, 'Select File', path, file_filter)
        dlg.setLabelText(QFileDialog.Accept, "Select")
        if dlg.exec_():
            setattr(p_dict['parent_class'], p_dict['param_name'], dlg.selectedFiles()[0])
            file_edit_field.setText(dlg.selectedFiles()[0])
        self.do_param_widgets(None)

    def do_time_series(self, param_obj, id_edit_field):
        """Opens a time series editor.

        Args:
            param_obj (Param): The param object.
            id_edit_field (QLineEdit): The Qt line edit widget that is normally created.
        """
        p_dict = self.param_dict[param_obj]
        ts_id = getattr(p_dict['parent_class'], p_dict['param_name'])
        df = None

        added = False
        if ts_id and ts_id > 0:
            for ts in self.time_series:
                if ts_id in self.time_series.keys():
                    df = self.time_series[ts_id].time_series
                elif self.time_series[ts].series_id == ts_id:
                    df = self.time_series[ts].time_series
            added = False

        if df is None:
            # add a new series
            if not self.time_series:
                ts_id = 1
            else:
                ts_id = max(self.time_series.keys()) + 1
            df = pd.DataFrame(data=[[0.0, 0.0]], columns=['X', 'Y'])
            added = True

        time_series_editor = TimeSeriesEditor(df, parent=self.parent_dialog)
        if time_series_editor.run():
            setattr(p_dict['parent_class'], p_dict['param_name'], ts_id)
            id_edit_field.setText(str(ts_id))
            if added:
                self.time_series[ts_id] = TimeSeries()
                self.time_series[ts_id].series_id = ts_id
            self.time_series[ts_id].time_series = time_series_editor.series
        self.do_param_widgets(None)

    def do_param_widgets(self, param_obj):
        """Updates the visible state of widgets based on the conditions in the param classes.

        Args:
            param_obj (Param): The param object to be skipped, None if all param widgets values are to update.
        """
        if self.doing_param_widgets:
            return
        self.doing_param_widgets = True

        if param_obj is not None:
            p_dict = self.param_dict[param_obj]
            # get the value for this parameter for its widget and set it in the param class
            val = p_dict['value_getter']()
            if type(param_obj) is param.ObjectSelector:
                if val not in param_obj.get_range().values():
                    val = param_obj.get_range()[val]
            setattr(p_dict['parent_class'], p_dict['param_name'], val)

        # setting a param value can trigger changes to other members of the class so we need
        # to set the values of the other params to the widgets
        for par in self.param_dict.keys():
            if par == param_obj:
                continue
            self._set_param_widget_value(par)

        # hide or show widgets base on current precedence value of param objects
        for par in self.param_dict.keys():
            hide = par.precedence < 0
            p_dict = self.param_dict[par]
            for widget in p_dict['widget_list']:
                if hide:
                    widget.hide()
                else:
                    widget.show()

        self.doing_param_widgets = False

    def _set_param_widget_value(self, par):
        p_dict = self.param_dict[par]
        val = getattr(p_dict['parent_class'], p_dict['param_name'])
        if type(par) in [param.Number, param.Integer]:
            val = str(val)
        elif type(par) is param.ObjectSelector:
            if not par.names:
                val = str(val)
            else:
                all_values = par.get_range()
                for key, obj_val in all_values.items():
                    if val == obj_val:
                        val = key
                        break
        if type(val) is pd.DataFrame:
            model = p_dict['value_setter']
            model.data_frame = val
        else:
            if p_dict['value_setter'] is not None:
                p_dict['value_setter'](val)
