"""A widget for friction values used by materials and boundary conditions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from PySide2.QtWidgets import (QWidget)

# 3. Aquaveo modules
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.adh.gui.widgets.friction_widget_ui import Ui_FrictionWidget


class FrictionWidget(QWidget):
    """A widget for assigning friction."""
    friction_types = [
        "Off",
        "Manning's N (MNG)",
        "Manning's Equation (MNC)",
        "Equivalent roughness height (ERH)",
        "Submerged aquatic vegetation (SAV)",
        "Un-submerged rigid vegetation (URV)",
        "Equivalent drag obstructions (EDO)",
        "Ice friction (ICE, IRH, BRH)",
        "Dune friction (DUN)",
        "Submerged dike (SDK)",
        "Bridge deck (BRD)",
    ]
    card_to_type = {
        None: "Off",
        'MNG': "Manning's N (MNG)",
        'MNC': "Manning's Equation (MNC)",
        'ERH': "Equivalent roughness height (ERH)",
        'SAV': "Submerged aquatic vegetation (SAV)",
        'URV': "Un-submerged rigid vegetation (URV)",
        'EDO': "Equivalent drag obstructions (EDO)",
        'ICE': "Ice friction (ICE, IRH, BRH)",
        'DUN': "Dune friction (DUN)",
        'SDK': "Submerged dike (SDK)",
        'BRD': "Bridge deck (BRD)",
    }

    def __init__(self, win_cont, is_arc):
        """Allows the user to edit friction.
        """
        super().__init__(win_cont)

        self.ui = Ui_FrictionWidget()
        self.ui.setupUi(self)

        # tuple has card name, series button
        self.CARD = 0
        self.WIDGETS = 1
        self.index_to_card = [
            (None, ()), ('MNG', (self.ui.mng_mannings_n, )), ('MNC', (self.ui.mnc_mannings_eq, )),
            ('ERH', (self.ui.erh_equivalent_roughness_height, )),
            ('SAV', (self.ui.sav_roughness_height_of_canopy, self.ui.sav_undeflected_stem_height)),
            (
                'URV',
                (self.ui.urv_bed_roughness_height, self.ui.urv_average_stem_diameter, self.ui.urv_average_stem_density)
            ),
            (
                'EDO',
                (self.ui.edo_bed_roughness_height, self.ui.edo_obstruction_diameter, self.ui.edo_obstruction_height)
            ),
            (
                'ICE', (
                    self.ui.ice_thickness,
                    self.ui.ice_density,
                    self.ui.ice_movement,
                    self.ui.irh_roughness_height,
                    self.ui.brh_bed_roughness_height,
                )
            ), ('DUN', (
                self.ui.dun_factor,
                self.ui.dun_sedlib_inclusion,
                self.ui.dun_d_50,
                self.ui.dun_d_90,
            )), ('SDK', (self.ui.sdk_dike_height, )), ('BRD', (self.ui.brd_elevation, self.ui.brd_thickness))
        ]
        self.EDGE = 0
        self.MID = 1
        self.is_arc = is_arc

        self._setup()
        self._add_options(is_arc)

    def _setup(self):
        self.ui.mng_mannings_n.setValidator(QxDoubleValidator())
        self.ui.mnc_mannings_eq.setValidator(QxDoubleValidator())
        self.ui.erh_equivalent_roughness_height.setValidator(QxDoubleValidator())
        self.ui.sav_roughness_height_of_canopy.setValidator(QxDoubleValidator())
        self.ui.sav_undeflected_stem_height.setValidator(QxDoubleValidator())
        self.ui.urv_bed_roughness_height.setValidator(QxDoubleValidator())
        self.ui.urv_average_stem_diameter.setValidator(QxDoubleValidator())
        self.ui.urv_average_stem_density.setValidator(QxDoubleValidator())
        self.ui.edo_bed_roughness_height.setValidator(QxDoubleValidator())
        self.ui.edo_obstruction_diameter.setValidator(QxDoubleValidator())
        self.ui.edo_obstruction_height.setValidator(QxDoubleValidator())
        self.ui.ice_thickness.setValidator(QxDoubleValidator())
        self.ui.ice_density.setValidator(QxDoubleValidator())
        self.ui.ice_movement.setValidator(QxDoubleValidator())
        self.ui.irh_roughness_height.setValidator(QxDoubleValidator())
        self.ui.brh_bed_roughness_height.setValidator(QxDoubleValidator())
        self.ui.dun_factor.setValidator(QxDoubleValidator())
        # self.ui.dun_sedlib_inclusion.setValidator(QxDoubleValidator())
        self.ui.dun_d_50.setValidator(QxDoubleValidator())
        self.ui.dun_d_90.setValidator(QxDoubleValidator())
        self.ui.sdk_dike_height.setValidator(QxDoubleValidator())
        self.ui.brd_elevation.setValidator(QxDoubleValidator())
        self.ui.brd_thickness.setValidator(QxDoubleValidator())
        self.adjustSize()
        self.resize(self.size().width() * 1.5, self.size().height())

    def _add_options(self, is_arc):
        if is_arc:
            self.ui.cbx_edge_mid.addItems(['Edgestring', 'Midstring'])
            # remove the material only friction options (remove ICE from arc material list???  Not sure how to do this.)
            friction_types = self.friction_types

        else:
            # remove the last two options as they only apply to arcs; this section is for materials
            friction_types = self.friction_types[:-2]
            self.ui.cbx_edge_mid.hide()
            self.ui.edge_mid_label.hide()
        self.ui.cbx_friction_type.addItems(friction_types)
        self.ui.cbx_friction_type.currentIndexChanged[int].connect(self.friction_type_changed)
        self.ui.dun_sedlib_inclusion.toggled[bool].connect(self.dun_sedlib_changed)

    def set_friction_type(self, card_type, is_mid, values):
        """Sets the type of friction.

        Args:
            card_type (str): The card value after FR.
            is_mid (bool): True if this belongs to a midstring.
            values (list): A list of values associated with the friction type.
        """
        row = 0
        for idx, card_row in enumerate(self.index_to_card):
            if card_row[self.CARD] == card_type:
                row = idx
                for widgets, val in zip(card_row[self.WIDGETS], values):
                    if widgets == self.ui.dun_sedlib_inclusion:
                        widgets.setChecked(False if val == 0 else True)
                    else:
                        widgets.setText(str(val))
                break
        self.ui.cbx_friction_type.setCurrentIndex(row)
        edge_mid_idx = self.EDGE
        if is_mid:
            edge_mid_idx = self.MID
        if self.is_arc:
            self.ui.cbx_edge_mid.setCurrentIndex(edge_mid_idx)
        self.friction_type_changed(row)

    def get_friction_type(self):
        """Gets the friction type.

        Returns:
            A string of the card value that would appear immediately after FR.
        """
        return self.index_to_card[self.ui.cbx_friction_type.currentIndex()][self.CARD]

    def get_friction_values(self):
        """Gets the friction values.

        Returns:
            A list of float values associated with the friction type, inorder as they would appear in the card.
        """
        values = []
        cur_idx = self.ui.cbx_friction_type.currentIndex()
        for widget in self.index_to_card[cur_idx][self.WIDGETS]:
            if widget == self.ui.dun_sedlib_inclusion:
                values.append(1.0 if widget.isChecked() else 0.0)
            else:
                values.append(float(widget.text()))
        return values

    def get_friction_is_mid(self):
        """Gets whether the friction is associated with a midstring.

        Returns:
            True if the user chose to associate these friction values with a midstring.
        """
        return self.ui.cbx_edge_mid.currentIndex() == self.MID

    def friction_type_changed(self, index):
        """Called when the friction type combobox selection changed."""
        self.ui.friction_stack.setCurrentIndex(index)
        card_type = self.index_to_card[index][self.CARD]
        if card_type is None:
            self.ui.cbx_edge_mid.setVisible(False)
            self.ui.edge_mid_label.setVisible(False)
        elif card_type in ('SDK', 'BRD'):
            self.ui.cbx_edge_mid.setCurrentIndex(self.MID)
            self.ui.cbx_edge_mid.setVisible(True and self.is_arc)
            self.ui.edge_mid_label.setVisible(True and self.is_arc)
            self.ui.cbx_edge_mid.setEnabled(False)
            self.ui.edge_mid_label.setEnabled(False)
        else:
            self.ui.cbx_edge_mid.setVisible(True and self.is_arc)
            self.ui.edge_mid_label.setVisible(True and self.is_arc)
            self.ui.cbx_edge_mid.setEnabled(True and self.is_arc)
            self.ui.edge_mid_label.setEnabled(True and self.is_arc)

    def dun_sedlib_changed(self, checked):
        """Call when dun sedlib checkbox is changed."""
        if checked:
            self.ui.dun_d_50_label.setVisible(False)
            self.ui.dun_d_90_label.setVisible(False)
            self.ui.dun_d_50.setVisible(False)
            self.ui.dun_d_90.setVisible(False)
        else:
            self.ui.dun_d_50_label.setVisible(True)
            self.ui.dun_d_90_label.setVisible(True)
            self.ui.dun_d_50.setVisible(True)
            self.ui.dun_d_90.setVisible(True)
