"""Python wrapping for xms.api._xmsapi.dmi.Query."""
# 1. Standard python modules
import os
import shlex

# 2. Third party modules

# 3. Aquaveo modules
from xms.core.filesystem import filesystem as io_util

# 4. Local modules
from xms.api.dmi._QueryImpl import QueryImpl
from xms.api.dmi._QueryPlaybackImpl import QueryPlaybackImpl
from xms.api.dmi._QueryRecordImpl import QueryRecordImpl
from xms.api.dmi import XmsEnvironment as xmenv


class Query:
    """The pure Python wrapper for C++ exposed xms.api._xmsapi.dmi.Query objects."""
    def __init__(self, **kwargs):
        """Construct the wrapper.

        Args:
            instance (xms.api._xmsapi.dmi.Query, optional): The C++ object to wrap. Mutually exclusive with 'agent'
                kwarg.
            agent (xms.api.XmsAgent, optional): The XMS interprocess communication manager, mutually exclusive with
                'instance' kwarg.
            timeout (int): The interprocess communication timeout interval in milliseconds. Defaults to 5 minutes,
                which might be a little excessive. Used to have lots of script timeouts when we serialized more
                data through the pipe.
            retries (int): Number of times to retry after a failed message request. Defaults to one. I have never
                set it to anything else, and doing sometimes causes duplicate build data being created in XMS
                from import scripts.
            progress_script (bool): True if created in a progress script.
            migrate_script (bool): True if created in a migrate script.
            blocking (bool): False if this a non-blocking script.
        """
        self._impl = None
        self._init_impl(**kwargs)

    def _init_impl(self, **kwargs):
        """Create a record, playback, or normal mode Query based on environment.

        Args:
            **kwargs: See constructor
        """
        recording_folder = os.environ.get(xmenv.ENVIRON_FORCE_RECORD)  # Check for environment variable to force record
        if not recording_folder:  # Check for file hack that triggers record and playback mode.
            record_trigger_file = xmenv.xms_environ_record_trigger_file()
            if os.path.isfile(record_trigger_file):
                # Rename the file hack immediately to prevent the next script from recording.
                renamed_file_hack = os.path.join(
                    os.path.dirname(record_trigger_file), f'_{os.path.basename(record_trigger_file)}'
                )
                io_util.removefile(renamed_file_hack)
                os.rename(record_trigger_file, renamed_file_hack)
                # Read the recording directory from the file
                with open(renamed_file_hack, 'r') as f:
                    line = shlex.split(f.readline(), posix=False)
                    if not line or not line[0]:
                        raise ValueError(
                            f'First line of {record_trigger_file} must contain the path to the recording folder.'
                        )
                    recording_folder = line[0]

        if recording_folder:  # Either record or playback mode
            recording_folder = recording_folder.strip('"')
            if os.path.isdir(recording_folder):  # Recording folder exists, check for playback file
                playback_file = os.path.join(recording_folder, xmenv.xms_environ_playback_record_file())
                if os.path.isfile(playback_file):  # Playback file exists, mock XMS communication
                    self._impl = QueryPlaybackImpl(**kwargs)
                else:  # Playback file does not exist, record XMS communication
                    self._impl = QueryRecordImpl(recording_folder, **kwargs)
            else:  # Recording folder does not exist, create it and record XMS communication
                os.makedirs(recording_folder)
                self._impl = QueryRecordImpl(recording_folder, **kwargs)
        else:  # Normal XMS communication
            self._impl = QueryImpl(**kwargs)

    # Environment variables set by XMS.
    @property
    def xms_temp_directory(self):
        """Returns the XMS temp directory or system temp if not set."""
        return xmenv.xms_environ_temp_directory()

    @property
    def process_temp_directory(self):
        """Returns the temp directory that is deleted when this process ends. Creates if it doesn't exist."""
        return xmenv.xms_environ_process_temp_directory()

    @property
    def xms_app_name(self):
        """Name of the XMS app that launched the script."""
        return xmenv.xms_environ_app_name()

    @property
    def xms_app_version(self):
        """Version of the XMS app that launched the script."""
        return xmenv.xms_environ_app_version()

    @property
    def xms_notes_database(self):
        """Path to the XMS notes database."""
        return xmenv.xms_environ_notes_database()

    @property
    def xms_project_path(self):
        """Path to the saved XMS project currently loaded or empty string if unsaved."""
        return xmenv.xms_environ_project_path()

    @property
    def xms_running_tests(self):
        """Returns 'TRUE' if XMS is currently running tests."""
        return xmenv.xms_environ_running_tests()

    # Data commonly retrieved at the root simulation/component level.
    @property
    def project_tree(self):
        """Returns the XMS project explorer tree."""
        return self._impl.project_tree

    def refresh_project_tree(self):
        """Returns the XMS project explorer tree."""
        return self._impl.refresh_project_tree()

    def copy_project_tree(self, tree_node=None):
        """Make a deep copy of the project explorer tree.

        Args:
            tree_node (TreeNode): Root of the project explorer tree to deep copy. If not provided, will use XMS project
                root.

        Returns:
            TreeNode: See description
        """
        return self._impl.copy_project_tree(tree_node)

    @property
    def global_time(self):
        """Returns the current XMS global time."""
        return self._impl.global_time

    @property
    def global_time_settings(self):
        """Returns the current XMS global time settings as a JSON string.

        See xms.guipy.time_format.XmsTimeFormatter for more details.
        """
        return self._impl.global_time_settings

    @property
    def display_projection(self):
        """Returns the current XMS display projection."""
        return self._impl.display_projection

    @property
    def cp(self):
        """Returns a dict of GUI module name text to enable/disable flag."""
        return self._impl.cp

    @property
    def xms_agent(self):
        """Returns the Query object's xms.api.XmsAgent."""
        return self._impl.xms_agent

    @property
    def read_file(self):
        """Returns the filesystem path to the file being imported by the process.

        Note only applicable if process is an import script.
        """
        return self._impl.read_file

    # Methods for retrieving data from XMS.
    def item_with_uuid(self, item_uuid, generic_coverage=False, model_name=None, unique_name=None):
        """Retrieves an item from XMS given its UUID.

        Note:
            To get a hidden component item_uuid should be the UUID of the hidden component's owner. model_name is the
            hidden component's XML-defined model name. unique_name is the hidden component's XML-defined unique name.

        Args:
            item_uuid (str): UUID of the item
            generic_coverage (bool): True if the item is one of the dumpable generic coverages
            model_name (str): The XML model name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify unique_name if model_name provided.
            unique_name (str): The XML unique name of a hidden component. Mutually exclusive with generic_coverage.
                Must specify model_name if unique_name provided.

        Returns:
            Pure python interface to the requested item.
        """
        return self._impl.item_with_uuid(
            item_uuid, generic_coverage=generic_coverage, model_name=model_name, unique_name=unique_name
        )

    def current_item(self):
        """Returns a KEYWORD_NONE Get request for the current position in the Query Context.

        Note assuming Context position is at something with a UUID. Don't try to use if you are down at the Coverage
        Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current Context position or None on failure.
        """
        return self._impl.current_item()

    def current_item_uuid(self):
        """Returns the UUID of the item at the current Context position, if it has one.

        Returns:
            str: The current item's UUID or empty string if it doesn't have one.
        """
        return self._impl.current_item_uuid()

    def parent_item(self):
        """Returns a KEYWORD_NONE Get request for the parent of the current position in the Query Context.

        Note assuming Context position's parent is something with a UUID. Don't try to use if you are down at the
        Coverage Arc level or something like that.

        Returns:
            The KEYWORD_NONE result for the current context position's parent or None on failure.
        """
        return self._impl.parent_item()

    def parent_item_uuid(self):
        """Returns the UUID of the current Context position's parent, if it has one.

        Returns:
            str: The UUID of the current item's parent or empty string if it doesn't have one.
        """
        return self._impl.parent_item_uuid()

    def named_executable_path(self, exe_name):
        """
        Get the path to a model executable that is defined in the XML.
        
        While the XML can provide a default path, users can override it with
        a preferred path in the XMS preferences. This method gets the user's
        preferred executable if set, or falls back to the default otherwise.
        
        It is possible that there is no path for the requested executable.
        This can happen if the requested name does not match anything in the
        XML, or if the XML provides no default *and* the user didn't provide
        a preferred one. If no path is found, an empty string is returned.

        Args:
            exe_name (str): The name of the executable to get the path for.
                This name is defined in the model's XML file. It is the content
                of the `<declare_parameter>` tag inside of the `<executable>`
                tag for the executable you want the path for. 

        Returns:
            str: Path to the named executable or empty string if not found
        """
        return self._impl.named_executable_path(exe_name)

    def load_component_ids(
        self,
        component,
        points=False,
        arcs=False,
        arc_groups=False,
        polygons=False,
        only_selected=False,
        delete_files=True
    ):
        """Query XMS for a dump of all the current component ids of the specified entity type.
        
        This calls `component.load_coverage_component_id_map` to initialize `component.comp_to_xms`.

        Args:
            component (xms.components.bases.coverage_component_base.CoverageComponentBase): The coverage component to
                load id files for
            points (bool): True if you want point component id maps.
            arcs (bool): True if you want arc component id maps.
            arc_groups (bool): True if you want arc group component id maps.
            polygons (bool): True if you want polygon component id maps.
            only_selected (bool): If True, only retrieves mappings for the currently selected features.
            delete_files (bool): If True, id files will be deleted before return.
                N.B. This is probably a legacy thing. The only thing you can really do with the files is pass them to
                `component.load_coverage_component_id_map`, which this method already does for you. You could use the
                files to initialize multiple instances of a component, but it's usually easier to just reuse the first
                instance. Keeping the files to save time fetching them during the next script invocation is asking for
                trouble (try assigning features, deleting the first one, and renumbering the coverage for fun times).
                True will almost always be the correct value for this.

        Returns:
            dict: The file dict.

            key = entity_type ('POINT', 'ARC', 'ARC_GROUP', 'POLYGON'),

            value = (att_id_file, comp_id_file).

            Note that if delete_files=True, these files will not exist upon
            return. If you need to make a copy of the component id files, set delete_files=False and manually clean
            up the files after copying.
        """
        return self._impl.load_component_ids(
            component,
            points=points,
            arcs=arcs,
            arc_groups=arc_groups,
            polygons=polygons,
            only_selected=only_selected,
            delete_files=delete_files
        )

    def coverage_attributes(self, cov_uuid, points=False, arcs=False, polygons=False, arc_groups=False):
        """Retrieve coverage attribute table from XMS.

        Args:
            cov_uuid (str): UUID of the coverage
            points (bool): True to retrieve feature point attributes
            arcs (bool): True to retrieve feature arc attributes
            polygons (bool): True to retrieve feature polygon attributes
            arc_groups (bool): True to retrieve feature arc group attributes

        Returns:
            dict: The requested attribute tables. Keyed by 'points', 'arcs', 'polygons', and 'arc_groups'.
        """
        return self._impl.coverage_attributes(
            cov_uuid, points=points, arcs=arcs, polygons=polygons, arc_groups=arc_groups
        )

    # Methods for adding data to be built in XMS.
    def add_simulation(self, do_sim, components=None, component_keywords=None):
        """Add a simulation to XMS.

        Args:
            do_sim (pydop.Simulation): The simulation to add
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]
        """
        self._impl.add_simulation(do_sim, components=components, component_keywords=component_keywords)

    def add_coverage(
        self,
        do_coverage,
        model_name=None,
        coverage_type=None,
        components=None,
        component_keywords=None,
        folder_path=None
    ):
        """Add a coverage to XMS.

        Args:
            do_coverage (Union[pydop.Coverage, xms.api._xmsapi.dmi.DataDumpIOBase]): The coverage to add. If type is an
                xmscoverage generic coverage dump, all other arguments are ignored.
            model_name (str): Name of the model that defined the coverage. This can be found in the defining model's
                XML file. It is the `name` attribute on the `<model>` tag. If specified, must provide coverage_type.
            coverage_type (str): The coverage's type. This can be found in the defining model's XML file. It is the
                `name` attribute on the `<declare_coverage>` tag for the coverage. If specified, must also provide
                model_name.
            components (list of pydop.Component): The simulation's hidden components. Component objects must have the
                model name and unique name attributes set.
            component_keywords (list of dict): Parallel with components if provided. Dicts of additional component
                keywords to send with the components, such as 'actions' or 'display_options'. See kwarg docs of
                add_component() for more details.
                ::
                    [{  # Possible component data keywords
                        'actions': [],
                        'messages': [],
                        'display_options': [],
                        'component_coverage_ids': [],
                        'shapefile_atts': {},
                    }]

            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        self._impl.add_coverage(
            do_coverage,
            model_name=model_name,
            coverage_type=coverage_type,
            components=components,
            component_keywords=component_keywords,
            folder_path=folder_path
        )

    def add_ugrid(self, do_ugrid):
        """Add a UGrid to be built in XMS.

        Args:
            do_ugrid (pydop.UGrid): The UGrid to add
        """
        self._impl.add_ugrid(do_ugrid)

    def add_dataset(self, do_dataset, folder_path=None):
        """Add a Dataset to be built in XMS.

        Args:
            do_dataset (Union[pydop.Dataset, DatasetReader, xms.datasets.dataset_writer.DatasetWriter]): The dataset to
                add
            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        self._impl.add_dataset(do_dataset, folder_path=folder_path)

    def add_particles(self, file_path):
        """Add a particle set to be loaded in XMS.

        Args:
            file_path: Path to the particle set file. Should be a file that XMS knows how to read. For SMS, this is
                an XMDF file. GMS has no support yet.
        """
        if not os.path.isabs(file_path):
            # This is going to another process, so make sure it doesn't get confused by conflicting current directories.
            file_path = os.path.abspath(file_path)
        self._impl.add_particles(file_path)

    def add_raster(self, filename, item_uuid=None):
        """Add a GIS raster file to be loaded in XMS.

        Args:
            filename (str): Full path to the raster file
            item_uuid (Optional[str]): UUID to assign the raster in XMS
        """
        self._impl.add_raster(filename, item_uuid)

    def add_shapefile(self, filename, item_uuid=None):
        """Add a GIS shape file to be loaded in XMS.

        Args:
            filename (str): Full path to the shape file
            item_uuid (Optional[str]): UUID to assign the shapefile in XMS
        """
        self._impl.add_shapefile(filename, item_uuid)

    def add_component(
        self,
        do_component,
        actions=None,
        messages=None,
        display_options=None,
        coverage_ids=None,
        shapefile_atts=None,
        folder_path=None
    ):
        """Add a non-hidden Component to be built in XMS.

        Args:
            do_component (pydop.Component): The component to add

            actions (list of pydop.ActionRequest): ActionRequests to send back with the component. See
                xms.components.bases.component_base.ComponentBase for more details.

            messages (list of tuple of str): Messages to send back with component. Should be formatted as follows:
                [('MSG_LEVEL', 'message text')...]  See xms.components.bases.component_base.ComponentBase for more
                details.

            display_options (list): Display options to send back with the component. Data should be in format returned
                by xms.components.bases.component_base.ComponentBase.get_display_options().

            coverage_ids (list): Coverage component id assignments to send back with the component. Data should be in
                format returned by
                xms.components.bases.coverage_component_base.CoverageComponentBase.get_component_coverage_ids().

            shapefile_atts (dict): Shapefile attributes to send back with the component. Data should be in
                format returned by xms.components.bases.component_base.ComponentBase.get_shapefile_att_files().

            folder_path (Optional, str): If provided, will create the specified tree folder structure under the item's
                parent and place the item in the terminal folder.
        """
        self._impl.add_component(
            do_component,
            actions=actions,
            messages=messages,
            display_options=display_options,
            coverage_ids=coverage_ids,
            shapefile_atts=shapefile_atts,
            folder_path=folder_path
        )

    def add_model_check_errors(self, errors):
        """Adds model check errors to be sent back to XMS.

        Errors will all be set at the Context level we had at construction. It is assumed that nothing else will be
        added during the script.

        Args:
            errors (Union[pydop.ModelCheckError, Sequence[pydop.ModelCheckError]): The model check errors to add. Can
                pass an iterable or a single error
        """
        self._impl.add_model_check_errors(errors)

    def link_item(self, taker_uuid, taken_uuid):
        """Link one item under another.

        Args:
            taker_uuid (str): UUID of the item to link the other item under. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to link under the other item
        """
        self._impl.link_item(taker_uuid, taken_uuid)

    def unlink_item(self, taker_uuid, taken_uuid):
        """Unlink one item from under another.

        Args:
            taker_uuid (str): UUID of the item to unlink the other item from. This should be the UUID of the parent tree
                item, even if the item is taken by a hidden component.
            taken_uuid (str): UUID of the item to unlink from the other item
        """
        self._impl.unlink_item(taker_uuid, taken_uuid)

    def delete_item(self, item_uuid):
        """Delete an item in XMS.

        Args:
            item_uuid (str): UUID of the item to delete
        """
        self._impl.delete_item(item_uuid)

    def rename_item(self, item_uuid, new_name):
        """Rename a tree item in XMS.

        Args:
            item_uuid (str): UUID of the item to rename
            new_name (str): New name to assign the item. Not guaranteed to be the name XMS assigns the item. Will be
                made unique if another item with the same name already exists.
        """
        self._impl.rename_item(item_uuid, new_name)

    def edit_ugrid_top_bottom_elevations(self, json_file):
        """Edit the top and bottom elevations of a UGrid in XMS.

        Args:
            json_file (str): Path to the JSON file containing the new elevations. See xms.components.dataset_metadata
                for more information of the expected JSON structure.
        """
        self._impl.edit_ugrid_top_bottom_elevations(json_file)

    def get_ugrid_selection(self, ugrid_uuid):
        """Get the currently selected UGrid cells and points.

        Returns:
            dict: Possible keys are 'POINT' and 'CELL'. Values are lists of UGrid poiont/cell ids. If key does not
            exist, no UGrid entities of that type are selected.
        """
        return self._impl.get_ugrid_selection(ugrid_uuid)

    def update_progress_percent(self, percent_done):
        """Update the progress percent of a running process in the XMS Simulation Run Queue.

        Obviously only makes sense if this is a progress script.

        Args:
            percent_done (Union[int, float]): The monitored process's new progress percent. Will be truncated if float.
        """
        self._impl.update_progress_percent(percent_done)

    # Send any built data to XMS.
    def send(self, component_event=False):
        """Send data queued from add() and set() calls to XMS.

        Args:
            component_event (bool): True if send is being sent from a component event script. They do all sorts of their
                own special messing with the build context, so don't try to manage it.

        Returns:
            xms.api.dmi.SetDataResult: Result object reply from the send request
        """
        return self._impl.send(component_event)

    def clear(self):
        """Call this to clear any data added to the Context to prevent it from being sent to XMS."""
        self._impl.clear()

    # Debug methods
    def get_children_ids(self, parent_id, relationship=''):
        """Debug method for getting the instance children of a specified instance id in the Query Context.

        Args:
            parent_id (int): Instance id of the node to retrieve children of
            relationship (str): If provided, only the children along edges with this name will be returned

        Returns:
            list of int: Child instance ids of the specified parent instance
        """
        return self._impl.get_children_ids(parent_id, relationship=relationship)

    def get_children_names(self, parent_id):
        """Debug method for getting the out edge names of a specified instance id in the Query Context.

        Args:
            parent_id (int): Instance id of the node to retrieve out edge names of

        Returns:
            list of str: Edge names between the specified parent instance and its children
        """
        return self._impl.get_children_names(parent_id)

    def get_possible_children_names(self, parent_id):
        """Debug method for getting the out edge names of a specified instance id in the Query Context.

        Args:
            parent_id (int): Instance id of the node to retrieve out edge names of

        Returns:
            list of str: See description
        """
        return self._impl.get_possible_children_names(parent_id)

    def get_vertex_type(self, instance_id):
        """Debug method for getting the type enum (as an int) of an instance in the Query Context.

        Args:
            instance_id (int): Instance id of the node to retrieve type of

        Returns:
            int: The instance's C++ ContextDataType enum as an int
        """
        return self._impl.get_vertex_type(instance_id)

    @property
    def playback_folder(self):
        """Returns the location of the playback folder if a QueryPlaybackImpl, else empty string."""
        return self._impl.playback_folder
