"""Class for managing data for profile and cross-section plots."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class PlotSimData:
    """Class for managing data for profile and cross-section plots."""
    def __init__(self, xms_data):
        """Initialize the class.

        Args:
            xms_data (:obj:`XmsData`): Object for retrieving data from XMS
        """
        self._xms_data = xms_data
        self._pe_tree = xms_data.project_tree if xms_data else None
        self.sim_data = {}
        self.sim_names = []
        self.grid_pt_data_extractor = {}
        self.grid_cell_data_extractor = {}
        self._get_sim_grids_datasets()

    def _get_sim_grids_datasets(self):
        """Get the simulation grids and datasets."""
        sims = self._sims_from_structure_coverage()
        self.sim_names = [sim.name for sim in sims]
        for sim in sims:
            wse = self._wse_from_sim(sim)
            times = ['0.0'] if not wse else self._xms_data.time_strings_from_dataset_times(wse)
            self.sim_data[sim.name] = {
                'grid': self._grid_from_sim(sim),
                'wse': wse,
                'velocity_mag': self._velocity_mag_from_sim(sim),
                'times': times,
            }

    def _sims_from_structure_coverage(self):
        """Get simulations that include the structure coverage."""
        sims = []
        if not self._xms_data:
            return sims
        cov_uuid = self._xms_data.coverage.uuid
        tree_nodes = [self._pe_tree]
        for tree_node in tree_nodes:
            tree_nodes.extend(tree_node.children)  # noqa: B038 editing a loop's mutable iterable
            if tree_node.uuid == cov_uuid and tree_node.is_ptr_item:
                sims.append(tree_node.parent)
        return sims

    def _grid_from_sim(self, sim):
        """Get the grid from the simulation."""
        rval = None
        tree_nodes = [sim]
        for tree_node in tree_nodes:
            tree_nodes.extend(tree_node.children)  # noqa: B038 editing a loop's mutable iterable
            if tree_node.item_typename in ['TI_MESH2D_PTR', 'TI_UGRID_PTR']:
                rval = self._xms_data.co_grid_from_uuid(tree_node.uuid)
                break
        return rval

    def _wse_from_sim(self, sim):
        """Get the grid from the simulation."""
        rval = None
        tree_nodes = [sim]
        for tree_node in tree_nodes:
            tree_nodes.extend(tree_node.children)  # noqa: B038 editing a loop's mutable iterable
            name = tree_node.name
            if 'Pressure' in name:
                continue
            if tree_node.item_typename == 'TI_DATASET_PTR' and 'Water_Elev' in tree_node.name:
                rval = self._xms_data.wse_dataset_from_uuid(tree_node.uuid)
                break
        return rval

    def _velocity_mag_from_sim(self, sim):
        """Get the grid from the simulation."""
        rval = None
        tree_nodes = [sim]
        for tree_node in tree_nodes:
            tree_nodes.extend(tree_node.children)  # noqa: B038 editing a loop's mutable iterable
            if tree_node.item_typename == 'TI_DATASET_PTR' and 'Vel_Mag' in tree_node.name:
                rval = self._xms_data.wse_dataset_from_uuid(tree_node.uuid)
                break
        return rval
