"""This is a dialog for specifying advanced cards that are not currently supported in the interface."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import webbrowser

# 2. Third party modules
from PySide2.QtCore import Qt
from PySide2.QtGui import QGuiApplication
from PySide2.QtWidgets import QApplication, QDialogButtonBox, QGroupBox, QVBoxLayout

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.cmsflow.gui.advanced_card_table_widget import AdvancedCardTableWidget


class AdvancedDlg(XmsDlg):
    """A dialog for editing advanced cards."""
    def __init__(self, sim_data, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            sim_data (SimulationData): The simulation data to view
            parent (Something derived from :obj:`QWidget`): The parent window
        """
        super().__init__(parent, 'xms.cmsflow.gui.advanced_dlg')
        self.help_url = 'https://cirpwiki.info/wiki/Advanced_Card_file'
        self.sim_data = sim_data
        flags = self.windowFlags()
        self.setWindowFlags(flags & ~Qt.WindowContextHelpButtonHint)
        self.setWindowTitle('Advanced Cards')

        self.vertical_layout = QVBoxLayout(self)
        self.vertical_layout.setObjectName(u"vertical_layout")

        self.cards_group_box = QGroupBox('User defined cards')
        self.cards_vertical_layout = QVBoxLayout()
        self.cards_group_box.setLayout(self.cards_vertical_layout)
        self.cards_table = AdvancedCardTableWidget(
            self.cards_group_box, self.sim_data.advanced_card_table.to_dataframe()
        )
        self.cards_table.setObjectName('cards_table')
        self.cards_table.setMinimumHeight(205)  # min height from the XML
        self.cards_table.setMinimumWidth(400)
        self.cards_vertical_layout.insertWidget(1, self.cards_table)

        self.btn_box = QDialogButtonBox(self)
        self.btn_box.setObjectName(u"btn_box")
        self.btn_box.setOrientation(Qt.Horizontal)
        self.btn_box.setStandardButtons(QDialogButtonBox.Cancel | QDialogButtonBox.Help | QDialogButtonBox.Ok)

        self.btn_box.accepted.connect(self.accept)
        self.btn_box.rejected.connect(self.reject)
        self.btn_box.helpRequested.connect(self.help_requested)

        self.vertical_layout.addWidget(self.cards_group_box)
        self.vertical_layout.addWidget(self.btn_box)

        # Pass the widget size past the scroll area up to the parent dialog
        size = self.vertical_layout.sizeHint()
        geom = self.geometry()
        geom.setSize(size)
        self.setGeometry(geom)

        # center dialog in screen containing mouse, was at top left of all screens
        frame_gm = self.frameGeometry()
        screen = QGuiApplication.screenAt(QApplication.desktop().cursor().pos())
        if screen:
            center_point = screen.geometry().center()
            frame_gm.moveCenter(center_point)
            self.move(frame_gm.topLeft())

    def _save_data(self):
        """Save data."""
        self.sim_data.advanced_card_table = self.cards_table.model.data_frame.to_xarray()

    def help_requested(self):
        """Called when the Help button is clicked."""
        webbrowser.open(self.help_url)

    def accept(self):
        """Save data from dialog on OK."""
        self._save_data()

        super().accept()
