"""Simulation component class."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules
from packaging.version import parse as parse_version

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg
from xms.guipy.time_format import XmsTimeFormatter

# 4. Local modules
from xms.cmswave.components.cmswave_base_component import CmswaveBaseComponent
from xms.cmswave.components.coverage_mapper_runner import CoverageMapperRunner
from xms.cmswave.data.simulation_data import SimulationData
from xms.cmswave.dmi.xms_data import XmsData
from xms.cmswave.gui.model_control_dialog import ModelControlDialog


class SimulationComponent(CmswaveBaseComponent):
    """A hidden Dynamic Model Interface (DMI) component for the CMS-Wave model simulation."""
    def __init__(self, main_file):
        """
        Initializes the data class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data = SimulationData(main_file)
        self.tree_commands = [
            ('Model Control...', 'open_model_control'),
            # ('Generate Snap Preview', 'create_snap_preview'), Not implemented yet
        ]
        if not os.path.isfile(self.main_file):
            self.data.commit()

    def save_to_location(self, new_path, save_type):
        """
        Save component files to a new location.

        Args:
            new_path (:obj:`str`): Path to the new save location.
            save_type (:obj:`str`): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.

                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (:obj:`tuple`): tuple containing:

                new_main_file (:obj:`str`): Name of the new main file relative to new_path, or an absolute path
                if necessary.

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        new_main_file, messages, actions = super().save_to_location(new_path, save_type)
        file_version = self.data.info.attrs.get('VERSION', '0.0.0')
        if parse_version(file_version) < parse_version('1.4.0'):
            if os.environ.get('XMS_OPENING_PROJECT', '') == 'TRUE':
                commit = False
                if self.data.info.attrs['observation_uuid'] != '':
                    self.data.info.attrs['observation'] = 1
                    commit = True
                if self.data.info.attrs['nesting_uuid'] != '':
                    self.data.info.attrs['nesting'] = 1
                    commit = True
                if commit:
                    self.data.commit()
        return new_main_file, messages, actions

    def open_model_control(self, query, params, win_cont):
        """
        Opens the dialog and saves component data state on OK.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS.
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        xms_data = XmsData(query)
        grid_data = {}
        do_ugrid = xms_data.do_ugrid
        if do_ugrid is not None:
            grid_data['cogrid'] = xms_data.cogrid
            grid_data['grid_name'] = do_ugrid.name
        time_formatter = XmsTimeFormatter(query.global_time_settings)
        dialog = ModelControlDialog(
            data=self.data,
            pe_tree=query.project_tree,
            query=query,
            time_format=time_formatter.qt_abs_specifier,
            grid_data=grid_data,
            parent=win_cont
        )
        if dialog.exec_():
            self.data.commit()
        return [], []

    def create_snap_preview(self, query, params, win_cont):
        """
        Creates mapped components to display CMS-Wave data on a grid.

        Args:
            query (:obj:`xmsapi.dmi.Query`): Object for communicating with XMS.
            params (:obj:`dict`): Generic map of parameters. Unused in this case.
            win_cont (:obj:`PySide2.QtWidgets.QWidget`): The window container.

        Returns:
            (:obj:`tuple`): tuple containing:

                messages (:obj:`list[tuple(str)]`): List of tuples with the first element of the
                tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                text.

                action_requests (:obj:`list[xmsapi.dmi.ActionRequest]`): List of actions for XMS to perform.
        """
        sim_uuid = query.parent_item_uuid()
        note = ''
        worker = CoverageMapperRunner(query, self.main_file)
        error_str = 'Error(s) encountered applying coverages to simulation. Review log output for more details.'
        warning_str = 'Warning(s) encountered applying coverages to simulation. Review log output for more details.'
        display_text = {
            'title': 'CMS-Wave Snap Preview',
            'working_prompt': 'Applying coverages to grid. Please wait...',
            'error_prompt': error_str,
            'warning_prompt': warning_str,
            'success_prompt': 'Successfully created snap preview',
            'note': note,
            'auto_load': 'Close this dialog automatically when exporting is finished.'
        }
        feedback_dlg = ProcessFeedbackDlg(display_text, 'xms.cmswave', worker, win_cont)
        feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
        if feedback_dlg.exec():  # Don't send data if user canceled.
            for component in worker.mapped_component:
                query.add_component(do_component=component[0], actions=component[1])
                query.link_item(taker_uuid=sim_uuid, taken_uuid=component[0].uuid)
        return [], []
