"""Module for exporting a CMS-Wave simulation with feedback."""

# 1. Standard Python modules
import logging
import os
import subprocess
import sys

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import LogEchoQSignalStream, ProcessFeedbackDlg, ProcessFeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import ensure_qapplication_exists

# 4. Local modules
from xms.cmswave.file_io.sim_writer import SimWriter
from xms.cmswave.file_io.spectral_coverage_writer import SpectralCoverageWriter
from xms.cmswave.file_io.structures_writer import StructuresWriter
from xms.cmswave.xml_entry_points.parallel_process import write_dataset_script_args


class ExportSimulationRunner(ProcessFeedbackThread):
    """Class for exporting a CMS-Wave simulation in a worker thread."""
    def __init__(self, parent):
        """Constructor.

        Args:
            parent (:obj:`QWidget`): Parent of the QThread, probably the hidden parent dialog created by XMS.
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = logging.getLogger('xms.cmswave')
        self.sim_writer = SimWriter()

    def _do_work(self):
        """Creates the snap preview of coverages onto the mesh."""
        try:
            self._logger.info('Exporting CMS-Wave simulation...')
            if not self.sim_writer.write():
                return  # If we couldn't export the simulation, don't bother with the other files.
            self._logger.info('Exporting CMS-Wave structures file...')
            struct_writer = StructuresWriter(self.sim_writer.query)
            struct_writer.write()
            # Fire off the spatial dataset script in another process before we write the spec file.
            self._logger.info('Exporting CMS-Wave spatial dataset files...')
            write_dataset_script_args(
                self.sim_writer.query, self.sim_writer.cogrid_file, self.sim_writer.data,
                self.sim_writer.simulation_name
            )
            dset_script = os.path.normpath(
                os.path.join(
                    os.path.dirname(os.path.dirname(__file__)), 'xml_entry_points', 'write_spatial_dataset.py'
                )
            )
            dset_proccess = subprocess.Popen([sys.executable, dset_script], env=os.environ)
            # Write the spectral file and wait for the spatial dataset script.
            self._logger.info('Exporting CMS-Wave spectral dataset files...')
            spec_writer = SpectralCoverageWriter(self.sim_writer.query)
            spec_writer.export_coverages()
            if dset_proccess is not None:
                self._logger.info('Waiting for spatial dataset export to complete...')
                dset_proccess.wait()
            self._logger.info('Successfully exported CMS-Wave simulation.')
        except Exception:
            self._logger.exception('Error exporting simulation:')


def export_simulation_with_feedback():
    """Export a simulation with a feedback dialog.

    Returns:
        SimWriter: The writer used to export. Useful when testing.
    """
    # Export all CMS-Wave executables to the same folder.
    ensure_qapplication_exists()
    worker = ExportSimulationRunner(None)
    display_text = {
        'title': 'CMS-Wave Export Simulation',
        'working_prompt': 'Exporting CMS-Wave simulation files. Please wait...',
        'warning_prompt': 'Warning(s) encountered while exporting simulation. Review log output for more details.',
        'error_prompt': 'Error(s) encountered while exporting simulation. Review log output for more details.',
        'success_prompt': 'Successfully exported simulation',
        'note': '',
        'auto_load': 'Close this dialog automatically when exporting is finished.'
    }
    feedback_dlg = ProcessFeedbackDlg(display_text=display_text, logger_name='xms', worker=worker, parent=None)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():
        if LogEchoQSignalStream.logged_error:
            XmEnv.report_export_error()
    else:
        XmEnv.report_export_aborted()
    return worker.sim_writer  # For testing
