"""Reads SRH-2D material file."""

__copyright__ = "(C) Copyright Aquaveo 2020"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules


class StructuresReader:
    """Reads CMS-Wave structures file."""
    def __init__(self, grid, struct_data):
        """Constructor.

        Args:
            grid (:obj:`xms.data_objects.parameters.UGrid`): The grid that has been read in
            struct_data (StructuresData): the structures data for the sim
        """
        self.logger = logging.getLogger('CMS-Wave')
        self.structure_count = 0
        # self.structure_names = {} # Dict of structure id to structure data
        self.structure_cells = {}  # Dict of structure id and list of cells
        self.lines = []  # List of lines from the file
        self.curr_line = 0  # Current line in the file we're reading
        self.line_count = 0  # Number of lines in the file
        self._grid = grid
        self._struct_data = struct_data

        # Make sure the unassigned structure is in the dict
        self.structure_cells[0] = []

    def _read_structures_count(self):
        """Reads the number of structures."""
        if self.curr_line < self.line_count:
            words = self.lines[self.curr_line].split()
            if words and len(words) > 0:
                self.structure_count = int(words[0])
        self.curr_line += 1

    def _read_cells(self):
        """Reads the cells associated with the structure."""
        count = 0
        use_mod = False
        mod_val = 0.0

        while count < self.structure_count:
            words = self.lines[self.curr_line].split()
            type_id = 1
            cell_id = -1
            if words and len(words) > 2:
                use_mod = len(words) > 3
                i = int(words[0])
                j = int(words[1])
                cell_id = self._grid.get_cell_index_from_ij(i, j)
                type_id = int(words[2])
                if use_mod is True:
                    mod_val = float(words[3])

            struct_id = self._struct_data.struct_id_from_info(type_id, use_mod, mod_val)
            cell_list = self.structure_cells.setdefault(struct_id, list())
            cell_list.append(cell_id)
            count += 1
            self.curr_line += 1

    def read(self, filename):
        """Reads the structures file.

        Args:
            filename (:obj:`str`): Filepath of structure file.
        """
        try:
            with open(filename, 'r') as file:
                self.lines = file.read().splitlines()
                self.line_count = len(self.lines)
                self._read_structures_count()
                self._read_cells()
        except Exception as error:  # pragma: no cover
            self.logger.exception(f'Error reading structures: {str(error)}')
            raise error
