"""This is a dialog for specifying CMS-Wave ref time data."""

__copyright__ = "(C) Copyright Aquaveo 2021"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.time_format import datetime_to_string, string_to_datetime

# 4. Local modules
from xms.cmswave.data import cmswave_consts as const
from xms.cmswave.gui.ref_time_dialog_ui import Ui_CMSWAVEReferenceTime


class RefTimeDialog(XmsDlg):
    """A dialog for viewing and editing the ref time and units of the simulation."""
    def __init__(self, data, time_format, parent=None):
        """Initializes the class, sets up the ui.

        Args:
            data (:obj:`SimulationData`): The simulation data for the dialog
            time_format (:obj:`str`): The display format for the datetime picker
            parent (:obj:`QWidget`): The parent window
        """
        super().__init__(parent, 'xms.cmswave.gui.ref_time_dialog')
        self._data = data
        self._ui = Ui_CMSWAVEReferenceTime()
        self._ui.setupUi(self)
        self._setup_dialog(time_format)

    def _setup_dialog(self, time_format):
        """Called when the dialog is loaded to set up the initial values in the dialog.

        Args:
            time_format (:obj:`str`): The display format for the datetime picker
        """
        reftime = string_to_datetime(self._data.info.attrs['reftime'])
        if time_format:
            self._ui.dt_reftime.setDisplayFormat(time_format)
        self._ui.dt_reftime.setDateTime(reftime)
        units = [const.TIME_UNITS_DAYS, const.TIME_UNITS_HOURS, const.TIME_UNITS_MINUTES]
        self._ui.cbx_units.addItems(units)
        try:
            idx = units.index(self._data.info.attrs['reftime_units'])
        except ValueError:
            idx = 1
        if idx != -1:
            self._ui.cbx_units.setCurrentIndex(idx)
        else:
            self._ui.cbx_units.setCurrentIndex(1)

    def accept(self):
        """Handles the accept action."""
        self._data.info.attrs['reftime_units'] = self._ui.cbx_units.currentText()
        self._data.info.attrs['reftime'] = datetime_to_string(self._ui.dt_reftime.dateTime())
        super().accept()
