"""Module for the VisibleCoverageComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"
__all__ = ['VisibleCoverageComponentBase', 'UNASSIGNED_TYPE', 'MULTIPLE_TYPES', 'MessagesAndRequests']

# 1. Standard Python modules
from abc import ABC, abstractmethod
import copy
from functools import cached_property
from pathlib import Path
from typing import final, Optional, Sequence

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Menu, MenuItem, Query
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.components.bases.component_with_menus_base import ComponentWithMenusBase, MenuCallback, MessagesAndRequests
from xms.components.bases.visible_coverage_component_base_data import VisibleCoverageComponentBaseData
from xms.components.display.display_options_helper import (
    DisplayOptionsHelper, FeatureDict, MULTIPLE_TYPES, MULTIPLE_TYPES_LABEL, TARGETS, UNASSIGNED_TYPE
)


class VisibleCoverageComponentBase(ComponentWithMenusBase, ABC):
    """
    A base class for DMI coverage components.

    XMS has numerous model-specific coverages, called "DMI coverages". The coverages (from the user's perspective) are
    actually made of two pieces (from the developer's perspective): a Coverage and a Component.

    Coverage is an object defined in `data_objects`. It contains geometry, such as points, arcs, and polygons, each of
    which has a feature ID. Every type of coverage uses Coverage to define its geometry. This part is always the same
    regardless of what type of coverage the user makes.

    Each Coverage can have a Component attached to it. Components can store arbitrary data, add and implement new menu
    items for the coverage, and assign labels and colors to features. The Component is what makes each coverage type
    different.

    VisibleCoverageComponentBase is a base class for other coverage components to build on. It aims to implement
    functionality that most components will need, and for which a generic implementation is reasonable.
    """
    def __init__(self, main_file: Optional[Path | str] = None):
        """Initializes the base component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)  # CoverageComponentBase doesn't like Path objects.

        #: Commands to show in the graphics window when right-clicking a feature. The first element of each tuple is
        #: the text to display, and the second is the method to call when the item is clicked.
        #:
        #: It is not necessary to add handlers for assigning features. Any targets mentioned in `self.features` will
        #: have assignment menu items created for them that forward to `self._assign_feature()`.
        self.point_commands: list[(str, MenuCallback)] = []
        self.arc_commands: list[(str, MenuCallback)] = []
        self.polygon_commands: list[(str, MenuCallback)] = []

        #: Titles to use for dialogs for assigning feature attributes.
        self.point_dlg_title = 'Point Properties'
        self.arc_dlg_title = 'Arc Properties'
        self.polygon_dlg_title = 'Polygon Properties'

        #: Names to use for storing dialog geometry in the registry. Normally these don't need to be changed, but
        #: they're sometimes useful for referring to. Material coverages, for example, might use the same dialog to
        #: both view materials and assign features.
        self._point_dialog_name = f'{self.module_name}.assign_point_dialog'
        self._arc_dialog_name = f'{self.module_name}.assign_arc_dialog'
        self._polygon_dialog_name = f'{self.module_name}.assign_polygon_dialog'

        #
        # Class setup
        #
        if not isinstance(self.__class__.data, cached_property):
            # Derived classes will almost always override self.data. It's easy to forget and make it a regular property,
            # but it needs to be a cached property because this class reads from it multiple times. If it's a regular
            # property, then each read will get a new instance, and they'll trample on each other and be inconsistent.
            raise AssertionError('self.data is not a cached property')

        self.tree_commands.append(('Display options...', self.open_display_options))

        if TargetType.point in self.features:
            self.point_commands.append(('Assign point attributes...', self._internal_assign_point))
        if TargetType.arc in self.features:
            self.arc_commands.append(('Assign arc attributes...', self._internal_assign_arc))
        if TargetType.polygon in self.features:
            self.polygon_commands.append(('Assign polygon attributes...', self._internal_assign_polygon))

        self._initialize_event_handlers.append(self._request_display_update_wrapper)
        self._initialize_event_handlers.append(self._initialize_component_and_feature_id_mapping)

        self._create_event_handlers.append(self._initialize_coverage_uuid)
        self._create_event_handlers.append(self._initialize_display_json_files)
        self._create_event_handlers.append(self._request_display_update_wrapper)

        self._duplicate_event_handlers.append(self._initialize_coverage_uuid)
        self._duplicate_event_handlers.append(self._reset_display_uuids)
        self._duplicate_event_handlers.append(self._request_display_update_wrapper)

        # At time of writing, when XMS creates a new component from scratch, it appears to follow logic similar to:
        # - Construct the component in the UI event loop
        # - If the constructor created a main-file, run the create event with an initialized query
        # - Otherwise, run the create event with an uninitialized query
        # It's anyone's guess why it does that, but the create event is somewhat pointless with an uninitialized query,
        # so this ensures we get the initialized one.
        self.data.touch()

    @property
    @abstractmethod
    def features(self) -> FeatureDict:
        """The features this coverage supports."""
        # The dict you return here should have one TargetType for each target you support.
        #
        # The value of each key is a list of features. Each feature is a tuple of (name, label), where name is how you
        # want to refer to the feature internally (the display options helper and coverage builder take these), and
        # label is how you want to display it to the user (the graphics window displays these).
        #
        # Example:
        # >>> return {
        # ...     TargetType.point: [  # This supports points
        # ...         ('point_1', 'Point 1'),  # 'point_1' is displayed as 'Point 1'
        # ...         ('point_2', 'Point 2'),  # 'point_2' is displayed as 'Point 2'
        # ...     ],
        # ...     TargetType.arc: [  # And arcs, but note that polygons are absent below, so they're unsupported
        # ...         ('arc_1', 'Arc 1'),  # 'arc_1 is displayed as 'Arc 1'
        # ...     ]
        # ... }
        pass

    @property
    def default_features(self) -> dict[TargetType, str]:
        """
        Defines how unassigned features are rendered.

        Mapping from `target_type -> name`. The name is the first element of one of the tuples returned by
        `self.features`. `self.features[target_type][i][0] == name` must be true for exactly one integer i. The feature
        for which it is true will be considered the default one, and unassigned features will be rendered as if they
        have been assigned that type.

        Targets that are not present in the result will be rendered as if they have no type assigned.
        """
        return {}

    @cached_property
    @abstractmethod
    def data(self) -> VisibleCoverageComponentBaseData:
        """The component's data manager."""
        # Derived classes should override this to return something derived from VisibleCoverageComponentData.
        pass

    @abstractmethod
    def _assign_feature(
        self, parent: QWidget, dialog_name: str, window_title: str, target: TargetType, feature_ids: list[int]
    ):
        """
        Display the Assign feature dialog and persist data if accepted.

        Args:
            parent: Parent widget for any dialog windows created.
            dialog_name: Suggested name for any dialog being shown. By default, the name is based on the component's
                module name and the current target being assigned, making it unique to this particular combination. It
                may be useful as a registry key name to store dialog geometry, or a lookup key for a dialog's Help
                button. It can be overridden if necessary by setting the appropriate self._*_dialog_name member.
            window_title: Suggested title for the window. Defaults to something like "Point Properties", with variations
                for each target type. Can be controlled by setting the appropriate self.*_dlg_title member.
            target: One of point, arc, or polygon.
            feature_ids: Feature IDs of selected features.

        Returns:
            Messages and requests for XMS to handle.
        """
        # A typical implementation would do the following things:
        # - Use self.get_comp_id to get component IDs for the provided feature IDs
        # - Retrieve values for the component ID(s) from component-specific storage
        # - - Components derived from GMI's base components can pass the feature ID directly to self.data instead and
        #     combine this step with the previous one.
        # - Ask the user for new values to assign
        # - Generate a new component ID*
        # - Store the assigned values along with that component ID
        # - Use self.assign_feature_ids to associate the passed feature IDs with the new component ID
        #
        # * This is commonly done by storing a counter in the component's data file. Each time a new component ID is
        #   needed, the counter can be incremented and used as the new ID.
        #
        #   Note that reusing component ID(s) is dangerous, particularly for arcs and polygons. It's possible that XMS
        #   split a feature in two or duplicated it (e.g. during a clean operation), in which case both features will
        #   be assigned the same component ID. Then, if only one of the features is passed in here, you might surprise
        #   the user by reassigning a feature that wasn't selected. Doing this right would entail getting the full
        #   component_id<->feature_id mapping from XMS and either verifying that all the features that share the ID were
        #   selected or reassigning the unselected ones to different IDs. Component IDs are cheap, so it's much simpler
        #   to just use a new one.
        #
        # Overrides can append to self.messages and self.requests if necessary, but most have no need.
        pass

    #
    # Feature assignment
    #
    @final
    def _internal_assign_point(self, query: Query, params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Run the Assign Point dialog.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            parent: Parent widget for any dialog windows created.

        Returns:
            Messages and requests for XMS to handle.
        """
        return self._internal_assign_feature(
            params=params,
            parent=parent,
            query=query,
            dialog_name=self._point_dialog_name,
            window_title=self.point_dlg_title,
            target=TargetType.point
        )

    @final
    def _internal_assign_arc(self, query: Query, params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Opens the Assign arc dialog and saves component data state on OK.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            parent: Parent widget for any dialog windows created.

        Returns:
            Messages and requests for XMS to handle.
        """
        return self._internal_assign_feature(
            params=params,
            parent=parent,
            query=query,
            dialog_name=self._arc_dialog_name,
            window_title=self.arc_dlg_title,
            target=TargetType.arc
        )

    @final
    def _internal_assign_polygon(self, query: Query, params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Opens the Assign polygon dialog and saves component data state on OK.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            parent: Parent widget for any dialog windows created.

        Returns:
            Messages and requests for XMS to handle.
        """
        return self._internal_assign_feature(
            params=params,
            parent=parent,
            query=query,
            dialog_name=self._polygon_dialog_name,
            window_title=self.polygon_dlg_title,
            target=TargetType.polygon
        )

    def _internal_assign_feature(
        self, params: list[dict], parent: QWidget, query: Query, dialog_name: str, window_title: str, target: TargetType
    ) -> MessagesAndRequests:
        """
        Assign new values to a feature.

        This method is meant for use by the base class. Derived classes should override self._assign_feature if they
        want to respond to feature assignment.

        Args:
            params: A list of one dictionary, which has one key, 'selection', whose value is a list of selected
                feature IDs.
            parent: Parent widget to use for any created dialogs.
            query: Interprocess communication object.
            dialog_name: The name of the dialog. Used to store data such as its size and location in the registry.
            window_title: Title to use for the window.
            target: Target type of the feature(s) being assigned.

        Returns:
            Messages and requests for XMS to handle.
        """
        # XMS doesn't tell us what operation is being run, so we have methods like `_internal_assign_point` to figure
        # that out and apply a target and dialog name to the operation. Then they all forward through here so we can do
        # error checking and parameter unpacking in one central location. Then we send it to `_assign_feature` so the
        # derived class can override it to do something useful.
        if not params or 'selection' not in params[0] or not params[0]['selection']:
            return [('INFO', 'No feature selected. Select one or more features to assign boundary conditions.')], []

        selected_features = params[0]['selection']
        if target == TargetType.point:
            query.load_component_ids(self, points=True, only_selected=True)
        elif target == TargetType.arc:
            query.load_component_ids(self, arcs=True, only_selected=True)
        elif target == TargetType.polygon:
            query.load_component_ids(self, polygons=True, only_selected=True)

        self._assign_feature(
            parent=parent,
            dialog_name=dialog_name,
            window_title=window_title,
            target=target,
            feature_ids=selected_features
        )

        return self.messages, self.requests

    #
    # Misc. stuff
    #

    @property
    def cov_uuid(self) -> str:
        """The UUID of the coverage this component is attached to."""
        return self.data.coverage_uuid

    @cov_uuid.setter
    def cov_uuid(self, value: str):
        """
        Set the UUID of the coverage this component is attached to.

        The UUID is persisted, so there is no need to assign it again unless the coverage's UUID changes somehow.

        Args:
            value: The new UUID.
        """
        assert value
        self.data.coverage_uuid = value
        self.data.commit()

    def feature_label_from_name(self, target: TargetType, internal_name: str) -> str:
        """
        Get a feature's label given its internal name.

        Args:
            target: The target type of the feature.
            internal_name: An internal name in self.features.

        Returns:
            The label associated with internal_name.
        """
        for internal, label in self.features[target]:
            if internal == internal_name:
                return label
        # Only use the default if the component hasn't overridden it.
        if internal_name == MULTIPLE_TYPES:
            return MULTIPLE_TYPES_LABEL

        raise ValueError(f'Unknown internal name: {internal_name}')

    def assign_feature_ids(
        self,
        target: TargetType,
        feature_type: str,
        feature_ids: Sequence[int],
        comp_id: int,
        query: Optional[Query] = None
    ):
        """
        Associate feature IDs with a component ID.

        Args:
            target: Target type of the features.
            feature_type: Type of the feature. Should be one of the internal names in self.features.
            feature_ids: IDs of the features to associate.
            comp_id: Component ID to associate the features with.
            query: Interprocess communication object.
        """
        if query:
            query.load_component_ids(self, points=True, arcs=True, polygons=True)

        for feature_id in feature_ids:
            self.update_component_id(target, feature_id, comp_id)

        if feature_type == UNASSIGNED_TYPE:
            return

        label = self.feature_label_from_name(target, feature_type)

        with DisplayOptionsHelper(self.main_file) as helper:
            helper.assign_component_id(target, comp_id, label)
            self.request_display_update(helper)

    def load_coverage_component_id_map(self, file_map: dict[str, tuple[str, str]]) -> None:
        """Reads the binary id files dumped by XMS and populates the internal id map.

        Args:
            file_map: Key is entity type string, value is tuple of two str where the first is file location of the
             xms id file and the second is the file location of the component id file.
        """
        if not self.cov_uuid:
            # This should be initialized by either the create event or the coverage component builder. If it isn't, then
            # the display will be broken.
            raise AssertionError('Coverage UUID was uninitialized')

        super().load_coverage_component_id_map(file_map)

        # The superclass doesn't always initialize self.comp_to_xms. It won't do anything if there are no files, or the
        # files are all empty. The former case is weird - you have to tell Query to load the component IDs, but not
        # request any feature types, which is obviously wrong. The latter case happens whenever the coverage has no
        # assigned features, which happens whenever the user tries to assign the first one. That one is essential.
        dct = self.comp_to_xms.get(self.cov_uuid, {})
        self.data.component_id_map = {}

        for target in TARGETS:
            comp_to_xms: dict[int, list[int]] = dct.get(target, {})  # component_id -> list[feature_id]
            self.data.component_id_map[target] = {}
            for component_id in comp_to_xms:
                for feature_id in comp_to_xms[component_id]:
                    self.data.component_id_map[target][feature_id] = component_id

    def update_component_id(self, entity_type: TargetType, xms_id: int, comp_id: int, label_text: Optional[str] = None):
        """Update a coverage component id mapping to be sent to XMS after an ActionRequest.

        Args:
            entity_type: The type of entity this mapping is for. Valid values are TargetType.point, TargetType.arc,
                TargetType.polygon.
            xms_id: The entity's current id in XMS. Its feature ID. Should get from XMS to ensure it is up-to-date.
            comp_id: The component id to associate with this entity.
            label_text: Display label text for this component id. Will be the display category description text if not
                specified.
        """
        if not self.cov_uuid:
            raise AssertionError('Attempt to update component ID before assigning coverage UUID.')
        super().update_component_id(entity_type, xms_id, comp_id, label_text)

    #
    # Display options
    #
    def open_display_options(self, query: Query, params: list[dict], parent: QWidget) -> MessagesAndRequests:
        """
        Run the display options dialog.

        Args:
            query: Object for communicating with XMS
            params: Generic map of parameters. Contains selection map and component id files.
            parent: The window container.

        Returns:
            Messages and requests for XMS to handle.
        """
        with DisplayOptionsHelper(self.main_file) as helper:
            self.display_option_list = helper.run_display_options_dialog(parent, self.module_name, self.cov_uuid)
        return [], []

    def request_display_update(self, helper: DisplayOptionsHelper):
        """
        Request that the display be updated.

        Args:
            helper: Display options helper.
        """
        messages = helper.get_update_messages(self.cov_uuid)
        self.display_option_list.extend(messages)

    def _request_display_update_wrapper(self, _query: Query):
        """Wrapper for self.request_display_update that makes it compatible with the create event handler."""
        with DisplayOptionsHelper(self.main_file) as helper:
            self.request_display_update(helper)

    def update_feature_types(self, target: TargetType):
        """
        Update the features supported for a given target.

        This is useful for when the set of supported features has changed, e.g. because a new material was added. After
        calling this, the display options will support all the features in `self.features` for the given target.

        Args:
            target: Target type to update.
        """
        with DisplayOptionsHelper(self.main_file) as helper:
            helper.update_features(target, self.features[target])
        self.refresh_display_id_files([target])

    #
    # Menus
    #
    def get_display_menus(
        self, selection: dict[str, list[int]], lock_state: bool, id_files: dict[str, tuple[str, str]]
    ) -> list[Menu | MenuItem | None]:
        """
        Get a list of menu items for the graphics view.

        Args:
            selection: A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of ids of the selected feature objects.
            lock_state: True if the component is locked for editing. Do not change the files if locked.
            id_files: Key is entity type string, value is tuple of two str where first is the file location of the XMS
                coverage id binary file. Second is file location of the component coverage id binary file. Only
                applicable for coverage selections. File will be deleted after event. Copy if need to persist.

        Returns:
            A list of menus and menu items to be shown. Note that this list can have objects of type `xms.api.dmi.Menu`
                as well as `xms.api.dmi.MenuItem`. `None` may be added to the list to indicate a separator.
        """
        menu_list = [None]  # None == spacer
        items = [('POINT', self.point_commands), ('ARC', self.arc_commands), ('POLYGON', self.polygon_commands)]

        for key, commands in items:
            if key in selection and selection[key]:
                for command_text, command in commands:
                    params = {'selection': selection[key]}
                    item = self._make_menu_item(text=command_text, command=command, params=params)
                    menu_list.append(item)

        return menu_list

    def get_double_click_actions_for_selection(
        self, selection: dict[str, list[int]], lock_state: bool, id_files: dict[str, tuple[str, str]]
    ) -> MessagesAndRequests:
        """
        Get the menu item for a double click in the graphics view.

        Args:
            selection: A dictionary with the key being a string of the feature entity type (POINT, ARC, POLYGON).
                The value of the dictionary is a list of ids of the selected feature objects.
            lock_state: True if the component is locked for editing. Do not change the files if locked.
            id_files: Key is entity type string, value is tuple of two str where first is the file location of the XMS
                coverage id binary file. Second is file location of the component coverage id binary file. Only
                applicable for coverage selections. File will be deleted after event and should be copied if persistence
                is needed.

        Returns:
            Messages and requests for XMS to handle.
        """
        menus = self.get_display_menus(selection, lock_state, id_files)
        # If selected item menu commands have been defined, the first will be a separator, and the
        # second will be the double click action.
        actions = [menus[1].action_request] if len(menus) >= 2 else []  # The first one is None for a separator.
        return [], actions

    #
    # Component creation
    #
    def get_create_action(self) -> ActionRequest:
        """
        Get an ActionRequest for the component's create event.

        When XMS creates a component from scratch, it runs the component's create event. When it receives a component
        from an import script, it gets the component's initial display options, which we forward to the create event.
        But when it receives a component from an ActionRequest callback, it does absolutely nothing at all, so the
        create event never runs. Sending this request back to XMS will prompt it to create the component after all.
        """
        return self._make_request(self._create_event, needs_window=False, params={'locked': False})

    def _initialize_coverage_uuid(self, query: Query, _locked: bool = False):
        """
        Get the coverage UUID from XMS.

        Args:
            query: Object for communicating with XMS.
            _locked: Whether the component is locked for editing. Ignored.
        """
        self.cov_uuid = query.parent_item_uuid()  # The query should be at the component, so the coverage is our parent.
        if not self.cov_uuid:
            self.messages.append(('ERROR', 'Could not get the coverage UUID.'))

    def _initialize_display_id_files(self, _query: Query):
        """Wrapper for self.refresh_display_id_files that makes it compatible with the create event."""
        self.refresh_display_id_files()

    def _initialize_component_and_feature_id_mapping(self, _query: Query):
        """Tell XMS about the mapping between component and feature IDs."""
        with DisplayOptionsHelper(self.main_file) as helper:
            for target, component_id, feature_id in helper.features_needing_update():
                self.update_component_id(target, feature_id, component_id)

    def _initialize_display_json_files(self, _query: Optional[Query] = None, _locked: bool = False):
        """
        Ensure that the display .json files exist.

        This method is safe to call multiple times as long as the cost of hitting the filesystem isn't a problem. It
        won't do anything if the files already exist.
        """
        with DisplayOptionsHelper(self.main_file) as helper:
            helper.initialize_display_options(self.uuid, self.module_name, self.features, self.default_features)

    def _reset_display_uuids(self, _query: Optional[Query] = None, _locked: bool = False):
        """Reset the UUIDs in the component's display .json files to match this component."""
        with DisplayOptionsHelper(self.main_file) as helper:
            helper.reset_display_uuids(self.uuid)

    def refresh_display_id_files(self, targets: Optional[list[TargetType]] = None):
        """
        Write or rewrite the component ID files that XMS uses to determine how each feature should be displayed.

        Args:
            targets: Target types to write. If None, writes all of them.
        """
        features = self.features  # retrieving this results in a read from disk in xmsgmi.
        if targets is None:
            targets = list(features.keys())

        with DisplayOptionsHelper(self.main_file) as helper:
            for target in targets:
                items = copy.copy(features[target])
                items.append((MULTIPLE_TYPES, MULTIPLE_TYPES_LABEL))
                for name, label in items:
                    component_ids = self.data.component_ids_with_type(target, name)
                    helper.assign_component_ids_for_feature_type(target, label, component_ids)

                self.request_display_update(helper)
