"""Xy Series reading/writing."""

__copyright__ = "(C) Copyright Aquaveo 2022"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime
import logging
from pathlib import Path

# 2. Third party modules
import h5py
from h5py import Group

# 3. Aquaveo modules

# 4. Local modules
from xms.coverage.xy.xy_series import XYS, XySeries


class XyReader:
    """Reads xy series from files."""
    def __init__(
        self, date_times_to_floats: bool = False, start_date_time: datetime | None = None, time_units: str = None
    ):
        """Initializes the class.

        Args:
            date_times_to_floats: If true and start_date_time and time_units are given, converts date/time
             strings to relative offsets from the start_date_time.
            start_date_time: Starting date/time.
            time_units: Time units: 'UNKNOWN', 'SECONDS', 'MINUTES', 'HOURS', 'DAYS', 'YEARS'
        """
        self._date_times_to_floats = date_times_to_floats
        self._start_date_time = start_date_time
        self._time_units = time_units
        self._current_id = None

    def read_from_text_file(self, xy_filename: str | Path) -> list[XySeries]:
        """Reads the XY series text file and returns it as a list.

        Args:
            xy_filename: File path of xy file.

        Returns:
            (list[XySeries]): The list of XySeries.
        """
        with open(xy_filename, 'r') as file:
            return self.read_from_text_stream(file)

    def read_from_text_stream(self, file):
        """Reads the XY series text stream and returns it as a list.

        Args:
            file: xy file stream.

        Returns:
            (list[XySeries]): The list of XySeries.
        """
        xy_series_list = []
        for line_number, line in enumerate(file):
            if line.lower().startswith('xys') or line.lower().startswith('xy1'):
                series = XySeries.from_file(
                    file, line, line_number, self._date_times_to_floats, self._start_date_time, self._time_units
                )
                if series:
                    xy_series_list.append(series)
        return xy_series_list

    def read_from_open_h5(self, all_xy_group: Group) -> list[XySeries]:
        """Reads all the xy series in all_xy_group and returns them as a dict[int, XySeries].

        Args:
            all_xy_group: H5 group containing all the xy series.

        Returns:
            (list[XySeries]): The list of XySeries.
        """
        xy_series_list = []
        count = 1
        while True:
            xy_group = all_xy_group.get(f'XYS{count}')
            if not xy_group:
                break
            series = XySeries.from_h5(xy_group)
            xy_series_list.append(series)
            count += 1
        return xy_series_list

    def read_from_h5(self, filepath: Path, all_xy_group_name: str = 'Project/XYS') -> list[XySeries]:
        """Opens the .h5 file and reads the XySeries, returning them in a dict.

        Args:
            filepath (Path): File path of .h5 file.
            all_xy_group_name (str): Name of group where all xy series are stored.

        Returns:
            (list[XySeries]): The list of XySeries.
        """
        if not filepath.is_file():
            return []

        xy_series_list = []
        h5_file = h5py.File(str(filepath), 'r')
        try:
            if not (all_xy_group := h5_file.get(all_xy_group_name)):
                h5_file.close()
                return xy_series_list

            xy_series_list = self.read_from_open_h5(all_xy_group)
        except Exception as e:  # pragma: no cover - I don't want to test the exception
            logger = logging.getLogger('xms.coverage')
            logger.error(str(e))

        h5_file.close()
        return xy_series_list


def write_to_h5(filepath: Path, xy_series_list: list[XySeries], all_xy_group_name: str = 'Project/XYS') -> None:
    """Creates a new file, writes the xy series to it, and closes the file.

    Args:
        filepath (Path): The file path.
        xy_series_list (list[XySeries]): The list of XySeries.
        all_xy_group_name (str): Name of group where all xy series are stored.
    """
    h5_file = h5py.File(str(filepath), 'w')
    try:  # Use try/except block to make sure h5 file gets closed even if there's an error
        all_xy_group = h5_file.create_group(all_xy_group_name)
        write_to_open_h5(all_xy_group, xy_series_list)
    except Exception as e:  # pragma: no cover - I don't want to test the exception
        logger = logging.getLogger('xms.coverage')
        logger.error(str(e))
    h5_file.close()


def write_to_open_h5(all_xy_group: Group, xy_series_list: list[XySeries]) -> None:
    """Writes the xy series to the h5 group in an already opened file.

    Args:
        all_xy_group (Group): The H5 group to write all the XySeries to.
        xy_series_list (list[XySeries]): The list of XySeries.
    """
    for i, xy_series in enumerate(xy_series_list):
        group_name = f'{XYS}{i + 1}'
        group = all_xy_group.create_group(group_name)
        xy_series.to_h5(group)
