"""Methods for writing XMDF data set files."""

# 1. Standard Python modules
from array import array
import os
from typing import Optional, Sequence, Tuple, Union

# 2. Third party modules
import h5py
import orjson

# 3. Aquaveo modules

# 4. Local modules
from xms.datasets.dataset_metadata import DatasetMetadata

# This is a reasonable default null value
# It can be any floating point value for example -8888, 8888 or 1.2345
DSET_NULL_VALUE = -9999999.0


def write_dset_metadata_to_json(filename: str, metadata: DatasetMetadata) -> None:
    """Writes dataset metadata to a JSON file.

    Args:
         filename (:obj:`str`): Fileystem path to the output file. Will be replaced if exists.
         metadata (:obj:`DatasetMetadata`): The dataset metadata

    """
    # For now, this is all we need to get functionality for replacing the cell top and bottom datasets of a UGrid.
    # We could include other stuff like null values, reftimes, transient time files, etc. to eliminate reliance
    # on XMDF in data_objects.
    with open(filename, 'wb') as file:
        data = orjson.dumps(dict(metadata))
        file.write(data)


def read_dset_metadata_to_json(filename: str) -> DatasetMetadata:
    """Reads dataset metadata from a JSON file.

    Args:
         filename (:obj:`str`): Fileystem path to the input JSON file.

    Returns:
         metadata (:obj:`DatasetMetadata`): The dataset metadata

    """
    metadata = DatasetMetadata()
    if os.path.isfile(filename):
        with open(filename, 'rb') as file:
            metadata.from_dict(orjson.loads(file.read()))
    return metadata


def write_dimension_values(filename: str, values: Sequence[float]) -> None:
    """Write an array of float dataset values to a file.

    Args:
        filename (:obj:`str`): Filesystem path of the output file
        values (:obj:`list[float]`): The values of the dataset dimension

    """
    dim_array = array('d', values)
    with open(filename, 'wb') as file:
        dim_array.tofile(file)


def read_dimension_values(filename: str) -> Sequence[float]:
    """Read an array of float dataset dimension values from a binary file.

    Args:
        filename (:obj:`str`): Filesystem path of the input binary file

    Returns:
        (:obj:`list[float]`): The dataset values for the dimension in the file

    """
    if os.path.isfile(filename):
        dim_array = array('d', [])
        with open(filename, 'rb') as file:
            dim_array.fromfile(file, int(os.path.getsize(file.name) / 8))
        return dim_array
    return []


def find_min_and_max(values: Sequence[float],
                     activity: Sequence[float]) -> Tuple[Union[float, None], Union[float, None]]:
    """Returns the min and max values while considering activity.

    Args:
        values: List of values.
        activity: List of activity, same size as values.

    Returns:
        (:obj:`tuple`): tuple containing:

            minimum value

            maximum value

    """
    assert len(values) == len(activity)
    mn = None
    mx = None
    for v, a in zip(values, activity):
        if a:
            if mn is not None:
                mn = min(mn, v)
                mx = max(mx, v)
            else:
                mn = v
                mx = v
    return mn, mx


def update_geom_uuid(filename: str, geom_uuid: str, multi_datasets_group: Optional[str] = 'Datasets') -> None:
    """Update the geometry UUID in an XMDF dataset file.

    Args:
        filename (:obj:`str`): Path to the XMDF dataset file. All file handles should be closed, we are going to open
            it for writing. If you have an open file handle, this will throw.
        geom_uuid (:obj:`str`): The geometry's UUID
        multi_datasets_group (:obj:`Optional[str]`): H5 group path to the XMDF multi-datasets group
    """
    with h5py.File(filename, 'a') as f:
        multi_grp = f[multi_datasets_group]
        if 'Guid' in multi_grp:  # Geometry UUID property exists, update it
            multi_grp['Guid'][0] = geom_uuid.encode('ascii', 'ignore')
        else:  # Geometry UUID property does not exist, create it
            multi_grp.create_dataset('Guid', shape=(1, ), dtype='S37', data=[geom_uuid.encode('ascii', 'ignore')])
        # Make sure the multi-datasets 'Grouptype' attribute is set
        if 'Grouptype' not in multi_grp.attrs:
            ascii_list = ['MULTI DATASETS'.encode('ascii', 'ignore')]
            multi_grp.attrs.create('Grouptype', data=ascii_list, shape=(1, ), dtype='S15')
