"""Worker thread for the solution importer."""
__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import logging
from pathlib import Path

# 2. Third party modules
from PySide2.QtCore import Signal

# 3. Aquaveo modules
from xms.api.dmi import XmsEnvironment as XmEnv
from xms.guipy.dialogs.process_feedback_dlg import ProcessFeedbackDlg, ProcessFeedbackThread

# 4. Local modules
from xms.funwave.dmi.xms_data import XmsData
from xms.funwave.file_io.input_txt_reader import read_input_txt_values
from xms.funwave.file_io.read_solution import read_solution


class SolutionLoadWorkerThread(ProcessFeedbackThread):
    """Worker thread for importing FUNWAVE solutions."""
    processing_finished = Signal()

    def __init__(self, parent, input_txt_file, geom_uuid, xms_data):
        """Construct the worker.

        Args:
            parent (:obj:`QWidget`): The Qt parent
            input_txt_file (:obj:`str`): Path to the control file
            geom_uuid (:obj:`str`): The UUID of the grid
            xms_data (:obj:`XmsData`): Simulation data retrieved from SMS
        """
        super().__init__(parent=parent, do_work=self._do_work)
        self._logger = logging.getLogger('xms.funwave')
        self._input_txt_file = input_txt_file
        self._geom_uuid = geom_uuid
        self._xms_data = xms_data
        self.datasets = []

    def _do_work(self):
        """Thread runner that exports the fort.14."""
        # Look for a mapped BC coverage first.
        self._logger.info('Loading FUNWAVE simulation run solution files...')

        try:
            input_values = read_input_txt_values(Path(self._input_txt_file))
            output_folder = Path(self._input_txt_file).parent / input_values.get('RESULT_FOLDER', 'output')
            m_glob = input_values.get('Mglob')
            n_glob = input_values.get('Nglob')
            if m_glob is None or n_glob is None:
                raise ValueError('Unable to find grid rows and columns.')
            m_glob = int(m_glob)
            n_glob = int(n_glob)
            value_count = m_glob * n_glob
            geom_uuid = self._geom_uuid
            binary = input_values.get('FIELD_IO_TYPE', 'ASCII') == 'BINARY'
            self.datasets = read_solution(output_folder, value_count, geom_uuid, binary, self._xms_data)
        except Exception:
            self._logger.exception('Error(s) encountered while loading FUNWAVE solution files.')

    def add_datasets(self, query):
        """Send imported data to XMS.

        Args:
            query (:obj:`Query`): XMS interprocess communication object
        """
        for dataset in self.datasets:
            query.add_dataset(dataset)


def import_solution_with_feedback(input_txt_file, geom_uuid, query, parent):
    """Entry point to read a FUNWAVE solution.

    Args:
        input_txt_file (:obj:`str`): Path to the control file
        query (:obj:`Query`): XMS interprocess communication object
        geom_uuid (:obj:`str`): The UUID of the grid
        parent (:obj:`QWidget`): The Qt parent
    """
    # The Context for this ActionRequest is at some temporary model run component, we want the parent simulation.
    xms_data = XmsData(query)
    worker = SolutionLoadWorkerThread(parent, input_txt_file, geom_uuid, xms_data)
    display_text = {
        'title': 'Loading FUNWAVE Solution Files',
        'working_prompt': 'Reading FUNWAVE solution files. Please wait...',
        'warning_prompt': 'Warning(s) encountered reading solution. Review log output for more details.',
        'error_prompt': 'Error(s) encountered reading solution. Review log output for more details.',
        'success_prompt': 'Successfully read FUNWAVE solution',
        'note': '',
        'auto_load': 'Auto load solution datasets into SMS when operation is complete',
    }
    feedback_dlg = ProcessFeedbackDlg(display_text, 'xms', worker, parent)
    feedback_dlg.testing = XmEnv.xms_environ_running_tests() == 'TRUE'
    if feedback_dlg.exec():  # Send even if errors because we may have a partial solution
        worker.add_datasets(query)
