"""This module allows SMS to run GenCade and read its solution on completion."""

__copyright__ = "(C) Copyright Aquaveo 2024"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os

# 2. Third party modules

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, ExecutableCommand
from xms.api.tree import tree_util
from xms.components.bases.run_base import RunBase

# 4. Local modules
# from xms.cmsflow.feedback.read_solution_worker_thread import load_solution_file


class SimRunner(RunBase):
    """Class that handles running GENCADE.

    """

    def __init__(self, dummy_mainfile=''):
        """Initializes the class.

        Args:
            dummy_mainfile (str): Unused. Just to keep constructor consistent with component classes.

        """
        super().__init__()
        self.proj_name = None
        self.grid_uuid = None

    @staticmethod
    def read_solution(query, params, win_cont):
        """Reads the GenCade Solution.

        Args:
            query (:obj:`data_objects.parameters.Query`): a Query object to communicate with SMS.
            params (:obj:`dict`): Generic map of parameters. Contains the structures for various components that
             are required for adding vertices to the Query Context with Add().
            win_cont (QWidget): The parent window

        Returns:
            (:obj:`tuple`): tuple containing:
                - messages (:obj:`list` of :obj:`tuple` of :obj:`str`): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests(:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): List of actions for XMS to perform.
        """
        # proj_name = params[0]['project_name']
        # file_location = params[0]['file_location']
        # grid_uuid = params[0]['grid_uuid']
        return  # load_solution_file(query, win_cont, file_location, proj_name, grid_uuid)

    def get_executables(self, sim, query, filelocation):
        """Get the executable commands for any Simulation object given.

        This function will find the correct information that you need for your Simulation object. This function
        determines the correct executables needed, and the correct import scripts needed to load solutions. This
        function determines the correct progress plots needed.

        Args:
            sim (:obj:`data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`data_objects.parameters.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xmsapi.dmi.ExecutableCommand`):
                The executable objects to run and the action requests that go with it.

        """
        # Get the project name
        if not self.proj_name:
            self._get_sms_data(query)
        # load_sol = self.get_solution_load_actions(sim, query, filelocation)[0]
        cmd = ExecutableCommand(executable='GenCade Beta', model='GenCade', executable_order=0, display_name='GenCade',
                                run_weight=100, progress_script='gencade_progress.py')
        cmd.add_commandline_arg(f'{self.proj_name}.gen')
        # cmd.add_solution_file(load_sol)
        return [cmd]

    def get_solution_load_actions(self, sim, query, filelocation):
        """Get the simulation load ActionRequests for any Simulation object given.

        This method is called when we are loading an existing solution from a previous model run. get_executables is
        called when running or rerunning a simulation.

        Args:
            sim (:obj:`data_objects.parameters.Simulation`): The Simulation you want to load the solution for.
            query (:obj:`data_objects.parameters.Query`): a Query object to communicate with SMS.
            filelocation (str): The location of input files for the simulation.

        Returns:
            (:obj:`list` of :obj:`xmsapi.dmi.ActionRequest`): The solution load ActionRequests for the simulation

        """
        if not self.proj_name:
            self._get_sms_data(query)

        # Create an ActionRequest to load the solution located in file_location. Just give it a dummy main file.
        parameters = {'project_name': self.proj_name, 'file_location': filelocation, 'grid_uuid': self.grid_uuid}
        load_sol = ActionRequest(main_file='foo', modality='MODAL', parameters=parameters, class_name='SimRunner',
                                 module_name='xms.gencade.dmi.sim_runner', method_name='read_solution')
        return [load_sol]

    def _get_sms_data(self, query):
        """Gets the project name, grid_uuid, and the temporary directory for the running XMS process.

        Args:
            query (Query): A class to communicate with SMS.
        """
        # Set self.proj_name
        proj_result = os.path.basename(query.xms_project_path)
        if proj_result:
            self.proj_name = os.path.splitext(proj_result)[0]

        # Get the UUID of the linked Quadtree
        sim_uuid = query.current_item_uuid()
        sim_item = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
        grid_item = tree_util.descendants_of_type(sim_item, xms_types=['TI_UGRID_PTR'], allow_pointers=True,
                                                  recurse=False, only_first=True)
        self.grid_uuid = grid_item.uuid if grid_item else None

        # Get temp directory
        self.xms_temp_dir = os.path.join(query.xms_temp_directory, 'Components')
