"""This is a table widget for specifying wave events."""

# 1. Standard Python modules
import datetime

# 2. Third party modules
from PySide2.QtWidgets import QAbstractItemView

# 3. Aquaveo modules
from xms.guipy.delegates.edit_field_validator import EditFieldValidator
from xms.guipy.models.rename_model import RenameModel
from xms.guipy.validators.qx_double_validator import QxDoubleValidator

# 4. Local modules
from xms.gencade.gui.gencade_table_widget import GenCadeTableWidget


class WaveTableWidget(GenCadeTableWidget):
    """Table widget base class."""
    def __init__(self, parent, data_frame):
        """Construct the widget.

        Args:
            parent (Something derived from :obj:`QObject`): The parent object.
            data_frame (:obj:`pandas.DataFrame`): The model data.
        """
        self.rename_model = None
        self.edit_delegate = None
        self.dbl_validator = None

        # set the first column to datetime64
        data_frame[data_frame.columns[0]] = data_frame[data_frame.columns[0]].astype(dtype='datetime64[ns]')

        super().__init__(parent, data_frame, 0, {'wave_date': [datetime.datetime(2000, 1, 1)],
                                                 'wave_height': 0.0, 'wave_period': 1.0, 'wave_direction': 0.0},
                         time_format='%Y-%m-%d %H:%M:%S')

    def setup_ui(self):
        """Sets up the delegates, validators, and filter model for this table."""
        # All wave-related units are will be in metric
        self.rename_model = RenameModel(['Date/Time', 'H0 (m)', 'Period (sec)',
                                         'Direction (deg)'], self)
        self.dbl_validator = QxDoubleValidator(parent=self)
        self.edit_delegate = EditFieldValidator(self.dbl_validator)
        delegates = {1: self.edit_delegate, 2: self.edit_delegate, 3: self.edit_delegate}
        super()._setup_ui(delegates, False, False, self.rename_model)
        # Select the entire row
        self.table_view.setSelectionBehavior(QAbstractItemView.SelectRows)
