"""GmiSimComponentBase class."""

__copyright__ = '(C) Copyright Aquaveo 2022'
__license__ = 'All rights reserved'
__all__ = ['SimComponentBase']

# 1. Standard Python modules
from abc import abstractmethod
from functools import cached_property
from pathlib import Path
from typing import Optional

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.api.dmi import Query
from xms.components.bases.sim_component_base import SimComponentBase as ComponentsSimComponentBase

# 4. Local modules
from xms.gmi.component_bases.dataset_selector_mixin import DatasetSelectorMixin
from xms.gmi.data.generic_model import Section
from xms.gmi.data_bases.sim_base_data import SimBaseData
from xms.gmi.gui.section_dialog import SectionDialog


class SimComponentBase(DatasetSelectorMixin, ComponentsSimComponentBase):
    """A hidden Dynamic Model Interface (DMI) component for GMI-based model simulations."""
    def __init__(self, main_file: Optional[str | Path] = None):
        """Initializes the component class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.data.touch()
        self._section_dialog_keyword_args = {'dataset_callback': self._dataset_callback}
        self._query: Optional[Query] = None

    @abstractmethod
    def _get_global_parameter_section(self) -> Section:
        """Get the global parameter section."""
        pass  # pragma: nocover

    @cached_property
    def data(self) -> SimBaseData:
        """The component's data manager."""
        return SimBaseData(self.main_file)

    def _open_model_control(self, dialog_name: str, query: Query, parent: QWidget):
        """
        Run the model control dialog.

        Args:
            dialog_name: A name that can be used as a registry key for loading and saving dialog settings, such as its
                geometry and active selections.
            query: Interprocess communication object.
            parent: Parent widget for the dialog.
        """
        self._query = query
        section = self._get_global_parameter_section()
        values = self.data.global_values
        section.restore_values(values)

        dlg = SectionDialog(
            parent=parent,
            section=section,
            dlg_name=dialog_name,
            window_title='Model control',
            **self._section_dialog_keyword_args,
        )
        if dlg.exec():
            values = dlg.section.extract_values()
            self.data.global_values = values
            self.data.commit()
