"""Data class for GMI simulation component."""

# 1. Standard Python modules
import os
from pathlib import Path

# 2. Third party modules

# 3. Aquaveo modules

# 4. Local modules
from xms.gmi.data.generic_model import GenericModel
from xms.gmi.data_bases.base_data import BaseData


class SimBaseData(BaseData):
    """Manages data file for a simulation component."""
    def __init__(self, data_file: str | Path):
        """
        Initialize the data class.

        Args:
            data_file: The netcdf file (with path) associated with this instance data. Probably the owning
                       component's main file. If not provided, a default one will be chosen.
        """
        super().__init__(data_file)

        self.info.attrs['FILE_TYPE'] = 'GMI_SIM_BASE'

        self.commit()

    @property
    def global_values(self) -> str:
        """Values for global parameters."""
        if 'GLOBAL_VALUES' in self.info.attrs:
            return self.info.attrs['GLOBAL_VALUES']
        else:  # Let GenericModel define the default ones so it can change them later if necessary.
            return GenericModel().global_parameters.extract_values()

    @global_values.setter
    def global_values(self, value: str):
        """Values for global parameters."""
        self.info.attrs['GLOBAL_VALUES'] = value

    def update_file_paths(self):
        """Called before resaving an existing project.

        All referenced filepaths should be converted to relative from the project directory. Should already be stored
        in the component main file since this is a resave operation.

        Returns:
            (str): Message on failure, empty string on success
        """
        proj_dir = self.info.attrs['proj_dir']
        if not os.path.exists(proj_dir):
            return 'Unable to update selected file paths to relative from the project directory.'
        self.commit()
        return ''

    def copy_external_files(self):
        """Called when saving a project as a package. All components need to copy referenced files to the save location.

        Returns:
            (str): Message on failure, empty string on success
        """
        # Wipe the stored project directory. Paths will be absolute in GUI until resave.
        self.info.attrs['proj_dir'] = ''
        return ''

    def update_proj_dir(self):
        """Called when saving a project for the first time or saving a project to a new location.

        All referenced filepaths should be converted to relative from the new project location. If the file path is
        already relative, it is relative to the old project directory. After updating file paths, update the project
        directory in the main file.

        Returns:
            (str): Message on failure, empty string on success.
        """
        try:
            # Get the new project location. Three directories above the component UUID folder.
            comp_folder = os.path.dirname(self._filename)
            new_proj_dir = os.path.normpath(os.path.join(comp_folder, '../../..'))
            self.info.attrs['proj_dir'] = new_proj_dir
            self.commit()  # Save the updated project directory and referenced filepaths.
            return ''  # Don't report errors, leave that to model checks.
        except Exception:
            return (
                'There was a problem updating file GMI Model Control paths to be relative from the'
                ' project directory. Any selected file paths will remain absolute.\n'
            )
