"""ListSelector class."""

__copyright__ = "(C) Copyright Aquaveo 2019"
__license__ = "All rights reserved"

# 1. Standard Python modules
from typing import Iterable

# 2. Third party modules
from PySide2.QtWidgets import (QCheckBox, QDialog, QDialogButtonBox, QListWidgetItem, QRadioButton, QWidget)

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.dialogs import dialog_util
from xms.guipy.dialogs.list_selector_dialog_ui import Ui_ListSelectorDialog
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg


def run(
    title: str,
    items: Iterable,
    heading: str = '',
    multi_select: bool = True,
    cancel_button: bool = True,
    parent: QWidget = None
) -> list[str] | None:
    """Runs the dialog and returns the list of selected items, or None on Cancel.

    Args:
        title (str): Dialog title
        items (list[str]): The items in the list. Must be strings.
        heading (str): Optional line of text just above the list
        multi_select (bool): True if user can select multiple items. Will use checkboxes instead of radio buttons
        cancel_button (bool): True to show a cancel button
        parent (Something derived from QWidget): The parent window.

    Returns:
        See description.
    """
    dialog_util.ensure_qapplication_exists()
    dialog = ListSelectorDialog(title, items, heading, multi_select, cancel_button, parent)
    if dialog.exec() == QDialog.Accepted:
        return dialog.selected_items()
    return None


class ListSelectorDialog(XmsDlg):
    """A dialog for selecting items from a list."""
    def __init__(
        self,
        title: str,
        items: Iterable,
        heading: str = '',
        multi_select: bool = True,
        cancel_button: bool = True,
        parent: QWidget = None
    ):
        """Initializes the class, sets up the ui.

        Args:
            title (str): Dialog title
            items (list[str]): The items in the list. Must be strings.
            heading (str): Optional line of text just above the list
            multi_select (bool): True if user can select multiple items. Will use checkboxes instead of radio buttons
            cancel_button (bool): True to show a cancel button
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent, 'xms.guipy.dialogs.list_selector')
        self.items = items
        self.multi_select = multi_select

        self.ui = Ui_ListSelectorDialog()
        self.ui.setupUi(self)
        self.setWindowTitle(title)
        self.setup_heading(heading)
        self.setup_list(multi_select)
        self.setup_ok_cancel(cancel_button)

    def setup_heading(self, heading):
        """Sets up the heading text."""
        self.ui.txt_heading.setText(heading)
        self.ui.txt_heading.setVisible(heading != '')

    def setup_list(self, multi_select):
        """Add the widgets.

        Args:
            multi_select (bool): True if user can select multiple items
        """
        for item in self.items:
            list_widget_item = QListWidgetItem(self.ui.lst_widget)
            if multi_select:
                self.ui.lst_widget.setItemWidget(list_widget_item, QCheckBox(item))
            else:
                self.ui.lst_widget.setItemWidget(list_widget_item, QRadioButton(item))

    def setup_ok_cancel(self, cancel_button):
        """Hides the cancel button if necessary.

        Args:
            cancel_button (bool): True to show a cancel button
        """
        self.ui.buttonBox.button(QDialogButtonBox.Cancel).setVisible(cancel_button)

    def selected_items(self):
        """Returns a list of the selected items.

        Returns:
            (list[str]): See description.
        """
        selected_items = []
        for row in range(self.ui.lst_widget.count()):
            list_widget_item = self.ui.lst_widget.item(row)
            item_widget = self.ui.lst_widget.itemWidget(list_widget_item)
            if item_widget.isChecked():
                selected_items.append(item_widget.text())
                if not self.multi_select:
                    break
        return selected_items
