"""Widget for defining XMS display options."""

# 1. Standard Python modules
from enum import IntEnum

# 2. Third party modules
from PySide2.QtCore import QModelIndex, QSize, QSortFilterProxyModel, Qt, Slot
from PySide2.QtGui import QFont, QStandardItem, QStandardItemModel
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules

# 4. Local modules
from xms.guipy.data.category_display_option import CategoryDisplayOption
from xms.guipy.data.category_display_option_list import CategoryDisplayOptionList
from xms.guipy.data.line_style import LineOptions
from xms.guipy.data.point_symbol import PointOptions
from xms.guipy.data.polygon_texture import PolygonOptions
from xms.guipy.data.target_type import TargetType
from xms.guipy.delegates.check_box_no_text import CheckBoxNoTextDelegate
from xms.guipy.delegates.display_option import DisplayOptionDelegate
from xms.guipy.dialogs.line_display_options import LineDisplayOptionsDialog
from xms.guipy.dialogs.point_display_options import PointDisplayOptionsDialog
from xms.guipy.dialogs.polygon_display_options import PolygonDisplayOptionsDialog
from xms.guipy.validators.qx_double_validator import QxDoubleValidator
from xms.guipy.widgets.category_display_options_list_ui import Ui_CategoryDisplayOptionsWidget
from xms.guipy.widgets.check_box_header import CheckBoxHeader
from xms.guipy.widgets.display_option_icon_factory import DisplayOptionIconFactory

BTN_WIDTH = 80
CHECK_WIDTH = 25

ROLE_STYLE_OPTIONS = Qt.UserRole
ROLE_LABEL_OPTIONS = Qt.UserRole
ROLE_DESCRIPTION_LABELS = Qt.UserRole + 1
ROLE_FILE = Qt.UserRole
ROLE_ID = Qt.UserRole + 1
ROLE_UNASSIGNED_CATEGORY = Qt.UserRole + 2


class Columns(IntEnum):
    """An enumeration matching the columns in the widget and the XMS C++ enumeration."""
    CHECK = 0
    COLOR = 1
    FONT_CHECK = 2
    FONT = 3
    DESCRIPTION = 4


class EnableModel(QSortFilterProxyModel):
    """A model to set enabled/disabled states."""
    def __init__(self, parent=None):
        """Initializes the filter model.

        Args:
            parent (Something derived from QObject): The parent object.
        """
        super().__init__(parent)
        self.enable_poly_labels = {}  # row idx, polygon label enabled flag

    def flags(self, index):
        """Get the flags for an item in the model.

        Args:
            index (QModelIndex): Item in the model to get flags for

        Returns:
            (Qt.ItemFlag): See description
        """
        ret_flags = super().flags(index)
        col = index.column()
        row = index.row()

        # First check if this is a polygon category row and if polygon labels have been enabled at the display list
        # level by the component.
        if col in [Columns.FONT_CHECK, Columns.FONT] and not self.enable_poly_labels.get(row):
            style_index = index.sibling(row, Columns.COLOR)
            options = style_index.data(ROLE_STYLE_OPTIONS)
            if isinstance(options, PolygonOptions):
                ret_flags &= ~Qt.ItemIsEnabled
                return ret_flags

        if col in [Columns.COLOR, Columns.FONT]:
            check_index = index.sibling(row, col - 1)
            check = check_index.data(Qt.EditRole)
            check_flags = check_index.flags()
            if check and check_flags & Qt.ItemIsEnabled:
                ret_flags |= Qt.ItemIsEnabled
            else:
                ret_flags &= ~Qt.ItemIsEnabled
        elif col == Columns.FONT_CHECK:
            check = index.sibling(row, Columns.CHECK).data(Qt.EditRole)
            if check:
                ret_flags |= Qt.ItemIsEnabled
            else:
                ret_flags &= ~Qt.ItemIsEnabled
        return ret_flags


class CategoryDisplayOptionsWidget(QWidget):
    """A widget to let the user set point display options."""
    class ListInfo:
        """Store the input category display list data that is not editable by the user and is not category specific."""
        def __init__(
            self, row_start, row_end, target_type, is_ids, display_uuid, comp_uuid, obs_reftime, obs_dset_uuid: str,
            enable_polygon_labels
        ):
            """Constructor.

            Args:
                row_start (int): Row index in the display options table of the first category in the list
                row_end (int): Row index + 1 in the display options table of the last category in the list.
                    1 beyond for iteration purposes
                target_type (TargetType): The list's target type enum value
                is_ids (bool): True if drawing on ids, False if drawing at specified locations
                display_uuid (str): UUID of the category display list
                comp_uuid (str): UUID of the component that owns the display list
                obs_reftime (datetime.datetime): The observation reference datetime
                obs_dset_uuid: Uuid of the dataset associated with observations.
                enable_polygon_labels (bool): True if enabling labels on polygons should even be an option for
                    the user. For historical reasons we want the default behavior to never show polygon labels unless
                    the component has explicitly enabled it.
            """
            self.row_start = row_start
            self.row_end = row_end
            self.target_type = target_type
            self.is_ids = is_ids
            self.display_uuid = display_uuid
            self.comp_uuid = comp_uuid
            self.obs_reftime = obs_reftime
            self.obs_dset_uuid = obs_dset_uuid
            self.enable_polygon_labels = enable_polygon_labels

    def __init__(self, parent=None):
        """Initializes the widget.

        Args:
            parent (Something derived from QWidget): The parent window.
        """
        super().__init__(parent)

        self.ui = Ui_CategoryDisplayOptionsWidget()
        self.ui.setupUi(self)
        self.model = QStandardItemModel(self)
        self.filter_model = EnableModel(self)
        self.list_starts = []
        self.change_all_color = False
        self.all_option_point = None
        self.all_option_line = None
        self.all_option_polygon = None
        self.delegate = DisplayOptionDelegate(self)
        self.check_delegate = CheckBoxNoTextDelegate(self)
        self.check_delegate.state_changed[QModelIndex].connect(self.state_changed)
        self.show_on_off = True  # Show the column of checkboxes and the 'All On', 'All Off' buttons

        self.filter_model.setSourceModel(self.model)
        self.ui.table_view_categories.setModel(self.filter_model)

        self.ui.table_view_categories.setItemDelegateForColumn(Columns.COLOR, self.delegate)
        self.ui.table_view_categories.setItemDelegateForColumn(Columns.FONT, self.delegate)

        self.ui.table_view_categories.setItemDelegateForColumn(Columns.CHECK, self.check_delegate)
        self.ui.table_view_categories.setItemDelegateForColumn(Columns.FONT_CHECK, self.check_delegate)

        header = CheckBoxHeader([Columns.CHECK, Columns.FONT_CHECK], Qt.Horizontal, self)
        header.center_checkboxes = True
        header.clicked.connect(self.on_section_checked)
        header.setStretchLastSection(True)
        self.ui.table_view_categories.setHorizontalHeader(header)
        self.ui.table_view_categories.verticalHeader().hide()

        self.ui.btn_change_all_label.set_font(QFont("Arial", 8))

        self.ui.btn_all_on.clicked.connect(self.all_on)
        self.ui.btn_all_off.clicked.connect(self.all_off)
        self.ui.btn_change_all_label.current_font_changed[QFont].connect(self.change_all_label)

        # Validator for the calibration target scale line edit
        validator = QxDoubleValidator(parent=self, bottom=0.0)
        self.ui.edt_target_scale.setValidator(validator)

        self.ui.btn_change_all_point.clicked.connect(self.change_all_point)
        self.ui.btn_change_all_line.clicked.connect(self.change_all_line)
        self.ui.btn_change_all_polygon.clicked.connect(self.change_all_polygon)
        self.ui.tog_show_targets.stateChanged.connect(self.on_tog_show_targets)

    def set_show_on_off(self, show_on_off):
        """Sets whether the checkboxes column and the 'All On', 'All Off' buttons are hidden.

        Args:
            show_on_off (bool): True to show.
        """
        self.show_on_off = show_on_off
        if not self.show_on_off:
            self.ui.table_view_categories.hideColumn(Columns.CHECK)
            self.ui.btn_all_on.setVisible(False)
            self.ui.btn_all_off.setVisible(False)
        else:
            self.ui.table_view_categories.showColumn(Columns.CHECK)
            self.ui.table_view_categories.setColumnWidth(Columns.CHECK, CHECK_WIDTH)
            self.ui.btn_all_on.setVisible(False)  # These buttons aren't used anymore. Checkbox header handles it.
            self.ui.btn_all_off.setVisible(False)

    def set_show_targets(self, show_targets):
        """Hide/show the calibration targets widgets.

        Args:
            show_targets (bool): True to show
        """
        self.ui.grp_targets.setVisible(show_targets)

    def set_category_lists(self, category_lists):
        """Initializes the widget.

        Args:
            category_lists (list[CategoryDisplayOptionList]): The categories and the target type of the list.
        """
        row = 0
        row_start = 0
        self.all_option_point = None
        self.all_option_line = None
        self.all_option_polygon = None

        for cat_list in category_lists:
            # These options are stored on each list, but assumed to be same for all lists in the dialog.
            self.ui.tog_show_targets.setChecked(cat_list.show_targets)
            self.ui.tog_logarithmic_scale.setChecked(cat_list.logarithmic_scale)
            self.ui.edt_target_scale.setText(str(cat_list.target_scale))

            if not self.all_option_point and cat_list.target_type in [
                TargetType.point, TargetType.ugrid_point, TargetType.ugrid_cell_center
            ]:
                self.all_option_point = PointOptions()
            elif not self.all_option_line and cat_list.target_type in [
                TargetType.arc, TargetType.ugrid_pointstring, TargetType.ugrid_cellstring
            ]:
                self.all_option_line = LineOptions()
            elif not self.all_option_polygon and cat_list.target_type in [
                TargetType.polygon, TargetType.ugrid_cellface_fill, TargetType.ugrid_cellface_fill_interior
            ]:
                self.all_option_polygon = PolygonOptions()
            for cat in cat_list.categories:
                self.set_item_to_row(cat, row, cat_list.enable_polygon_labels)
                row += 1
            list_info = self.ListInfo(
                row_start, row, cat_list.target_type, cat_list.is_ids, cat_list.uuid, cat_list.comp_uuid,
                cat_list.obs_reftime, cat_list.obs_dset_uuid, cat_list.enable_polygon_labels
            )
            self.list_starts.append(list_info)
            row_start = row

        self.model.setHeaderData(Columns.CHECK, Qt.Horizontal, "", Qt.DisplayRole)
        self.model.setHeaderData(Columns.COLOR, Qt.Horizontal, "Color/style", Qt.DisplayRole)
        self.model.setHeaderData(Columns.FONT_CHECK, Qt.Horizontal, "", Qt.DisplayRole)
        self.model.setHeaderData(Columns.FONT, Qt.Horizontal, "Label", Qt.DisplayRole)
        self.model.setHeaderData(Columns.DESCRIPTION, Qt.Horizontal, "Description", Qt.DisplayRole)

        if not self.show_on_off:
            self.ui.table_view_categories.hideColumn(Columns.CHECK)
        else:
            self.ui.table_view_categories.showColumn(Columns.CHECK)
            self.ui.table_view_categories.setColumnWidth(Columns.CHECK, CHECK_WIDTH)
        self.ui.table_view_categories.setColumnWidth(Columns.FONT_CHECK, CHECK_WIDTH)
        self.ui.table_view_categories.setColumnWidth(Columns.COLOR, BTN_WIDTH)
        self.ui.table_view_categories.setColumnWidth(Columns.FONT, BTN_WIDTH)

        self.model.dataChanged.connect(self.update_text_color)

        if self.all_option_point:
            icon_point = DisplayOptionIconFactory.get_icon(
                self.all_option_point,
                self.ui.btn_change_all_point.size().width()
            )
            self.ui.btn_change_all_point.setIcon(icon_point)
            self.ui.btn_change_all_polygon.show()
        else:
            self.ui.btn_change_all_point.hide()

        if self.all_option_line:
            icon_line = DisplayOptionIconFactory.get_icon(
                self.all_option_line,
                self.ui.btn_change_all_line.size().width()
            )
            self.ui.btn_change_all_line.setIcon(icon_line)
            self.ui.btn_change_all_polygon.show()
        else:
            self.ui.btn_change_all_line.hide()

        if self.all_option_polygon:
            icon_polygon = DisplayOptionIconFactory.get_icon(
                self.all_option_polygon,
                self.ui.btn_change_all_polygon.size().width()
            )
            self.ui.btn_change_all_polygon.setIcon(icon_polygon)
            self.ui.btn_change_all_polygon.show()
        else:
            self.ui.btn_change_all_polygon.hide()

        # Initialize state of calibration target widgets
        self.on_tog_show_targets(self.ui.tog_show_targets.checkState())
        self._update_header_from_data()

    def set_item_to_row(self, cat, row, enable_polygon_labels):
        """Set a new category row in the category list.

        Args:
            cat (CategoryDisplayOption): The new category to add
            row (int): The row to set the category
            enable_polygon_labels (bool): True if the label columns should be enabled for polygon categories
        """
        btn_size = QSize(BTN_WIDTH, 10)
        check_size = QSize(CHECK_WIDTH, 10)

        item_on = QStandardItem()
        item_option = QStandardItem()
        item_label_on = QStandardItem()
        item_label = QStandardItem()
        item_description = QStandardItem()
        item_on.setData(cat.on, Qt.EditRole)
        item_option.setData(cat.options, ROLE_STYLE_OPTIONS)
        item_label_on.setData(cat.label_on, Qt.EditRole)
        item_label.setData(cat.label_options, ROLE_LABEL_OPTIONS)
        item_label.setData(cat.use_description_for_label, ROLE_DESCRIPTION_LABELS)
        item_description.setData(cat.description, Qt.EditRole)
        item_description.setData(cat.file, ROLE_FILE)
        item_description.setData(cat.id, ROLE_ID)
        item_description.setData(cat.is_unassigned_category, ROLE_UNASSIGNED_CATEGORY)
        item_description.setEditable(False)
        item_option.setData(btn_size, Qt.SizeHintRole)
        item_label.setData(btn_size, Qt.SizeHintRole)
        item_on.setData(check_size, Qt.SizeHintRole)
        item_label_on.setData(check_size, Qt.SizeHintRole)
        item_label.setData(cat.options.color, Qt.DecorationRole)

        # If these are polygon options and polygon labels are not enabled for this category, disable the label columns.
        if isinstance(cat.options, PolygonOptions):
            self.filter_model.enable_poly_labels[row] = enable_polygon_labels
        else:  # Short circuit the check in the filter model if these are not polygon options.
            self.filter_model.enable_poly_labels[row] = True

        self.model.setItem(row, Columns.CHECK, item_on)
        self.model.setItem(row, Columns.COLOR, item_option)
        self.model.setItem(row, Columns.FONT_CHECK, item_label_on)
        self.model.setItem(row, Columns.FONT, item_label)
        self.model.setItem(row, Columns.DESCRIPTION, item_description)

    def get_category_lists(self):
        """Returns the current category display options list.

        Returns:
            cat_list (list[CategoryDisplayOptionList]): The current category display options list.
        """
        cat_lists = []
        for list_info in self.list_starts:
            cat_list = CategoryDisplayOptionList()
            cat_list.target_type = list_info.target_type
            cat_list.is_ids = list_info.is_ids
            cat_list.uuid = list_info.display_uuid
            cat_list.comp_uuid = list_info.comp_uuid
            cat_list.show_targets = self.ui.tog_show_targets.isChecked()
            cat_list.logarithmic_scale = self.ui.tog_logarithmic_scale.isChecked()
            cat_list.target_scale = float(self.ui.edt_target_scale.text())
            cat_list.obs_reftime = list_info.obs_reftime
            cat_list.obs_dset_uuid = list_info.obs_dset_uuid
            cat_list.enable_polygon_labels = list_info.enable_polygon_labels
            for row in range(list_info.row_start, list_info.row_end):
                cat_row = CategoryDisplayOption()
                cat_row.on = self.model.data(self.model.index(row, Columns.CHECK), Qt.EditRole)
                cat_row.options = self.model.data(self.model.index(row, Columns.COLOR), ROLE_STYLE_OPTIONS)
                cat_row.label_on = self.model.data(self.model.index(row, Columns.FONT_CHECK), Qt.EditRole)
                cat_row.label_options = self.model.data(self.model.index(row, Columns.FONT), ROLE_LABEL_OPTIONS)
                cat_row.use_description_for_label = self.model.data(
                    self.model.index(row, Columns.FONT), ROLE_DESCRIPTION_LABELS
                )
                cat_row.description = self.model.data(self.model.index(row, Columns.DESCRIPTION), Qt.EditRole)
                cat_row.file = self.model.data(self.model.index(row, Columns.DESCRIPTION), ROLE_FILE)
                cat_row.id = self.model.data(self.model.index(row, Columns.DESCRIPTION), ROLE_ID)
                cat_row.is_unassigned_category = self.model.data(
                    self.model.index(row, Columns.DESCRIPTION), ROLE_UNASSIGNED_CATEGORY
                )
                cat_list.categories.append(cat_row)
            cat_lists.append(cat_list)
        return cat_lists

    def _update_header_from_data(self) -> None:
        """Update the state of the checkboxes in the header based on the model data."""
        header = self.ui.table_view_categories.horizontalHeader()
        for col_idx in [Columns.CHECK, Columns.FONT_CHECK]:
            checked = False
            unchecked = False
            for row_idx in range(self.model.rowCount()):
                state = self.model.data(self.model.index(row_idx, col_idx), role=Qt.EditRole)
                if state:
                    checked = True
                else:
                    unchecked = True

            if checked and unchecked:
                header.update_check_state(col_idx, Qt.PartiallyChecked)
            elif checked:
                header.update_check_state(col_idx, Qt.Checked)
            else:
                header.update_check_state(col_idx, Qt.Unchecked)

    @Slot()
    def all_on(self):
        """Called when the color, size, or symbol changes."""
        for row in range(self.model.rowCount()):
            self.model.setData(self.model.index(row, Columns.CHECK), True, Qt.EditRole)
            self.model.setData(self.model.index(row, Columns.FONT_CHECK), True, Qt.EditRole)
        self.ui.table_view_categories.setFocus()

    @Slot()
    def all_off(self):
        """Called when the color, size, or symbol changes."""
        for row in range(self.model.rowCount()):
            self.model.setData(self.model.index(row, Columns.CHECK), False, Qt.EditRole)
            self.model.setData(self.model.index(row, Columns.FONT_CHECK), False, Qt.EditRole)
        self.ui.table_view_categories.setFocus()

    @Slot()
    def change_all_point(self):
        """Called when the color, size, or symbol changes."""
        dlg = PointDisplayOptionsDialog(self.all_option_point, self, self.change_all_color)

        if dlg and dlg.exec():
            self.all_option_point = dlg.get_options()
            for row in range(self.model.rowCount()):
                old_options = self.model.data(self.model.index(row, Columns.COLOR), Qt.UserRole)
                if isinstance(old_options, PointOptions):
                    old_options.size = self.all_option_point.size
                    old_options.symbol = self.all_option_point.symbol
                    if self.change_all_color:
                        old_options.color = self.all_option_point.color
                    self.model.setData(self.model.index(row, Columns.COLOR), old_options, Qt.UserRole)

    @Slot()
    def change_all_line(self):
        """Called when the color, size, or symbol changes."""
        dlg = LineDisplayOptionsDialog(self.all_option_line, self, self.change_all_color)

        if dlg and dlg.exec():
            self.all_option_line = dlg.get_options()
            for row in range(self.model.rowCount()):
                line_data = self.model.data(self.model.index(row, Columns.COLOR), Qt.UserRole)
                if isinstance(line_data, LineOptions):
                    line_data.style = self.all_option_line.style
                    line_data.width = self.all_option_line.width
                    if self.change_all_color:
                        line_data.color = self.all_option_line.color
                    self.model.setData(self.model.index(row, Columns.COLOR), line_data, Qt.UserRole)

    @Slot()
    def change_all_polygon(self):
        """Called when the color, size, or symbol changes."""
        dlg = PolygonDisplayOptionsDialog(self.all_option_polygon, self, self.change_all_color)

        if dlg and dlg.exec():
            self.all_option_polygon = dlg.get_options()
            for row in range(self.model.rowCount()):
                poly_data = self.model.data(self.model.index(row, Columns.COLOR), Qt.UserRole)
                if isinstance(poly_data, PolygonOptions):
                    poly_data.texture = self.all_option_polygon.texture
                    if self.change_all_color:
                        poly_data.color = self.all_option_polygon.color
                    self.model.setData(self.model.index(row, Columns.COLOR), poly_data, Qt.UserRole)

    @Slot()
    def change_all_label(self, new_font):
        """Called when the color, size, or font changes."""
        for row in range(self.model.rowCount()):
            self.model.setData(self.model.index(row, Columns.FONT), new_font, Qt.UserRole)

    @Slot()
    def state_changed(self, index):
        """Called enabled state should change."""
        column = index.column()
        self.ui.table_view_categories.update(index.model().index(index.row(), column + 1))
        if column == 0:
            self.ui.table_view_categories.update(index.model().index(index.row(), Columns.FONT_CHECK))
            self.ui.table_view_categories.update(index.model().index(index.row(), Columns.FONT))
        self._update_header_from_data()

    @Slot()
    def on_tog_show_targets(self, state):
        """Called when the show calibration targets toggle state changes.

        Args:
            state (Qt.CheckState): Current toggle state
        """
        self.ui.tog_logarithmic_scale.setEnabled(state == Qt.Checked)
        self.ui.edt_target_scale.setEnabled(state == Qt.Checked)

    @Slot()
    def update_text_color(self, top_left_index, bottom_right_index, roles):
        """Called when something might have changed. If the color changed, update the font color."""
        if Qt.UserRole not in roles:
            return
        # skip if the color/style columnn did not change
        if top_left_index.column() >= Columns.FONT_CHECK:
            return
        if bottom_right_index.column() == Columns.CHECK:
            return
        top_row = top_left_index.row()
        bottom_row = bottom_right_index.row()
        for i in range(top_row, bottom_row + 1):
            display_options = self.model.index(i, Columns.COLOR).data(Qt.UserRole)
            self.model.setData(self.model.index(i, Columns.FONT), display_options.color, Qt.DecorationRole)

    @Slot()
    def on_section_checked(self, idx, checked):
        """Called when a column checkbox is clicked in the header view."""
        if idx not in [Columns.CHECK, Columns.FONT_CHECK]:
            return
        for row in range(self.model.rowCount()):
            self.model.setData(self.model.index(row, idx), checked, Qt.EditRole)
        self.filter_model.invalidate()
