"""HydrographCoverageComponent class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.gmi.data.generic_model import Group, Section
from xms.guipy.data.target_type import TargetType

# 4. Local modules
from xms.hgs.components.hgs_coverage_component_base import HgsCoverageComponentBase
from xms.hgs.data import hydrograph_generic_model
from xms.hgs.mapping.hydrograph_map_att import HydrographMapAtt


class HydrographCoverageComponent(HgsCoverageComponentBase):
    """A component for hydrographs."""
    def __init__(self, main_file: str) -> None:
        """Initializes the class.

        Args:
            main_file: The main file associated with this component.
        """
        super().__init__(main_file)
        self.polygon_commands = []  # We don't have any polygon properties

    def _section(self, target_type: TargetType) -> Section:
        """
        Get a GMI Section from the coverage's generic model based on its `TargetType`.

        Args:
            target_type: `TargetType` for the desired section.

        Returns:
            Section: The section for the feature type.
        """
        return hydrograph_generic_model.create().section_from_target_type(target_type)

    def make_map_att(self, feature, att_type: str, group: Group):
        """Returns an MapAtt.

        Args:
            feature: A point, arc, or polygon.
            att_type (str): Attribute type (e.g. 'Flux nodal')
            group (Group): The BC as a generic model group.

        Returns:
            (Type[MapAtt]): The att item.
        """
        return HydrographMapAtt(feature, att_type, group)
