"""SimControlDialog class."""

__copyright__ = '(C) Copyright Aquaveo 2024'
__license__ = 'All rights reserved'

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
import pandas as pd
from PySide2.QtCore import QDateTime, Qt
from PySide2.QtWidgets import QDialog, QFileDialog

# 3. Aquaveo modules
from xms.api._xmsapi.dmi import DatasetItem
from xms.api.dmi import Query
from xms.api.tree import tree_util
from xms.guipy import file_io_util
from xms.guipy.dialogs import date_time_dialog
from xms.guipy.dialogs.treeitem_selector import TreeItemSelectorDlg
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg
from xms.guipy.time_format import string_to_datetime
from xms.tool_core.table_definition import FloatColumnType, TableDefinition

# 4. Local modules
from xms.hgs.data.domains import Domains
from xms.hgs.gui import gui_util
from xms.hgs.gui.gui_exchange import GuiExchange
from xms.hgs.gui.sim_control_dialog_ui import Ui_simulation_control_dialog
from xms.hgs.misc import util


def _keep_node_condition(node):
    """Method used by util.filter_tree to filter the tree.

    Args:
        node (TreeNode): A node.

    Returns:
        (bool): True if the node should be kept.
    """
    return node.item_typename == 'TI_SCALAR_DSET' and node.data_location == 'CELL'


def _tooltip_filepath() -> Path:
    """Returns the path to the tooltip file."""
    return Path(__file__).parent / 'sim_control_dialog_tooltips.json'


class SimControlDialog(XmsDlg):
    """SimControlDialog class."""
    def __init__(self, data, query: Query, parent=None):
        """Initializes the class.

        Args:
            data: Data dict of all sim control data.
            query (xmsapi.dmi.Query): Object for communicating with GMS.
            parent: Parent widget.
        """
        super().__init__(parent, 'xms.hgs.gui.sim_control_dialog')
        self.data = None  # Not private because it's accessed by the component
        self._query = query
        self._tooltips: dict[str, str]
        self.ui = Ui_simulation_control_dialog()
        self.ui.setupUi(self)
        self._gx = GuiExchange(self.ui, self.data, tooltip_filepath=_tooltip_filepath())
        self._gx.enable_whats_this(self)
        self._gx.read_tooltips()
        self._help_getter = gui_util.help_getter('xms.hgs.gui.sim_control_dialog')

        self._setup_signals()
        self._setup_open_and_save_buttons()
        self._setup_with_sim_data(data)

    def _setup_with_sim_data(self, data):
        """Sets up the dialog given the simulation data.

        Args:
            data: Data dict of simulation data.
        """
        self.data = data  # Not private because it's accessed by the component
        self._gx.set_data(data)
        self._exchange_data(set_data=False)

        # Call these to enable things
        self._on_chk_surface_flow()
        self._on_chk_et()

    def _exchange_data(self, set_data: bool) -> None:
        """Either sets the data dict from the controls (set_data==True) or vice-versa (set_data==False).

        Args:
            set_data (bool): See description.
        """
        # General
        self._gx.do_plain_txt(set_data, 'description')
        self._gx.do_chk(set_data, 'transient flow', True)
        self._gx.do_chk(set_data, 'unsaturated', True)
        self._gx.do_chk(set_data, 'finite difference mode')
        self._gx.do_chk(set_data, 'control volume')

        # General - Domains
        self._gx.do_chk(set_data, 'porous media', True)
        self.ui.chk_porous_media.setEnabled(False)  # Porous media can't be turned off
        self._gx.do_chk(set_data, 'surface flow')
        self._gx.do_chk(set_data, 'dual nodes for surface flow', True)
        self.ui.chk_dual_nodes_for_surface_flow.setEnabled(False)  # Dual nodes... can't be turned off
        self._gx.do_chk(set_data, 'et')

        # Units
        self._gx.do_cbx(set_data, 'units')
        self._gx.do_chk_edt(set_data, 'gravitational acceleration', '', 9.80665)

        # Saturated flow
        self._gx.do_chk(set_data, 'no fluid mass balance')
        self._gx.do_chk_spn(set_data, 'flow solver maximum iterations', 2000)
        self._gx.do_chk_edt(set_data, 'flow solver convergence criteria', '', 1.0e-10)
        self._gx.do_chk_cbx(set_data, 'flow solver detail')

        # Variably saturated flow
        self._gx.do_chk_edt(set_data, 'upstream weighting factor', '', 1.0)
        self._gx.do_chk(set_data, 'remove negative coefficients', True)
        self._gx.do_chk(set_data, 'no nodal flow check', True)
        self._gx.do_chk_edt(set_data, 'nodal flow check tolerance', '', 1.0e-2)
        self._gx.do_chk_edt(set_data, 'underrelaxation factor', '', 1.0)

        # Timestepping
        self._gx.do_chk_edt(set_data, 'initial time', '', 0.0)
        self._gx.do_chk_edt(set_data, 'initial timestep', '', 0.01)
        self._gx.do_chk_edt(set_data, 'maximum timestep', '', 1.0e25)
        self._gx.do_chk_edt(set_data, 'minimum timestep', '', 1.0e-10)
        self._gx.do_chk_btn(set_data, 'time varying maximum timestep', '')
        self._gx.do_chk_btn(set_data, 'target times', '')
        self._gx.do_chk_edt(set_data, 'minimum timestep multiplier', '', 0.5)
        self._gx.do_chk_edt(set_data, 'maximum timestep multiplier', '', 2.0)
        self._gx.do_chk_edt(set_data, 'jacobian epsilon', '', 1.0e-4)
        self._gx.do_chk_edt(set_data, 'minimum relaxation factor for convergence', '', 0.95)

        self._gx.do_chk_spn(set_data, 'newton maximum iterations', 15)
        self._gx.do_chk_spn(set_data, 'newton minimum iterations', 0)
        self._gx.do_chk_edt(set_data, 'newton absolute convergence criteria', '', 1.0e-5)
        self._gx.do_chk_edt(set_data, 'newton residual convergence criteria', '', 1.0e-8)
        self._gx.do_chk_edt(set_data, 'newton maximum update for head', '', 1.0)
        self._gx.do_chk_edt(set_data, 'newton maximum update for depth', '', 1.0e-2)
        self._gx.do_chk_edt(set_data, 'newton absolute maximum residual', '', 0.0)
        self._gx.do_chk_edt(set_data, 'newton maximum residual increase', '', 1.0e30)

        self._gx.do_chk_edt(set_data, 'head control', '', 1.0)
        self._gx.do_chk_edt(set_data, 'water depth control', '', 1.0)
        self._gx.do_chk_edt(set_data, 'saturation control', '', 0.1)
        self._gx.do_chk_spn(set_data, 'newton iteration control', 8)

        # Output
        self._do_output_times_table(set_data)
        self._gx.do_chk_edt(set_data, 'start date time', '', '')
        self._gx.do_chk_btn(set_data, 'start date time', '')

        # Zones
        self._gx.do_chk_edt(set_data, 'read porous media zones from file', 'porous media zones dataset', '')
        self._gx.do_chk_btn(set_data, 'read porous media zones from file', 'select porous media zones dataset')
        self._gx.do_chk_edt(set_data, 'read surface flow zones from file', 'surface flow zones dataset', '')
        self._gx.do_chk_btn(set_data, 'read surface flow zones from file', 'select surface flow zones dataset')
        self._gx.do_chk_edt(set_data, 'read et zones from file', 'et zones dataset', '')
        self._gx.do_chk_btn(set_data, 'read et zones from file', 'select et zones dataset')

        # Initial conditions
        self._gx.do_rbt_edt(set_data, 'initial head', True, 'initial head', 0.0)
        self._gx.do_rbt_edt(
            set_data, 'initial head depth to water table', False, 'initial head depth to water table', 0.0
        )
        self._gx.do_rbt(set_data, 'initial head surface elevation')
        self._gx.do_rbt_edt(set_data, 'initial head from output file', False, 'initial head file', '')
        self._gx.do_rbt_btn(set_data, 'initial head from output file', 'select initial head file')
        self._gx.do_rbt_edt(set_data, 'initial water depth', True, 'initial water depth', 0.0001)
        self._gx.do_rbt_edt(set_data, 'initial water depth from file', False, 'initial water depth file', '')
        self._gx.do_rbt_btn(set_data, 'initial water depth from file', 'select initial water depth file')

    def _setup_signals(self):
        """Sets up the signals."""
        self.ui.chk_surface_flow.toggled.connect(self._on_chk_surface_flow)
        self.ui.chk_et.toggled.connect(self._on_chk_et)
        self.ui.btn_time_varying_maximum_timestep.clicked.connect(self._on_btn_time_varying_maximum_timestep)
        self.ui.btn_target_times.clicked.connect(self._on_btn_target_times)
        self.ui.btn_start_date_time.clicked.connect(self._on_btn_start_date_time)
        self.ui.btn_select_porous_media_zones_dataset.clicked.connect(self._on_btn_select_porous_media_zones_dataset)
        self.ui.btn_select_surface_flow_zones_dataset.clicked.connect(self._on_btn_select_surface_flow_zones_dataset)
        self.ui.btn_select_et_zones_dataset.clicked.connect(self._on_btn_select_et_zones_dataset)
        self.ui.btn_select_initial_head_file.clicked.connect(self._on_btn_initial_head_from_file)
        self.ui.btn_select_initial_water_depth_file.clicked.connect(self._on_btn_initial_water_depth_from_file)
        self.ui.btn_open.clicked.connect(self._on_btn_open)
        self.ui.btn_save.clicked.connect(self._on_btn_save)
        self.ui.buttonBox.helpRequested.connect(self.help_requested)

    def _setup_open_and_save_buttons(self):
        """Sets up the open button."""
        debug_file_exists = Path('C:/temp/debug_xmshgs_sim_control_dialog.dbg').is_file()
        self.ui.btn_open.setVisible(debug_file_exists)
        self.ui.btn_save.setVisible(debug_file_exists)
        self.ui.btn_save.setEnabled(False)

    def _on_btn_open(self):
        """Lets us open a sim_comp.json file."""
        rv = QFileDialog.getOpenFileName(self, 'Sim Control json file', '', '')
        if rv and rv[0]:
            self._opened_file = Path(rv[0])
            sim_data = file_io_util.read_json_file(self._opened_file)
            self._setup_with_sim_data(sim_data)
            self.ui.btn_save.setEnabled(True)

    def _on_btn_save(self):
        """Saves to the opened materials.json file."""
        self._exchange_data(set_data=True)
        file_io_util.write_json_file(self.data, self._opened_file)

    def _on_chk_surface_flow(self):
        """Called when the Surface flow domain checkbox is clicked."""
        self.ui.wid_surface_flow_zones.setEnabled(self.ui.chk_surface_flow.isChecked())
        self.ui.grp_water_depth.setEnabled(self.ui.chk_surface_flow.isChecked())

    def _on_chk_et(self):
        """Called when the ET domain checkbox is clicked."""
        self.ui.wid_et_zones.setEnabled(self.ui.chk_et.isChecked())

    def _run_table_dialog(self, title: str, table_definition: TableDefinition, button: str) -> None:
        """Runs the TableDialog.

        Args:
            title (str): Window title.
            table_definition (TableDefinition): Defines the table (column types, fixed row count or not).
            button (str): Name of the button widget.
        """
        json_str = self.data.get(button, '')
        if json_str:
            if isinstance(json_str, str):
                if 'schema' not in json_str:
                    json_str = pd.read_json(json_str).to_json(orient='table')  # Switch string to new format
            else:
                json_str = table_definition.to_pandas('', rows=json_str).to_json(orient='table')
        json_str = gui_util.run_table_dialog(title, table_definition, json_str, self, button)
        if json_str is not None:
            self._gx.get_data()[button] = json_str

    def _on_btn_time_varying_maximum_timestep(self) -> None:
        """Called when the button is clicked."""
        table_def = TableDefinition(
            [
                FloatColumnType(header='Time', tool_tip='Time [T]', default=0.0),
                FloatColumnType(header='Max Timestep', tool_tip='Maximum timestep size [T]', default=0.0),
            ],
        )
        self._run_table_dialog('Time varying maximum timestep', table_def, 'btn_time_varying_maximum_timestep')

    def _on_btn_target_times(self) -> None:
        """Called when the target times button is clicked."""
        table_def = TableDefinition([
            FloatColumnType(header='Target Times', tool_tip='Target Times', default=0.0),
        ], )
        self._run_table_dialog('Target Times', table_def, 'btn_target_times')

    def _on_btn_start_date_time(self) -> None:
        """Opens the date/time dialog."""
        date = None
        time = None
        date_time_str = self.ui.edt_start_date_time.text()
        if date_time_str:
            date_time = string_to_datetime(date_time_str, None)
            if date_time:
                q_date_time = QDateTime.fromString(date_time.isoformat(), Qt.ISODate) if date_time else None
                if q_date_time.isValid():
                    date = q_date_time.date()
                    time = q_date_time.time()
        rv, date, time = date_time_dialog.run(date, time)
        if rv:
            date_time = QDateTime(date, time)
            self.ui.edt_start_date_time.setText(date_time.toString(Qt.ISODate))

    def _on_btn_select_zones_dataset(self, domain) -> None:
        # Open tree picker to let user pick the dataset
        previous_selection = self._previous_selected_zones_dataset_uuid(domain)
        filtered_tree = util.filter_tree(self._query.copy_project_tree(), _keep_node_condition)
        dialog = TreeItemSelectorDlg(
            title='Select Dataset',
            target_type=DatasetItem,
            pe_tree=filtered_tree,
            previous_selection=previous_selection,
            override_icon=gui_util.override_icon,
            parent=self,
            always_show_check_boxes=False
        )
        if dialog.exec() == QDialog.Accepted:
            dataset_uuid = dialog.get_selected_item_uuid()
            dataset_path = tree_util.build_tree_path(self._query.project_tree, dataset_uuid)
            edit_field = self._zones_edt_from_domain(domain)
            edit_field.setText(dataset_path)

    def _on_btn_select_porous_media_zones_dataset(self) -> None:
        self._on_btn_select_zones_dataset(Domains.PM)

    def _on_btn_select_surface_flow_zones_dataset(self) -> None:
        self._on_btn_select_zones_dataset(Domains.OLF)

    def _on_btn_select_et_zones_dataset(self) -> None:
        self._on_btn_select_zones_dataset(Domains.ET)

    def _on_btn_select_file(self, command, edt) -> None:
        filename = self._gx.get_data().get(command, '')
        rv = QFileDialog.getOpenFileName(self, 'File', filename, '')
        self._gx.get_data()[f'edt_{command}'] = rv[0]
        edt.setText(rv[0])

    def _on_btn_initial_head_from_file(self) -> None:
        self._on_btn_select_file('initial_head_file', self.ui.edt_initial_head_file)

    def _on_btn_initial_water_depth_from_file(self) -> None:
        self._on_btn_select_file('initial_water_depth_file', self.ui.edt_initial_water_depth_file)

    def _zones_edt_from_domain(self, domain):
        """Returns the zones dataset edit field that goes with the domain."""
        return {
            Domains.PM: self.ui.edt_porous_media_zones_dataset,
            Domains.OLF: self.ui.edt_surface_flow_zones_dataset,
            Domains.ET: self.ui.edt_et_zones_dataset
        }[domain]

    def _previous_selected_zones_dataset_uuid(self, domain) -> str:
        """If we have the data for a zones dataset, return the uuid string from it."""
        previous_selection = ''
        edit_field = self._zones_edt_from_domain(domain)
        dataset_path = edit_field.text()
        if dataset_path:
            item = tree_util.item_from_path(self._query.project_tree, dataset_path)
            if item:
                previous_selection = item.uuid
        return previous_selection

    def _do_output_times_table(self, set_data: bool) -> None:
        """Exchanges data between the gui widgets and the data dict."""
        if set_data:
            widget, widget_name = self._gx.get_widget_and_widget_name('tbl', 'output times')
            df = widget.get_values()
            df.sort_values(by=['Output Times'], inplace=True, ignore_index=True)  # Sort the values
            df.index += 1  # Sorting reset the index so we make it 1-based again
            self.data[widget_name] = df.to_json(orient='table')
        else:
            table_definition = TableDefinition(
                [FloatColumnType(header='Output Times', tool_tip='Output Times', default=0.0)]
            )
            json_str = self.data.get('tbl_output_times', '')
            if json_str and 'schema' not in json_str:
                json_str = pd.read_json(json_str).to_json(orient='table')  # Switch string to new format
            df = table_definition.to_pandas(json_str)
            self.ui.tbl_output_times.setup(table_definition, df)
            self._gx.add_tooltip(self.ui.tbl_output_times, 'output times')

    def accept(self) -> None:
        """Called on OK."""
        self._exchange_data(set_data=True)
        super().accept()

    def exec(self):
        """If testing, just accept immediately."""
        from xms.api.dmi import XmsEnvironment
        if XmsEnvironment.xms_environ_running_tests() == 'TRUE':  # If testing, just accept immediately.
            self.accept()
            return QDialog.Accepted
        else:
            return super().exec_()  # pragma no cover - can't hit this line if testing
