"""FileImporter class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from pathlib import Path

# 2. Third party modules
from PySide2.QtWidgets import QWidget

# 3. Aquaveo modules
from xms.guipy.dialogs import dialog_util, process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.testing.type_aliases import Pathlike

# 4. Local modules
from xms.mf6.data.base_file_data import BaseFileData
from xms.mf6.file_io import io_factory
from xms.mf6.file_io.writer_options import WriterOptions


def export_file(data: BaseFileData, filepath: Pathlike, win_cont: QWidget) -> bool:
    """Export a file and return True if there were no errors.

    Args:
        data: The data.
        filepath: Where to export the data.
        win_cont: The window container.

    Returns:
        See description.
    """
    thread = FileExportFeedbackThread(data, filepath)
    return process_feedback_dlg.run_feedback_dialog(thread, win_cont) and thread.rv


class FileExportFeedbackThread(FeedbackThread):
    """Thread for exporting a file."""
    def __init__(self, data: BaseFileData, filepath: Pathlike):
        """Initializes the class.

        Args:
            data: The data.
            filepath: Where to export the data.
        """
        super().__init__(is_export=False, create_query=False)

        self._data = data
        self._filepath = filepath
        self.rv = True
        self.display_text |= {
            'title': 'Export MODFLOW 6 file',
            'working_prompt': f'Exporting MODFLOW 6 file \"{self._filepath}\".',
            'error_prompt': 'Error(s) encountered while exporting.',
            'warning_prompt': 'Warning(s) encountered while exporting.',
            'success_prompt': f'Successfully exported \"{self._filepath}\".',
        }

    def _run(self) -> None:
        """Does the work."""
        orig_filename = self._data.filename
        try:
            with dialog_util.wait_cursor_context():
                self._data.filename = self._filepath
                mfsim_dir = Path(self._filepath).parent
                writer_options = WriterOptions(mfsim_dir=mfsim_dir, use_open_close=False, use_periods_db=True)
                writer = io_factory.writer_from_ftype(self._data.ftype, writer_options)
                writer.write(self._data)
        except Exception:
            self.rv = False
        self._data.filename = orig_filename
