"""IdomainMapperRunner class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules

# 3. Aquaveo modules
from xms.guipy.dialogs import process_feedback_dlg
from xms.guipy.dialogs.feedback_thread import FeedbackThread
from xms.guipy.dialogs.xms_parent_dlg import XmsDlg

# 4. Local modules
from xms.mf6.gui import gui_util
from xms.mf6.misc import log_util
from xms.mf6.misc.settings import Settings


class IdomainMapperFeedbackThread(FeedbackThread):
    """Thread for mapping a coverage to the IDOMAIN array."""
    def __init__(self, package_dlg, cov_uuids: list[str]):
        """Initializes the class.

        Args:
            package_dlg: The package dialog calling this function.
            cov_uuids: Coverage uuids.
        """
        super().__init__(is_export=False, create_query=False)
        self._package_dlg = package_dlg
        self._cov_uuids = cov_uuids

        self._query = self._package_dlg.dlg_input.query
        self._coverages = []
        self._coverage_att_files = {}
        self._dis_pack = None
        self._cogrid = None
        self._ugrid_uuid = ''
        self.display_text |= {
            'title': 'MODFLOW 6 Cell Activator',
            'working_prompt': 'Running cell activator...',
            'error_prompt': 'Error(s) encountered in the cell activator.',
            'warning_prompt': 'Warning(s) encountered in the cell activator.',
            'success_prompt': 'No warnings or errors encountered in the cell activator.',
        }

    def _run(self) -> None:
        """Does the work."""
        mapper = self._setup_mapper()
        mapper.do_work()

    def _setup_mapper(self):
        """Sets up the IdomainMapper class."""
        self._get_xms_data()
        from xms.mf6.mapping.idomain_mapper import IdomainMapper
        return IdomainMapper(
            dis_package=self._dis_pack,
            cogrid=self._cogrid,
            coverages=self._coverages,
            coverage_att_files=self._coverage_att_files
        )

    def _get_xms_data(self):
        """Get data from XMS needed to do this operation."""
        for uuid in self._cov_uuids:
            self._coverages.append(self._query.item_with_uuid(uuid))
            self._coverage_att_files[uuid] = self._query.coverage_attributes(
                cov_uuid=uuid, points=False, arcs=True, polygons=True
            )

        # Get ugrid uuid
        self._dis_pack = self._package_dlg.dlg_input.data
        model = self._dis_pack.model
        self._cogrid = model.get_cogrid()
        self._ugrid_uuid = self._cogrid.uuid


class IdomainMapperRunner:
    """Maps idomain from coverages."""
    def __init__(self, package_dlg: XmsDlg):
        """Initializes the class.

        Args:
            package_dlg: The package dialog calling this function.
        """
        self._package_dlg = package_dlg
        self.query = package_dlg.dlg_input.query
        self.win_cont = package_dlg
        self._error = False
        self._mapper = None
        self._dis_pack = None
        self._ugrid_uuid = ''
        self._cogrid = None
        self._cov_uuids = []
        self._coverages = []
        self._coverage_att_files = {}
        self._testing = False
        self._log = log_util.get_logger()

    def run_with_feedback_dialog(self) -> bool:
        """Runs the model checker using the ProcessFeedbackDialog."""
        self._cov_uuids = self._get_coverage_uuids()
        if not self._cov_uuids:
            return False
        thread = IdomainMapperFeedbackThread(self._package_dlg, self._cov_uuids)
        return process_feedback_dlg.run_feedback_dialog(thread, self._package_dlg)

    def _get_coverage_uuids(self) -> list[str]:
        """Prompts user to select one or more coverages.

        Returns:
            (list of str): List of uuid strings identifying the coverages.
        """
        last_selected = Settings.get(self._package_dlg.dlg_input.data.filename, 'LAST_ACTIVATE_COVERAGE')
        uuids = gui_util.select_coverages_dialog(self._package_dlg, self.query.project_tree, last_selected)
        if uuids:
            Settings.set(self._package_dlg.dlg_input.data.filename, 'LAST_ACTIVATE_COVERAGE', uuids[0])
        return uuids
