"""SimComponent class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
import os
from pathlib import Path
import subprocess

# 2. Third party modules
from PySide2.QtCore import QDir, Qt
from PySide2.QtWidgets import QApplication, QDialog, QWidget
from typing_extensions import override

# 3. Aquaveo modules
from xms.api.dmi import ActionRequest, Query, XmsEnvironment as XmEnv
from xms.api.tree import tree_util, TreeNode
from xms.guipy import settings
from xms.guipy.dialogs import message_box

# 4. Local modules
from xms.mf6.check.model_checker import ModelChecker
from xms.mf6.components import duplication, mf6_exporter, new_sim_dialog_runner
from xms.mf6.components.package_component_base import PackageComponentBase
from xms.mf6.data.mfsim_data import MfsimData
from xms.mf6.file_io import io_factory, io_util
from xms.mf6.file_io.writer_options import WriterOptions
from xms.mf6.gui import gui_util
from xms.mf6.gui.dialog_input import DialogInput
from xms.mf6.gui.export_simulation_dialog import ExportSimulationDialog
from xms.mf6.gui.sim_dialog import SimDialog
from xms.mf6.simulation_runner import sim_runner


def _get_last_item_to_be_duplicated(query: Query) -> TreeNode:
    """Returns the last component that will be duplicated.

    Assumes that duplication is done in the order that the project tree is in.

    Args:
        query (xmsapi.dmi.Query): Object for communicating with GMS

    Returns:
        (TreeNode): See description.
    """
    last = None
    uuid = query.parent_item_uuid()
    sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, uuid)
    descendants = tree_util.descendants_of_type(sim_node, xms_types=['TI_COMPONENT'])
    for descendant in descendants:
        ancestor = tree_util.ancestor_of_type(descendant, xms_types=['TI_SOLUTION_FOLDER'])
        if not ancestor:
            last = descendant
    return last


def _duplicate_sim(query: Query, old_main_file: str, new_main_file: str) -> None:
    """Replace UUIDs in all the child models and packages after duplicating a simulation.

    Args:
        query (xmsapi.dmi.Query): Object for communicating with GMS
        old_main_file (str): Main file of the simulation being duplicated
        new_main_file (str): Main file of the new duplicated simulation
    """
    def _find_new_sim(tree_node, main_file):
        right_type = tree_node and tree_node.item_typename == 'TI_DYN_SIM'
        if right_type and os.path.normpath(tree_node.main_file) == os.path.normpath(main_file):
            return tree_node
        for child in tree_node.children:
            sim = _find_new_sim(child, main_file)
            if sim:
                return sim
        return None

    def _fill_old_to_new(old_node, new_node, fill_map):
        if not old_node or not new_node:
            return
        if old_node.item_typename == 'TI_COMPONENT':
            fill_map[old_node.main_file] = new_node.main_file
        for old_child, new_child in zip(old_node.children, new_node.children):
            _fill_old_to_new(old_child, new_child, fill_map)

    old_to_new = {old_main_file: new_main_file}
    old_sim = tree_util.find_tree_node_by_uuid(query.project_tree, query.parent_item_uuid())
    new_sim = _find_new_sim(query.project_tree, new_main_file)
    _fill_old_to_new(old_sim, new_sim, old_to_new)
    duplication.replace_old_uuids(old_to_new)


def saved_sim(sim_name: str):
    """Returns the location of the saved simulation.

    Args:
        sim_name: Name of the simulation.

    Returns:
        (str): See description.
    """
    project_file = XmEnv.xms_environ_project_path()
    project_dir = os.path.dirname(project_file)
    project_base = os.path.splitext(os.path.basename(project_file))[0]
    mfsim_nam = os.path.join(project_dir, f'{project_base}_models', 'MODFLOW 6', sim_name, 'mfsim.nam')
    return mfsim_nam


class SimComponent(PackageComponentBase):
    """A Dynamic Model Interface (DMI) component for a MODFLOW 6 simulation."""
    def __init__(self, main_file):
        """Initializes the class.

        Args:
            main_file (str): The main file associated with this component.
        """
        super().__init__(main_file)

        self.main_file = main_file.strip('"\'')
        self.ftype = 'MFSIM6'  # (Not an official MODFLOW ftype)
        self.show_check_sim_dialog = True

    def save_to_location(self, new_path, save_type):
        """Save component files to a new location.

        Args:
            new_path (str): Path to the new save location.
            save_type (str): One of DUPLICATE, PACKAGE, SAVE, SAVE_AS, LOCK.
                DUPLICATE happens when the tree item owner is duplicated. The new component will always be unlocked to
                start with.

                PACKAGE happens when the project is being saved as a package. As such, all data must be copied and all
                data must use relative file paths.

                SAVE happens when re-saving this project.

                SAVE_AS happens when saving a project in a new location. This happens the first time we save a project.

                UNLOCK happens when the component is about to be changed and it does not have a matching uuid folder in
                the temp area. May happen on project read if the XML specifies to unlock by default.

        Returns:
            (tuple): tuple containing:
                - new_main_file (str): Name of the new main file relative to new_path, or an absolute path if necessary.
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        new_main_file, messages, action_requests = super().save_to_location(new_path, save_type)
        if save_type == 'DUPLICATE':
            action = self._make_duplication_action_request(new_main_file)
            action_requests.append(action)
        return new_main_file, messages, action_requests

    def _make_duplication_action_request(self, new_main_file: str) -> ActionRequest:
        """Create an action request so that we can determine the last component to be duplicated.

        We need this so that we can fix all the paths using the duplication helper file when the last item is
        duplicated. We have to wait until the last item because items can reference other items (e.g. IMS references
        model uuids in its settings.json file). This action request slows things down a bit, but we only do it for the
        sim, and I can't think of any other way to do duplication correctly.

        Returns:
            (ActionRequest): See description.
        """
        return ActionRequest(
            main_file=self.main_file,
            modality='NO_DIALOG',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='_duplicate_sim_action',
            comp_uuid=Path(self.main_file).parent.name,
            parameters={'new_main_file': new_main_file}
        )

    def _duplicate_sim_action(self, query, params):
        """Called via ActionRequest when duplicating a simulation to can get the last component that will be duplicated.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        _duplicate_sim(query, self.main_file, params[0]['new_main_file'])
        return [], []

    def create_event(self, lock_state):
        """This will be called when the component is created from nothing.

        Args:
            lock_state (bool): True if the component is locked for editing. Do not change the files if locked.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        with open(self.main_file, 'w') as _:
            pass

        new_sim_action = ActionRequest(
            main_file=self.main_file,
            modality='modal',
            class_name=self.class_name,
            module_name=self.module_name,
            method_name='open_new_sim_dialog'
        )
        messages = []
        action_requests = [new_sim_action]
        return messages, action_requests

    def open_new_sim_dialog(self, query: Query, params, win_cont: QWidget):
        """Opens the New Simulation dialog and creates the model files indicated.

        Args:
            query: Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.
            win_cont: The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        help_id = gui_util.help_id_from_key('NewSimDialog')
        return new_sim_dialog_runner.run_dialog(self.main_file, query, params, win_cont, model_str='', help_id=help_id)

    @override
    def get_project_explorer_menus(self, main_file_list):
        """This will be called when right-click menus in the project explorer area of XMS are being created.

        Args:
            main_file_list (list of str): A list of the main files of the selected components of this type.

        Returns:
            menu_items (list of xmsapi.dmi.MenuItem): A list of menus and menu items to be shown. Note
            that this list can have objects of type xmsapi.dmi.Menu as well as xmsapi.dmi.MenuItem. "None" may be
            added to the list to indicate a separator.
        """
        if not main_file_list or len(main_file_list) > 1:
            return []  # Multi-select or nothing selected

        menu_list = [None]  # None == spacer
        self._add_tree_menu_command('Open...', 'open_dialog', '', menu_list)
        self._add_tree_menu_command('Export...', 'export_dialog', '', menu_list)
        self._add_tree_menu_command('Check Simulation...', 'check_simulation', '', menu_list)
        self._add_tree_menu_command('Open Containing Folder', '_open_containing_folder', '', menu_list)
        menu_list.append(None)
        return menu_list

    def export_dialog(self, query: Query, params, win_cont: QWidget):
        """Opens a dialog with options on how and where to export the simulation.

        Args:
            query: Object for communicating with GMS
            params: Generic map of parameters. Unused in this case.
            win_cont: The window container.
        """
        messages = []
        actions = []

        dir_ = settings.get_file_browser_directory()
        if not dir_:
            dir_ = QDir.homePath()
        gms_options = MfsimData.read_or_init_gms_options(os.path.dirname(self.main_file))
        dialog = ExportSimulationDialog(parent=win_cont, directory=dir_, options=gms_options)
        if dialog.exec() == QDialog.Accepted:
            folder = os.path.dirname(dialog.selectedFiles()[0])
            dialog.update_options(gms_options)
            io_util.write_gms_options(os.path.dirname(self.main_file), gms_options)
            settings.save_file_browser_directory(folder)
            mf6_exporter.run_export(query, mfsim_nam=self.main_file, export_dir=folder, win_cont=win_cont)

        return messages, actions

    def open_dialog(self, query, params, win_cont):
        """Opens the simulation model dialog.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        messages = []
        actions = []

        uuid = query.parent_item_uuid()
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, uuid)

        # Read the file
        mfsim_dir = os.path.dirname(self.main_file)
        reader = io_factory.reader_from_ftype('MFSIM6')
        data = reader.read(self.main_file, sim_node, query)

        # Do the dialog
        locked = False  # There's currently no way to lock / unlock the sim so I guess it's always unlocked
        help_id = gui_util.help_id_from_key(self.ftype)
        dlg_input = DialogInput(data=data, locked=locked, query=query, help_id=help_id)
        dialog = SimDialog(dlg_input, win_cont)
        dialog.setModal(True)
        if dialog.exec() == QDialog.Accepted and not locked:
            # Write the package
            writer_options = WriterOptions(
                mfsim_dir=mfsim_dir,
                use_open_close=True,
                use_input_dir=False,
                use_output_dir=False,
                dmi_sim_dir=os.path.join(mfsim_dir, '..')
            )
            writer = io_factory.writer_from_ftype('MFSIM6', writer_options)
            QApplication.setOverrideCursor(Qt.WaitCursor)
            writer.write(dialog.dlg_input.data)
            QApplication.restoreOverrideCursor()

        return messages, actions

    def _saved_sim(self, query):
        """Returns the location of the saved simulation, and the simulation tree node.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS

        Returns:
            tuple((str), (TreeNode)): See description.
        """
        sim_uuid = query.parent_item_uuid()
        sim_node = tree_util.find_tree_node_by_uuid(query.project_tree, sim_uuid)
        mfsim_nam = saved_sim(sim_node.name)
        return mfsim_nam, sim_node

    def check_simulation(self, query, params, win_cont):
        """Runs the model checker.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        messages = []
        actions = []

        mfsim_nam, sim_node = self._saved_sim(query)
        if not os.path.isfile(mfsim_nam):
            message_box.message_with_ok(
                parent=win_cont,
                message=f'Could not find the simulation at "{mfsim_nam}".'
                ' The simulation must be saved before running the model checker.',
            )
            return messages, actions

        runner = ModelChecker(
            mfsim_dir=os.path.dirname(mfsim_nam), modflow_exe_string=sim_runner.get_modflow_exe_string(sim_node)
        )
        runner.run_dialog = self.show_check_sim_dialog
        runner.run_with_feedback_dialog(win_cont=win_cont)

        return messages, actions

    def _open_containing_folder(self, query, params, win_cont):
        """Runs the menu command.

        Args:
            query (xmsapi.dmi.Query): Object for communicating with GMS
            params (list[dict]): ActionRequest parameters
            win_cont (QWidget): The window container.

        Returns:
            (tuple): tuple containing:
                - messages (list of tuple of str): List of tuples with the first element of the
                  tuple being the message level (DEBUG, ERROR, WARNING, INFO) and the second element being the message
                  text.
                - action_requests (list of xmsapi.dmi.ActionRequest): List of actions for XMS to perform.
        """
        del params  # Unused parameter
        messages = []
        actions = []

        try:
            mfsim_nam, sim_node = self._saved_sim(query)
            if not os.path.isfile(mfsim_nam):
                message_box.message_with_ok(
                    parent=win_cont,
                    message=f'Could not find the simulation at "{mfsim_nam}".'
                    ' The simulation must be saved first.',
                )
                return messages, actions

            subprocess.run(fr'explorer /select,"{mfsim_nam}"')
        except Exception as ex:
            messages.append(str(ex))

        return messages, actions
