"""GwfGwfData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.exchanges.gwx_gwx_data_base import GwxGwxDataBase
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxCheckbox, CheckboxComboBox


class GwfGwfData(GwxGwxDataBase):
    """Data class to hold the info from a GWT-GWT exchange package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'GWF6-GWF6'

    def dialog_title(self):
        """Returns the title to show in the dialog.

        You should override this method.

        Returns:
            (str): The dialog title.
        """
        return 'Groundwater Flow (GWF) Exchange'

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                # BOUNDNAMES is a little harder to support than in other packages because exchange dialogs use
                # ListBlockTableWidget, not PeriodListWidget. The latter has a change_boundnames() method.
                # I'm sure it's doable, but it will take a little work.
                # Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxComboBox(
                    'CELL_AVERAGING',
                    brief='Horizontal conductance calculation method',
                    items=['HARMONIC', 'LOGARITHMIC', 'AMT-LMK'],
                    value='HARMONIC'
                ),
                CheckboxCheckbox('VARIABLECV', option2='DEWATERED'),
                Checkbox('NEWTON', brief='Activate Newton-Raphson formulation for groundwater flow'),
                Checkbox('XT3D', brief='Use XT3D formulation between cells connected with this exchange'),
                CheckboxButton(
                    'GNC6 FILEIN',
                    brief='Ghost-node correction files',
                    button_text='Files...',
                    button_method='on_btn_gnc6_filein'
                ),
                CheckboxButton(
                    'MVR6 FILEIN',
                    brief='Water mover files',
                    button_text='Files...',
                    button_method='on_btn_mvr6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
            ]
        )
