"""UzfData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
import numpy as np
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data import data_util
from xms.mf6.data.list_package_data import ListPackageData
from xms.mf6.data.options_block import OptionsBlock
from xms.mf6.gui import gui_util, units_util
from xms.mf6.gui.options_defs import Checkbox, CheckboxButton, CheckboxField


class UzfData(ListPackageData):
    """Data class to hold the info from a UZF package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'UZF6'
        self.ntrailwaves = 7
        self.nwavesets = 40
        self.block_with_cellids = 'PACKAGEDATA'
        self.list_blocks['PACKAGEDATA'] = ''

    # @overrides
    def get_column_info(self, block, use_aux=True):
        """Returns column names, types, and defaults.

        The columns depend on the DIS package in use and the AUX variables.
        The package specific and AUX columns are type object because they
        might contain time series strings.

        Args:
            block (str): Name of the list block.
            use_aux (bool): True to include AUXILIARY variables.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        if block.upper() == 'PACKAGEDATA':
            id_columns = data_util.get_id_column_dict(self.grid_info())
            columns = {
                **{
                    'IUZNO': (np.int32, 1),
                },
                **id_columns,
                **{
                    'LANDFLAG': (np.int32, 1),
                    'IVERTCON': (np.int32, 1),
                    'SURFDEP': (np.float64, 0.0),
                    'VKS': (np.float64, 0.0),
                    'THTR': (np.float64, 0.0),
                    'THTS': (np.float64, 0.0),
                    'THTI': (np.float64, 0.0),
                    'EPS': (np.float64, 0.0),
                }
            }
            data_util.add_boundname_columns_to_dict(self.options_block, columns)

        else:  # This would be the stress periods
            columns = {
                'IUZNO': (np.int32, 1),
                'FINF': (object, ''),
                'PET': (object, ''),
                'EXTDP': (object, ''),
                'EXTWC': (object, ''),
                'HA': (object, ''),
                'HROOT': (object, ''),
                'ROOTACT': (object, ''),
            }
            self.add_aux_columns_to_dict(columns, use_aux=use_aux)

        names, types, defaults = gui_util.column_info_tuple_from_dict(columns)
        return names, types, defaults

    def get_column_tool_tips(self, block: str) -> dict[int, str]:
        """Returns a dict with column index and tool tip.

        Args:
            block (str): Name of the block.
        """
        names, _types, _defaults = self.get_column_info(block)
        if block.upper() == 'PACKAGEDATA':
            length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
            k_units = units_util.string_from_units(self, units_util.UNITS_K)
            return {
                names.index('IUZNO'): 'UZF cell number',
                names.index('LANDFLAG'):
                    'Set to one for land surface cells indicating that boundary conditions'
                    ' can be applied and data can be specified in the PERIOD block. A value'
                    ' of 0 specifies a non-land surface cell',
                names.index('IVERTCON'):
                    'Underlying UZF cell that receives water flowing to bottom of cell. If'
                    ' unsaturated zone flow reaches the water table before the cell bottom,'
                    ' then water is added to the GWF cell instead of flowing to the'
                    ' underlying UZF cell. A value of 0 indicates the UZF cell is not'
                    ' connected to an underlying UZF cell.',
                names.index('SURFDEP'): f'Surface depression depth of the UZF cell {length_units}',
                names.index('VKS'): f'Saturated vertical hydraulic conductivity of the UZF cell {k_units}',
                names.index('THTR'): 'Residual (irreducible) water content of the UZF cell',
                names.index('THTS'): 'Saturated water content of the UZF cell',
                names.index('THTI'): 'Initial water content of the UZF cell',
                names.index('EPS'): 'Exponent used in the Brooks-Corey function',
            }
        else:  # stress periods
            infiltration_units = units_util.string_from_units(self, units_util.UNITS_INFILTRATION)
            length_units = units_util.string_from_units(self, units_util.UNITS_LENGTH)
            return {
                names.index('IUZNO'): 'UZF cell number',
                names.index('FINF'): f'Applied infiltration rate of the UZF cell {infiltration_units}',
                names.index('PET'):
                    'Potential evapotranspiration rate of the UZF cell and specified GWF cell'
                    f' {infiltration_units}',
                names.index('EXTDP'): f'Evapotranspiration extinction depth of the UZF cell {length_units}',
                names.index('EXTWC'): 'Evapotranspiration extinction water content of the UZF cell',
                names.index('HA'): 'Air entry potential (head) of the UZF cell',
                names.index('HROOT'): 'Root potential (head) of the UZF cell',
                names.index('ROOTACT'): 'Root activity function of the UZF cell',
            }

    def package_column_info(self, block=''):
        """Returns the column info just for the columns unique to this package.

        You should override this method.

        Returns:
            (tuple): tuple containing:
                - column_names (list): Column names.
                - types (dict of str -> type): Column names -> column types.
                - default (dict of str -> value): Column names -> default values.
        """
        raise NotImplementedError()
        # return [], {}, {}

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Unsaturated Zone Flow (UZF) Package'

    @override
    def get_time_series_columns(self) -> list[int]:
        """Returns a list of the column indices that can contain time series.

        Returns:
            List of indices of columns that can contain time series.
        """
        names, _, _ = self.get_column_info('PERIODS')
        ts_columns = {'FINF', 'PET', 'EXTDP', 'EXTWC', 'HA', 'HROOT', 'ROOTACT'}
        return [i for i, name in enumerate(names) if name in ts_columns]

    def block_with_boundnames(self):
        """Returns the name of the block that can have aux variables.

        Returns:
            (str): The name of the block that can have aux variables.
        """
        return 'PACKAGEDATA'

    # @overrides
    def update_displayed_cell_indices(self) -> None:
        """Updates the cell indices file used to display symbols."""
        self._update_displayed_cell_indices_in_block('PACKAGEDATA')

    def stress_id_columns(self):
        """Returns the column name where the id exists that can be used to help identify this stress across periods.

        Typically is 'CELLIDX' which is added by GMS but is 'RNO' for SFR.

        Returns:
            See description.
        """
        return ['IUZNO']

    def plottable_columns(self):
        """Returns a set of columns (0-based) that can be plotted with the XySeriesEditor.

        Returns:
            See description.
        """
        column_count = len(self.get_column_info('')[0])
        return set(range(1, column_count))  # Start after the IUZNO column

    @override
    def _setup_options(self) -> OptionsBlock:
        """Returns the definition of all the available options.

        Returns:
            See description.
        """
        return OptionsBlock(
            [
                CheckboxButton(
                    'AUXILIARY',
                    button_text='Auxiliary Variables...',
                    check_box_method='on_chk_auxiliary',
                    button_method='on_btn_auxiliary'
                ),
                CheckboxField(
                    'AUXMULTNAME',
                    brief='Auxiliary variable used as multiplier of GWF cell area used by UZF cell',
                    type_='str'
                ),
                Checkbox('BOUNDNAMES', brief='Allow boundary names', check_box_method='on_chk_boundnames'),
                Checkbox('PRINT_INPUT', brief='Print input to listing file'),
                Checkbox('PRINT_FLOWS', brief='Print flows to listing file'),
                Checkbox('SAVE_FLOWS', brief='Save flows to budget file'),
                CheckboxField('WATER_CONTENT FILEOUT', brief='Save water content to a binary file', type_='str'),
                CheckboxField('BUDGET FILEOUT', brief='Save budget to file', type_='str'),
                CheckboxField('BUDGETCSV FILEOUT', brief='Save budget to CSV file', type_='str'),
                CheckboxField('PACKAGE_CONVERGENCE FILEOUT', brief='Save convergence info to file', type_='str'),
                CheckboxButton(
                    'TS6 FILEIN', brief='Time-series files', button_text='Files...', button_method='on_btn_ts6_filein'
                ),
                CheckboxButton(
                    'OBS6 FILEIN',
                    brief='Observation files',
                    button_text='Files...',
                    button_method='on_btn_obs6_filein'
                ),
                Checkbox('MOVER', brief='Can be used with the Water Mover (MVR) Package'),
                Checkbox(
                    'SIMULATE_ET', brief='ET in the unsaturated (UZF) and saturated zones (GWF) will be simulated'
                ),
                Checkbox('LINEAR_GWET', brief='Simulate groundwater ET using the ET formulation of MODFLOW-2005'),
                Checkbox(
                    'SQUARE_GWET', brief='Simulate groundwater ET using a constant rate between TOP and TOP-EXTDP'
                ),
                Checkbox(
                    'UNSAT_ETWC',
                    brief='Simulate ET in the unsaturated zone as a function of the specified PET rate...'
                ),
                Checkbox(
                    'UNSAT_ETAE',
                    brief='Simulate ET in the unsaturated zone using a capillary pressure based formulation'
                ),
            ]
        )
