"""ZoneData class."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules

# 2. Third party modules
from typing_extensions import override

# 3. Aquaveo modules

# 4. Local modules
from xms.mf6.data.array import Array
from xms.mf6.data.array_layer import ArrayLayer
from xms.mf6.data.grid_info import DisEnum, GridInfo
from xms.mf6.data.griddata_base import GriddataBase
from xms.mf6.gui import units_util


class ZoneData(GriddataBase):
    """Data class to hold the info from a zone package file."""
    def __init__(self, **kwargs):
        """Initializes the class.

        Args:
            **kwargs: Arbitrary keyword arguments.

        Keyword Args:
            ftype (str): The file type used in the GWF name file (e.g. 'WEL6')
            mfsim (MfsimData): The simulation.
            model (GwfData or GwtData): The GWF/GWT model. Will be None for TDIS, IMS, Exchanges (things below mfsim)
            grid_info (GridInfo): Information about the grid. Only used when testing individual packages. Otherwise,
             it comes from model and dis
        """
        super().__init__(**kwargs)
        self.ftype = 'ZONE6'
        self.ncells = 0  # We need this when reading the IZONE array
        self.add_block('GRIDDATA', ['IZONE'])
        self.can_import = False
        self.can_export = False

    def init_izone(self, grid_info: GridInfo) -> None:
        """Initializes griddata['IZONE'].

        Args:
            grid_info (GridInfo): Number of rows, cols etc.
        """
        if grid_info.dis_enum in [DisEnum.DIS, DisEnum.DISV]:
            array = Array(array_name='IZONE', storage='CONSTANT', layered=True)
            for _layer in range(grid_info.nlay):
                array_layer = ArrayLayer(numeric_type='int', constant=1, factor=1, name='IZONE')
                array.layers.append(array_layer)
        else:  # grid_info.dis_enum == DisEnum.DISU:
            array = Array(array_name='IZONE', storage='CONSTANT', layered=False)
            array_layer = ArrayLayer(numeric_type='int', constant=1, factor=1, name='IZONE')
            array.layers.append(array_layer)
        self.block('GRIDDATA').add_array(array)

    def get_units(self, array_name: str) -> str:
        """Returns the units string for the array.

        Args:
            array_name (str): The name of a array.

        Returns:
            (str): The units string like 'L' or 'L^3/T'.
        """
        match array_name:
            case 'IZONE':
                units_spec = units_util.UNITS_UNITLESS
            case _:
                units_spec = ''  # This is an error
        return units_spec

    def is_int_array(self, array_name):
        """Returns True if the array is integers.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'IZONE'

    @override
    def is_required_array(self, array_name: str) -> bool:
        """Returns True if the array is required.

        Args:
            array_name (str): The name of a array.

        Returns:
            (bool): True or False
        """
        return array_name.upper() == 'IZONE'

    def dialog_title(self):
        """Returns the title to show in the dialog.

        Returns:
            (str): The dialog title.
        """
        return 'Zone File'

    @override
    def array_size_and_layers(self, array_name: str, layered: bool) -> tuple[int, int, tuple]:
        """Returns True if the array can be LAYERED.

        Args:
            array_name (str): The name of a array.
            layered (bool): True if layered keyword was read.

        Returns:
            (tuple): tuple containing:
                - nvalues (int): Number of values to read.
                - layers_to_read (int): Number of layers to read.
        """
        grid_info = self.grid_info()
        if grid_info is not None and self.ncells == grid_info.cell_count():
            nvalues, shape = ArrayLayer.number_of_values_and_shape(layered, grid_info)
            layers_to_read = grid_info.nlay if layered else 1
            return nvalues, layers_to_read, shape
        else:
            nvalues = self.ncells
            shape = (self.ncells, 1)
            layers_to_read = self.grid_info().nlay if layered else 1
            return nvalues, layers_to_read, shape

    @override
    def get_tool_tip(self, tab_text: str) -> str:
        """Returns the tool tip that goes with the tab with the tab_name.

        Args:
            tab_text: Text of the tab

        Returns:
            (str): The tool tip.
        """
        txt = ''
        if tab_text == 'IZONE':
            txt = (
                'Integer array containing the zone number for each cell in the grid. IZONE values can be positive or'
                ' negative integers. An IZONE value of zero indicates cells for which no budget is desired or'
                ' calculated. The LAYERED option can only be used if a binary grid file is provided, and the model is'
                ' layered. For GWF Models, the DIS or DISV Package must be used in order for the model to be layered.'
            )
        return txt
