"""Time utility functions."""

__copyright__ = "(C) Copyright Aquaveo 2025"
__license__ = "All rights reserved"

# 1. Standard Python modules
from datetime import datetime, timedelta
from typing import Optional

# 2. Third party modules
from dateutil import parser
from PySide2.QtCore import (QDateTime, Qt)

# 3. Aquaveo modules
from xms.datasets.dataset_reader import DatasetReader

# 4. Local modules
from xms.mf6.gui.units_util import MF6_TIME_UNITS
from xms.mf6.misc import log_util

# Constants
seconds_per_day = 86400
seconds_per_hour = 3600
seconds_per_minute = 60
minutes_per_day = 1440
hours_per_day = 24
days_per_year = 365.2425
seconds_per_year = days_per_year * seconds_per_day


def q_date_time_from_arbitrary_string(time_string):
    """Parses a time string into a QDateTime the best way we know how.

    Args:
        time_string (str): The time string.

    Returns:
        (QDateTime): The date/time.
    """
    date_time = datetime_from_arbitrary_string(time_string)
    q_date_time = QDateTime.fromString(date_time.isoformat(), Qt.ISODate) if date_time else None
    return q_date_time


def q_date_time_from_iso_string(time_string):
    """Parses a time string into a QDateTime the best way we know how.

    Args:
        time_string (str): The time string.

    Returns:
        (QDateTime): The date/time.
    """
    date_time = datetime_from_iso_string(time_string)
    q_date_time = QDateTime.fromString(date_time.isoformat(), Qt.ISODate) if date_time else None
    return q_date_time


def datetime_from_iso_string(time_string: str) -> Optional[datetime]:
    """Parses a ISO-8601 date/time string into a Python datetime object.

    See https://www.w3.org/TR/NOTE-datetime and https://dateutil.readthedocs.io/en/stable/parser.html
    If non-iso date/time strings are needing parsing, use datetime.strftime() and specify the format.

    Args:
        time_string (str): The time string.

    Returns:
        (datetime): The date/time.
    """
    if not time_string:
        return None
    try:
        return parser.isoparse(time_string)
    except Exception as error:
        logger = log_util.get_logger()
        logger.error(str(error))
        return None


def datetime_from_arbitrary_string(time_string: str) -> Optional[datetime]:
    """Tries to parse a date/time string with unknown format into a Python datetime object.

    See https://dateutil.readthedocs.io/en/stable/parser.html

    Args:
        time_string (str): The time string.

    Returns:
        (datetime): The date/time.
    """
    if not time_string:
        return None
    try:
        return parser.parse(time_string)
    except Exception as error:
        logger = log_util.get_logger()
        logger.error(str(error))
        return None


def time_duration(start_date_time, end_date_time, time_units) -> float:
    """Returns the difference between start_date_time and end_date_time in the specified time units.

    Args:
        start_date_time (datetime):
        end_date_time (datetime):
        time_units (str): Time units: 'SECONDS', 'MINUTES', 'HOURS', 'DAYS', 'YEARS'

    Returns:
        (float): See description.
    """
    if time_units not in MF6_TIME_UNITS or time_units == 'UNKNOWN':
        raise RuntimeError('Invalid time units in data_util.time_duration')

    # Create a python timedelta object by subtracting the dates
    delta = end_date_time - start_date_time

    # Get the days and seconds from the timedelta (which are the only thing it stores)
    days = delta.days
    seconds = delta.seconds

    float_value = -999.0
    if time_units == 'SECONDS':
        float_value = (days * seconds_per_day) + seconds
    elif time_units == 'MINUTES':
        float_value = (days * minutes_per_day) + (seconds / seconds_per_minute)
    elif time_units == 'HOURS':
        float_value = (days * hours_per_day) + (seconds / seconds_per_hour)
    elif time_units == 'DAYS':
        float_value = days + (seconds / seconds_per_day)
    elif time_units == 'YEARS':
        float_value = (days / days_per_year) + (seconds / seconds_per_year)
    return float_value


def compute_end_date_py(date_time: datetime, duration: float, units: str) -> Optional[datetime]:  # pragma no cover
    """Given a starting datetime, a duration, and units, returns the end datetime.

    Args:
        date_time (datetime): Starting date/time.
        duration (float):
        units (str): 'YEARS', 'DAYS', 'HOURS', 'MINUTES', or 'SECONDS'

    Returns:
        (datetime): See description.
    """
    date = None
    if units == 'YEARS':
        # Years are problematic due to leap years
        date = date_time.replace(year=date_time.year + int(duration))
    elif units == 'DAYS':
        date = date_time + timedelta(days=duration)
    elif units == 'HOURS':
        date = date_time + timedelta(hours=duration)
    elif units == 'MINUTES':
        date = date_time + timedelta(minutes=duration)
    elif units == 'SECONDS':
        date = date_time + timedelta(seconds=duration)
    return date


def halfway_datetime(start_date: datetime, end_date: datetime) -> datetime:
    """Calculates the datetime halfway between two datetimes.

    Args:
        start_date: The starting datetime object.
        end_date: The ending datetime object.

    Returns:
        A datetime object representing the midpoint between the two input dates.
    """
    time_difference = end_date - start_date
    half_difference = time_difference / 2
    middle_date = start_date + half_difference
    return middle_date


def dataset_times(dataset: DatasetReader) -> list[float] | list[datetime]:
    """Return a list of the dataset times.

    Args:
        dataset: The dataset.

    Returns:
        See description.
    """
    times = []
    if dataset.ref_time is not None:
        for i in range(len(dataset.times)):
            times.append(dataset.ref_time + dataset.timestep_offset(i))
    else:
        times = dataset.times[()].tolist()
    return times
